/*
 * Copyright (c) 1997, 2018 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v. 2.0, which is available at
 * http://www.eclipse.org/legal/epl-2.0.
 *
 * This Source Code may also be made available under the following Secondary
 * Licenses when the conditions for such availability set forth in the
 * Eclipse Public License v. 2.0 are satisfied: GNU General Public License,
 * version 2 with the GNU Classpath Exception, which is available at
 * https://www.gnu.org/software/classpath/license.html.
 *
 * SPDX-License-Identifier: EPL-2.0 OR GPL-2.0 WITH Classpath-exception-2.0
 */

package com.sun.xml.rpc.processor.generator.nodes;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;

/**
 * @author  Qingqing Ouyang
 * @version 1.1.5
 */
public abstract class JaxRpcMappingNode extends java.lang.Object {

    private static final String QNAME_SEPARATOR = ":";
    private static String DEBUG =
        System.getProperty("com.sun.xml.rpc.j2ee.debug");

    /**
     *  <p>
     * @return the Document for the given node
     * </p>
     */
    static protected Document getOwnerDocument(Node node) {

        if (node instanceof Document) {
            return (Document) node;
        }
        return node.getOwnerDocument();
    }

    /**
     * <p>
     * Append a new element child to the current node 
     * </p>
     * @param parent is the parent node for the new child element
     * @param elementName is new element tag name
     * @return the newly created child node
     */
    public static Element appendChild(Node parent, String elementName) {
        Element child =
            getOwnerDocument(parent).createElementNS(
                JaxRpcMappingTagNames.J2EE_NAMESPACE,
                elementName);
        parent.appendChild(child);
        return child;
    }

    /**
     * <p>
     * Append a new text child
     * </p>
     * @param parent for the new child element
     * @param elementName is the new element tag name
     * @param text the text for the new element
     * @return the newly create child node
     */
    public static Node appendTextChild(
        Node parent,
        String elementName,
        String text) {

        if (text == null || text.length() == 0)
            return null;

        Node child = appendChild(parent, elementName);
        child.appendChild(getOwnerDocument(child).createTextNode(text));
        return child;
    }

    /**
     * <p>
     * Append a new text child
     * </p>
     * @param parent for the new child element
     * @param elementName is the new element tag name
     * @param value the int value for the new element
     * @return the newly create child node
     */
    public static Node appendTextChild(
        Node parent,
        String elementName,
        int value) {
        return appendTextChild(parent, elementName, String.valueOf(value));
    }

    /**
     * <p>
     * Append a new text child even if text is empty
     * </p>
     * @param parent for the new child element
     * @param elementName is the new element tag name
     * @param text the text for the new element
     * @return the newly create child node
     */
    public static Node forceAppendTextChild(
        Node parent,
        String elementName,
        String text) {

        Node child = appendChild(parent, elementName);
        if (text != null && text.length() != 0) {
            child.appendChild(getOwnerDocument(child).createTextNode(text));
        }
        return child;
    }

    /**
     * <p>
     * Append a new attribute to an element
     * </p>
     * @param parent for the new child element
     * @param elementName is the new element tag name
     * @param text the text for the new element
     */
    public static void setAttribute(
        Element parent,
        String elementName,
        String text) {

        if (text == null || text.length() == 0)
            return;
        parent.setAttributeNS(
            JaxRpcMappingTagNames.J2EE_NAMESPACE,
            elementName,
            text);
    }

    /**
     * Set a namespace attribute on an element.
     * @param element on which to set attribute
     * @param prefix raw prefix (without "xmlns:")
     * @param namespaceURI namespace URI to which prefix is mapped.
     */
    public static void setAttributeNS(
        Element element,
        String prefix,
        String namespaceURI) {

        String nsPrefix =
            prefix.equals("") ? "xmlns" : "xmlns" + QNAME_SEPARATOR + prefix;

        element.setAttributeNS(
            "http://www.w3.org/2000/xmlns/",
            nsPrefix,
            namespaceURI);
    }

    //XXX FIXME.  Logging in jaxrpc?
    protected void debug(String className, String msg) {
        if (DEBUG != null) {
            System.out.println("[" + className + "] --> " + msg);
        }
    }
}
