/*
 * Copyright (c) 1997, 2020 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package com.sun.xml.ws.rx.rm.runtime.sequence;

import com.sun.xml.ws.rx.rm.runtime.RmRuntimeVersion;
import com.sun.xml.ws.rx.rm.faults.AbstractSoapFaultException;
import com.sun.xml.ws.rx.rm.localization.LocalizationMessages;
import com.sun.xml.ws.rx.rm.runtime.RuntimeContext;
import java.util.List;
import javax.xml.namespace.QName;
import jakarta.xml.soap.Detail;

/**
 * An example of when this fault is generated is when a message is Received by 
 * the RM Source containing a SequenceAcknowledgement covering messages that have
 * not been sent.
 *
 * Properties:
 * [Code] Sender
 * [Subcode] wsrm:InvalidAcknowledgement
 * [Reason] The SequenceAcknowledgement violates the cumulative Acknowledgement invariant.
 * [Detail] <wsrm:SequenceAcknowledgement ...> ... </wsrm:SequenceAcknowledgement>
 *
 * Generated by: RM Source.
 * Condition : In response to a SequenceAcknowledgement that violate the invariants
 * stated in 2.3 or any of the requirements in 3.9 about valid combinations of AckRange,
 * Nack and None in a single SequenceAcknowledgement element or with respect to already
 * Received such elements.
 * Action Upon Generation : Unspecified.
 * Action Upon Receipt : Unspecified.
 *
 *
 * @author Marek Potociar (marek.potociar at sun.com)
 */
public final class InvalidAcknowledgementException extends AbstractSoapFaultException {
    private static final long serialVersionUID = 647447570493203088L;
    //
    private final List<Sequence.AckRange> ackedRanges;
    
    public InvalidAcknowledgementException(String sequenceId, long messageIdentifier, List<Sequence.AckRange> ackedRanges) {
        super(LocalizationMessages.WSRM_1125_ILLEGAL_MESSAGE_ID(sequenceId, messageIdentifier),
                "The SequenceAcknowledgement violates the cumulative Acknowledgement invariant.",
                false);

        this.ackedRanges = ackedRanges;
    }

    @Override
    public Code getCode() {
        return Code.Sender;
    }

    @Override
    public QName getSubcode(RmRuntimeVersion rv) {
        return rv.protocolVersion.invalidAcknowledgementFaultCode;
    }

    @Override
    public Detail getDetail(RuntimeContext rc) {
//        return new DetailBuilder(rc).addSequenceAcknowledgement(ackedRanges).build();
        return null;
    }


}
