/*
 * Copyright (c) 1997, 2021 Oracle and/or its affiliates. All rights reserved.
 *
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Distribution License v. 1.0, which is available at
 * http://www.eclipse.org/org/documents/edl-v10.php.
 *
 * SPDX-License-Identifier: BSD-3-Clause
 */

package com.sun.xml.ws.rx.rm.runtime.sequence;

import com.sun.xml.ws.rx.rm.runtime.RmRuntimeVersion;
import com.sun.xml.ws.rx.rm.faults.AbstractSoapFaultException;
import com.sun.xml.ws.rx.rm.localization.LocalizationMessages;
import com.sun.xml.ws.rx.rm.runtime.RuntimeContext;
import javax.xml.namespace.QName;
import javax.xml.soap.Detail;

/**
 * Inicates that the sequence with given sequence identifier is not available.
 * 
 * This exceptions is used under the following conditions:
 *  <ul>
 *      <li>sequence with such {@code sequenceId} is not registered with a given sequence manager</li>
 *  </ul>
 *
 * SOAP Fault data as per WS-RM spec:
 *
 * Properties:
 * [Code] Sender
 * [Subcode] wsrm:UnknownSequence
 * [Reason] The value of wsrm:Identifier is not a known Sequence identifier.
 * [Detail] {@code <wsrm:Identifier ...> xs:anyURI </wsrm:Identifier>}
 *
 * Generated by: RM Source or RM Destination.
 * Condition : In response to a message containing an unknown or terminated Sequence identifier.
 * Action Upon Generation : None.
 * Action Upon Receipt : MUST terminate the Sequence if not otherwise terminated.
 * 
 * @author Marek Potociar (marek.potociar at sun.com)
 */
public final class UnknownSequenceException extends AbstractSoapFaultException {
    private static final long serialVersionUID = 8837409835889666590L;
    //
    private final String sequenceId;
    
    /**
     * Constructs an instance of <code>NoSuchSequenceException</code> for the sequence with {@code sequenceId} identifier.
     * @param sequenceId the identifier of the unknown sequence.
     */
    public UnknownSequenceException(String sequenceId) {
        super(
                LocalizationMessages.WSRM_1124_NO_SUCH_SEQUENCE_ID_REGISTERED(sequenceId),
                "The value of wsrm:Identifier is not a known Sequence identifier.",
                true);

        this.sequenceId = sequenceId;
    }

    /**
     * Returns the identifier of the unknown sequence
     * @return the unknown sequence identifier
     */
    public String getSequenceId() {
        return sequenceId;
    }        
    @Override
    public Code getCode() {
        return Code.Sender;
    }

    @Override
    public QName getSubcode(RmRuntimeVersion rv) {
        return rv.protocolVersion.unknownSequenceFaultCode;
    }

    @Override
    public Detail getDetail(RuntimeContext rc) {
        return new DetailBuilder(rc).addSequenceIdentifier(sequenceId).build();
    }
}
