/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 * 
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 * 
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common Development
 * and Distribution License("CDDL") (collectively, the "License").  You
 * may not use this file except in compliance with the License. You can obtain
 * a copy of the License at https://glassfish.dev.java.net/public/CDDL+GPL.html
 * or glassfish/bootstrap/legal/LICENSE.txt.  See the License for the specific
 * language governing permissions and limitations under the License.
 * 
 * When distributing the software, include this License Header Notice in each
 * file and include the License file at glassfish/bootstrap/legal/LICENSE.txt.
 * Sun designates this particular file as subject to the "Classpath" exception
 * as provided by Sun in the GPL Version 2 section of the License file that
 * accompanied this code.  If applicable, add the following below the License
 * Header, with the fields enclosed by brackets [] replaced by your own
 * identifying information: "Portions Copyrighted [year]
 * [name of copyright owner]"
 * 
 * Contributor(s):
 * 
 * If you wish your version of this file to be governed by only the CDDL or
 * only the GPL Version 2, indicate your decision by adding "[Contributor]
 * elects to include this software in this distribution under the [CDDL or GPL
 * Version 2] license."  If you don't indicate a single choice of license, a
 * recipient has the option to distribute your version of this file under
 * either the CDDL, the GPL Version 2 or to extend the choice of license to
 * its licensees as provided above.  However, if you add GPL Version 2 code
 * and therefore, elected the GPL Version 2 license, then the option applies
 * only if the new code is made subject to such option by the copyright
 * holder.
 */
package org.glassfish.enterprise.iiop.util;

import com.sun.corba.ee.spi.orbutil.ORBConstants;
import com.sun.corba.ee.spi.orbutil.threadpool.*;
import com.sun.logging.LogDomains;
import org.glassfish.enterprise.iiop.impl.IIOPUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.logging.Level;
import java.util.logging.Logger;

public class S1ASThreadPoolManager implements ThreadPoolManager {

    static Logger _logger = LogDomains.getLogger(S1ASThreadPoolManager.class,
            LogDomains.UTIL_LOGGER);

    private static final int DEFAULT_NUMBER_OF_QUEUES = 0;
    private static final int DEFAULT_MIN_THREAD_COUNT = 10;
    private static final int DEFAULT_MAX_THREAD_COUNT = 200;

    private static HashMap idToIndexTable = new HashMap();
    private static HashMap indexToIdTable = new HashMap();
    private static ArrayList threadpoolList = new ArrayList();
    private static String defaultID;

    private static ThreadPoolManager s1asThreadPoolMgr = new S1ASThreadPoolManager();

    public static ThreadPoolManager getThreadPoolManager() {
        return s1asThreadPoolMgr;
    }


    S1ASThreadPoolManager() {

        IIOPUtils iiopUtils = IIOPUtils.getInstance();
        try {
            com.sun.enterprise.config.serverbeans.ThreadPool[] allThreadPools = iiopUtils.getAllThreadPools();
            for (int i = 0; i < allThreadPools.length; i++) {
                createThreadPools(allThreadPools[i], i);
            }
            defaultID = (String) indexToIdTable.get(new Integer(0));
        } catch (NullPointerException npe) {
            _logger.log(Level.FINE, "Server Context is NULL. Ignoring and proceeding.");
        }


    }


    private void createThreadPools(com.sun.enterprise.config.serverbeans.ThreadPool
            threadpoolBean, int index) {
        IIOPUtils iiopUtils = IIOPUtils.getInstance();
        String threadpoolId = null;
        String minThreadsValue, maxThreadsValue, timeoutValue, numberOfQueuesValue;
        int minThreads = DEFAULT_MIN_THREAD_COUNT;
        int maxThreads = DEFAULT_MAX_THREAD_COUNT;
        int idleTimeoutInSeconds = ORBConstants.DEFAULT_INACTIVITY_TIMEOUT;
        int numberOfQueues = DEFAULT_NUMBER_OF_QUEUES;

        try {
            threadpoolId = threadpoolBean.getThreadPoolId();
        } catch (NullPointerException npe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "ThreadPoolBean may be null ", npe);
            }
        }
        try {
            minThreadsValue = threadpoolBean.getMinThreadPoolSize();
            minThreads = Integer.parseInt(minThreadsValue);
        } catch (NullPointerException npe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "ThreadPoolBean may be null ", npe);
                _logger.log(Level.WARNING,
                        "Using default value for steady-threadpool-size = " + minThreads);
            }
        } catch (NumberFormatException nfe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "enterprise_util.excep_orbmgr_numfmt", nfe);
                _logger.log(Level.WARNING,
                        "Using default value for min-threadpool-size = " + minThreads);
            }
        }
        try {
            maxThreadsValue = threadpoolBean.getMaxThreadPoolSize();
            maxThreads = Integer.parseInt(maxThreadsValue);
        } catch (NullPointerException npe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "ThreadPoolBean may be null ", npe);
                _logger.log(Level.WARNING,
                        "Using default value for max-threadpool-size = " + maxThreads);
            }
        } catch (NumberFormatException nfe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "enterprise_util.excep_orbmgr_numfmt", nfe);
                _logger.log(Level.WARNING,
                        "Using default value for max-threadpool-size = " + maxThreads);
            }
        }
        try {
            timeoutValue = threadpoolBean.getIdleThreadTimeoutInSeconds();
            idleTimeoutInSeconds = Integer.parseInt(timeoutValue);
        } catch (NullPointerException npe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "ThreadPoolBean may be null ", npe);
                _logger.log(Level.WARNING,
                        "Using default value for idle-thread-timeout-in-seconds = " +
                                idleTimeoutInSeconds);
            }
        } catch (NumberFormatException nfe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "enterprise_util.excep_orbmgr_numfmt", nfe);
                _logger.log(Level.WARNING,
                        "Using default value for idle-thread-timeout-in-seconds = " +
                                idleTimeoutInSeconds);
            }
        }

        // Currently this value is not used but when multi-queue threadpools are
        // implemented this could be used to decide which one to instantiate and
        // number of queues in the multi-queue threadpool
        try {
            numberOfQueuesValue = threadpoolBean.getNumWorkQueues();
            numberOfQueues = Integer.parseInt(numberOfQueuesValue);
        } catch (NullPointerException npe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "ThreadPoolBean may be null ", npe);
                _logger.log(Level.WARNING,
                        "Using default value for num-work-queues = " +
                                numberOfQueues);
            }
        } catch (NumberFormatException nfe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "enterprise_util.excep_orbmgr_numfmt", nfe);
                _logger.log(Level.WARNING,
                        "Using default value for num-work-queues = " +
                                numberOfQueues);
            }
        }

        // Mutiplied the idleTimeoutInSeconds by 1000 to convert to milliseconds
        ThreadPoolFactory threadPoolFactory = new ThreadPoolFactory();
        ThreadPool threadpool =
                threadPoolFactory.create(minThreads, maxThreads,
                        idleTimeoutInSeconds * 1000, threadpoolId,
                        iiopUtils.getCommonClassLoader());

        // Add the threadpool instance to the threadpoolList
        threadpoolList.add(threadpool);

        // Associate the threadpoolId to the index passed
        idToIndexTable.put(threadpoolId, new Integer(index));

        // Associate the threadpoolId to the index passed
        indexToIdTable.put(new Integer(index), threadpoolId);

    }

    /**
     * This method will return an instance of the threadpool given a threadpoolId,
     * that can be used by any component in the app. server.
     *
     * @throws NoSuchThreadPoolException thrown when invalid threadpoolId is passed
     *                                   as a parameter
     */
    public ThreadPool
    getThreadPool(String id)
            throws NoSuchThreadPoolException {

        Integer i = (Integer) idToIndexTable.get(id);
        if (i == null) {
            throw new NoSuchThreadPoolException();
        }
        try {
            ThreadPool threadpool =
                    (ThreadPool)
                            threadpoolList.get(i.intValue());
            return threadpool;
        } catch (IndexOutOfBoundsException iobe) {
            throw new NoSuchThreadPoolException();
        }
    }

    /**
     * This method will return an instance of the threadpool given a numeric threadpoolId.
     * This method will be used by the ORB to support the functionality of
     * dedicated threadpool for EJB beans
     *
     * @throws NoSuchThreadPoolException thrown when invalidnumericIdForThreadpool is passed
     *                                   as a parameter
     */
    public ThreadPool getThreadPool(int numericIdForThreadpool)
            throws NoSuchThreadPoolException {

        try {
            ThreadPool threadpool =
                    (ThreadPool)
                            threadpoolList.get(numericIdForThreadpool);
            return threadpool;
        } catch (IndexOutOfBoundsException iobe) {
            throw new NoSuchThreadPoolException();
        }
    }

    /**
     * This method is used to return the numeric id of the threadpool, given a String
     * threadpoolId. This is used by the POA interceptors to add the numeric threadpool
     * Id, as a tagged component in the IOR. This is used to provide the functionality of
     * dedicated threadpool for EJB beans
     */
    public int getThreadPoolNumericId(String id) {
        Integer i = (Integer) idToIndexTable.get(id);
        return ((i == null) ? 0 : i.intValue());
    }

    /**
     * Return a String Id for a numericId of a threadpool managed by the threadpool
     * manager
     */
    public String getThreadPoolStringId(int numericIdForThreadpool) {
        String id = (String) indexToIdTable.get(new Integer(numericIdForThreadpool));
        return ((id == null) ? defaultID : id);
    }

    /**
     * Returns the first instance of ThreadPool in the ThreadPoolManager
     */
    public ThreadPool
    getDefaultThreadPool() {
        try {
            return getThreadPool(0);
        } catch (NoSuchThreadPoolException nstpe) {
            if (_logger.isLoggable(Level.WARNING)) {
                _logger.log(Level.WARNING, "No default ThreadPool defined ", nstpe);
            }
        }
        return null;
    }

    /**
     * Return an instance of ThreadPoolChooser based on the componentId that was
     * passed as argument
     */
    public ThreadPoolChooser getThreadPoolChooser(String componentId) {
        //FIXME: This method is not used, but should be fixed once
        //ORB's nio select starts working and we start using ThreadPoolChooser
        //This will be mostly used by the ORB
        return null;
    }

    /**
     * Return an instance of ThreadPoolChooser based on the componentIndex that was
     * passed as argument. This is added for improved performance so that the caller
     * does not have to pay the cost of computing hashcode for the componentId
     */
    public ThreadPoolChooser getThreadPoolChooser(int componentIndex) {
        //FIXME: This method is not used, but should be fixed once
        //ORB's nio select starts working and we start using ThreadPoolChooser
        //This will be mostly used by the ORB
        return null;
    }

    /**
     * Sets a ThreadPoolChooser for a particular componentId in the ThreadPoolManager. This
     * would enable any component to add a ThreadPoolChooser for their specific use
     */
    public void setThreadPoolChooser(String componentId, ThreadPoolChooser aThreadPoolChooser) {
        //FIXME: This method is not used, but should be fixed once
        //ORB's nio select starts working and we start using ThreadPoolChooser
        //This will be mostly used by the ORB
    }

    /**
     * Gets the numeric index associated with the componentId specified for a
     * ThreadPoolChooser. This method would help the component call the more
     * efficient implementation i.e. getThreadPoolChooser(int componentIndex)
     */
    public int getThreadPoolChooserNumericId(String componentId) {
        //FIXME: This method is not used, but should be fixed once
        //ORB's nio select starts working and we start using ThreadPoolChooser
        //This will be mostly used by the ORB
        return 0;
    }

    public void close() {
        //TODO
    }
} 


