// $ANTLR 2.7.6 (2005-12-22): "Semantic.g" -> "Semantic.java"$

    package com.sun.jdo.spi.persistence.support.ejb.ejbqlc;
    
    import java.util.ResourceBundle;
    import java.lang.reflect.Method;
    import com.sun.jdo.spi.persistence.utility.I18NHelper;
    import com.sun.jdo.spi.persistence.support.ejb.ejbc.MethodHelper;

import antlr.TreeParser;
import antlr.Token;
import antlr.collections.AST;
import antlr.RecognitionException;
import antlr.ANTLRException;
import antlr.NoViableAltException;
import antlr.MismatchedTokenException;
import antlr.SemanticException;
import antlr.collections.impl.BitSet;
import antlr.ASTPair;
import antlr.collections.impl.ASTArray;


/**
 * This class defines the semantic analysis of the EJBQL compiler.
 * Input of this pass is the AST as produced by the parser,
 * that consists of EJBQLAST nodes.
 * The result is a typed EJBQLAST tree.
 *
 * @author  Michael Bouschen
 * @author  Shing Wai Chan
 */
public class Semantic extends antlr.TreeParser       implements SemanticTokenTypes
 {

    /** Name of the property to disable order by validation. */
    public static final String DISABLE_ORDERBY_VALIDATION_PROPERTY =
        "com.sun.jdo.spi.persistence.support.ejb.ejbqlc.DISABLE_ORDERBY_VALIDATION"; // NOI18N

    /**
     * Property to disable order by validation. 
     * Note, the default is false, meaning the compiler checks that select 
     * clause and orderby clause are compatible.
     */
    private static final boolean DISABLE_ORDERBY_VALIDATION = 
        Boolean.getBoolean(DISABLE_ORDERBY_VALIDATION_PROPERTY);

    /** Symbol table handling names of variables and parameters. */
    protected SymbolTable symtab;

    /** Type info access helper. */
    protected TypeSupport typeSupport;
    
    /** Parameter info helper. */
    protected ParameterSupport paramSupport;
    
    /** The Method instance of the finder/selector method. */
    protected Method method;

    /** result-type-mapping element from the DD. */
    protected int resultTypeMapping;

    /** Flag indicating finder or selector. */
    protected boolean finderNotSelector;

    /** Flag indicating have aggregate function or not. */
    protected boolean isAggregate = false;

    /** The ejb-name. */
    protected String ejbName;

    /** I18N support. */
    protected final static ResourceBundle msgs = I18NHelper.loadBundle(
        Semantic.class);
    
    /**
     * Initializes the semantic analysis.
     * @param typeSupport type info access helper.
     * @param paramSupport parameter info helper.
     * @param method method instance of the finder/selector method.
     * @param resultTypeMapping result-type-mapping element from the DD
     * @param finderNotSelector <code>true</code> for finder; 
     * <code>false</code> for selector
     * @param ejbName the ejb name of the finder/selector method.
     */
    public void init(TypeSupport typeSupport, ParameterSupport paramSupport,
                     Method method, int resultTypeMapping,  
                     boolean finderNotSelector, String ejbName)
    {
        this.symtab = new SymbolTable();
        this.typeSupport = typeSupport;
        this.paramSupport = paramSupport;
        this.method = method;
        this.resultTypeMapping = resultTypeMapping;
        this.finderNotSelector = finderNotSelector;
        this.ejbName = ejbName;
    }

    /** */
    public void reportError(RecognitionException ex) {
        ErrorMsg.fatal(I18NHelper.getMessage(msgs, "ERR_SemanticError"), ex); //NOI18N
    }

    /** */
    public void reportError(String s) {
        ErrorMsg.fatal(I18NHelper.getMessage(msgs, "ERR_SemanticError") + s); //NOI18N
    }
    
    //========= Internal helper methods ==========

    /**
     * Checks the return type and the type of the select clause expression 
     * of a finder method.
     * <p>
     * The return type of a finder must be one of the following: 
     * <ul>
     * <li>java.util.Collection (multi-object finder)
     * <li>java.util.Enumeration (EJB 1.1 multi-object finder)
     * <li>the entity bean's remote interface (single-object finder)
     * <li>the entity bean's local interface (single-object finder)
     * </ul>
     * The type of the select clause expression of a finder must be 
     * the entity bean's local or remote interface.
     * @param returnType the return type of the finder/selector method object
     * @param selectClauseTypeInfo the type info of the select clause 
     * expression. 
     */
    private void checkFinderReturnType(
        Class returnType, Object selectClauseTypeInfo)
    {
        String selectClauseTypeName = typeSupport.getTypeName(selectClauseTypeInfo);
        Object returnTypeInfo = typeSupport.getTypeInfo(returnType);
        // The return type of a finder must be Collection or Enumeration or
        // the entity bean's remote or local interface 
        if ((returnType != java.util.Collection.class) &&
            (returnType != java.util.Enumeration.class) &&
            (!typeSupport.isRemoteInterfaceOfEjb(returnTypeInfo, ejbName)) &&
            (!typeSupport.isLocalInterfaceOfEjb(returnTypeInfo, ejbName))) {
            ErrorMsg.error(I18NHelper.getMessage(msgs, 
                "EXC_InvalidFinderReturnType", returnType.getName())); //NOI18N
                    
        }
        
        // The type of the select clause expression must be the ejb name 
        // of this bean.
        if (!selectClauseTypeName.equals(this.ejbName)) {
            ErrorMsg.error(I18NHelper.getMessage(msgs, 
                "EXC_InvalidFinderSelectClauseType", selectClauseTypeName)); //NOI18N
        }
    }

    /**
     * Implements type compatibility for selector. The method returns
     * <code>true</code> if returnTypeInfo is compatible with 
     * selectClauseTypeInfo.
     */
    private boolean isCompatibleSelectorSelectorReturnType(
            Object returnTypeInfo, Object selectClauseTypeInfo)
    {
        if (isAggregate) {
            return getCommonOperandType(selectClauseTypeInfo, returnTypeInfo) != TypeSupport.errorType;   
        } else {
            return typeSupport.isCompatibleWith(selectClauseTypeInfo, returnTypeInfo);
        }
    }
    

    /**
     * Checks the return type and the type of the select clause expression 
     * of a selector method.
     * <p>
     * The return type of a selector must be one of the following: 
     * <ul>
     * <li>java.util.Collection (multi-object selector)
     * <li>java.util.Set (multi-object selector)
     * <li>assignable from the type of the select clause expression 
     * (single-object selector)
     * </ul>
     * @param returnType the return type of the finder/selector method object
     * @param selectClauseTypeInfo the type info of the select clause 
     * expression. 
     */
    private void checkSelectorReturnType(
        Class returnType, Object selectClauseTypeInfo)
    {
        String selectClauseTypeName = typeSupport.getTypeName(selectClauseTypeInfo);
        Object returnTypeInfo = typeSupport.getTypeInfo(returnType);
        // The return type of a selector must be Collection or Set or 
        // assingable from the type of the select clause expression
        if ((returnType != java.util.Collection.class) &&
            (returnType != java.util.Set.class) &&
            !isCompatibleSelectorSelectorReturnType(returnTypeInfo,
                selectClauseTypeInfo)) {
            ErrorMsg.error(I18NHelper.getMessage(msgs,
                "EXC_InvalidSelectorReturnType", //NOI18N
                typeSupport.getTypeName(returnTypeInfo), selectClauseTypeName)); 
        }
    }

    /**
     * Checks the result-type-mapping element setting in the case of a finder 
     * method. Finder must not specify result-type-mapping.
     */
    private void checkFinderResultTypeMapping()
    {
        if (resultTypeMapping != MethodHelper.NO_RETURN) {
            ErrorMsg.error(I18NHelper.getMessage(msgs, 
                "EXC_InvalidResultTypeMappingForFinder")); //NOI18N
        }
    }

    /**
     * Checks the setting of the result-type-mapping element for a 
     * selector. Only selectors returning a entity object may 
     * specify this.
     * <p>
     * The method checks the following error cases:
     * <ul>
     * <li>result-type-mapping is specified as Remote, 
     * but bean does not have remote interface
     * <li>result-type-mapping is specified as Local, 
     * but bean does not have local interface
     * <li>single-object selector returns remote interface,
     * but result-type-mapping is not specified as Remote
     * <li>single-object selector returns local interface,
     * but result-type-mapping is specified as Remote
     * <li>result-type-mapping is specified for a selector returning 
     * non-entity objects.
     * </ul>
     * @param returnType the return type of the finder/selector method object
     * @param selectClauseTypeInfo the type info of the select clause.
     */
    private void checkSelectorResultTypeMapping(
        Class returnType, Object selectClauseTypeInfo)
    {
        Object returnTypeInfo = typeSupport.getTypeInfo(returnType);

        // case: multi-object selector returning entity objects
        if (typeSupport.isCollectionType(returnTypeInfo) && 
            typeSupport.isEjbName(selectClauseTypeInfo)) {
            if (resultTypeMapping == MethodHelper.REMOTE_RETURN) {
                // result-type-mapping is Remote => 
                // bean must have remote interface
                if (!typeSupport.hasRemoteInterface(selectClauseTypeInfo)) {
                    ErrorMsg.error(I18NHelper.getMessage(msgs, 
                        "EXC_InvalidRemoteResultTypeMappingForMultiSelector", //NOI18N
                        selectClauseTypeInfo)); 
                }
            }
            else {
                // result-type-mapping is Local or not specified => 
                // bean must have local interface
                if (!typeSupport.hasLocalInterface(selectClauseTypeInfo)) {
                    ErrorMsg.error(I18NHelper.getMessage(msgs,
                        "EXC_InvalidLocalResultTypeMappingForMultiSelector", //NOI18N
                        selectClauseTypeInfo)); 
                }
            }
        }
        // case: single-object selector returning remote interface
        else if (typeSupport.isRemoteInterface(returnTypeInfo)) {
            // result-type-mapping must be Remote
            if (resultTypeMapping != MethodHelper.REMOTE_RETURN) {
                ErrorMsg.error(I18NHelper.getMessage(msgs,
                    "EXC_InvalidLocalResultTypeMappingForSingleSelector")); //NOI18N     
            }
        }
        // case: single-object selector returning local interface
        else if (typeSupport.isLocalInterface(returnTypeInfo)) {
            // result-type-mapping must be Local or not specified
            if (resultTypeMapping == MethodHelper.REMOTE_RETURN) {
                ErrorMsg.error(I18NHelper.getMessage(msgs,
                    "EXC_InvalidRemoteResultTypeMappingForSingleSelector")); //NOI18N     
            }
        }
        // cases: single-object and multi-object selector 
        // returning non-enity object(s)
        else if (resultTypeMapping != MethodHelper.NO_RETURN) {
            // result-type-mapping must not be specified
            ErrorMsg.error(I18NHelper.getMessage(msgs, 
                "EXC_InvalidResultTypeMappingForSelector", //NOI18N
                selectClauseTypeInfo)); 
        }
    }

    /**
     * Checks that select clause and orderby clause are compatible.
     * <p>
     * The method checks the following error cases:
     * <ul>
     * <li>if the select clause is an identification variable or
     * a single valued cmr path expression, then the orderby item
     * must be a cmp field of the entity bean abstract schema
     * type value returned by the SELECT clause
     * <li>if the select clause is a cmp field, then
     * orderby item must be empty or the same cmp field.
     * </ul>
     * @param select the select clause of the query
     * @param orderby the orderby clause of the query
     */
    private void checkSelectOrderbyClause(EJBQLAST select, EJBQLAST orderby)
    {
        // nothing to check if no orderby clause or 
        // if orderby validation is disabled
        if ((orderby == null) || DISABLE_ORDERBY_VALIDATION) {
            return;
        }

        AST selectReturnAST = select.getFirstChild();
        // skip DISTINCT node, so selectReturnAST should be one of the following:
        //     Object(x), cmr-field, cmp-field
        // it is illegal to be an aggregate function node
        if (selectReturnAST.getType() == DISTINCT) {
            selectReturnAST = selectReturnAST.getNextSibling();
        }

        if (selectReturnAST.getType() == CMP_FIELD_ACCESS) {
            StringBuffer buf = new StringBuffer();
            genPathExpression(selectReturnAST, buf);
            String selectReturnPathExpr = buf.toString();
            for (AST sibling = orderby.getFirstChild();
                    sibling != null;
                    sibling = sibling.getNextSibling().getNextSibling()) {

                // share buf
                buf.setLength(0);
                genPathExpression(sibling, buf);
                String siblingPathExpr = buf.toString();
                if (!selectReturnPathExpr.equals(siblingPathExpr)) {
                    ErrorMsg.error(I18NHelper.getMessage(msgs, 
                    "EXC_InvalidOrderbyItemForCMPSelect", //NOI18N
                    siblingPathExpr)); 
                }
            }
        } else {
            AST abstractSchemaAST = null;
            if (selectReturnAST.getType() == SINGLE_CMR_FIELD_ACCESS) {
                abstractSchemaAST = selectReturnAST;
            } else if (selectReturnAST.getType() == OBJECT) {
                abstractSchemaAST = selectReturnAST.getFirstChild();
            } else { // it must be an aggregate function node
                ErrorMsg.error(I18NHelper.getMessage(msgs,
                "EXC_InvalidAggregateOrderby" //NOI18N
                ));
            }

            StringBuffer buf = new StringBuffer();
            genPathExpression(abstractSchemaAST, buf);
            String abstractSchemaExpr = buf.toString();
            for (AST sibling = orderby.getFirstChild();
                    sibling != null;
                    sibling = sibling.getNextSibling().getNextSibling()) {

                // share  buf
                buf.setLength(0);
                genPathExpression(sibling.getFirstChild(), buf);
                String siblingRootExpr = buf.toString();
                if (!abstractSchemaExpr.equals(siblingRootExpr)) {
                    buf.setLength(0);
                    genPathExpression(sibling, buf);
                    ErrorMsg.error(I18NHelper.getMessage(msgs, 
                    "EXC_InvalidOrderbyItem", //NOI18N
                    buf.toString())); 
                }
            }
        } 
    }

    /**
     * Form a string representation of a dot expression and append to given
     * StringBuffer.
     * @param ast the AST node representing the root the of the expression
     * @param buf the StringBuffer that will have result of path expression
     * append
     */
    //SW: We can write this method without recursion. Michael suggests to use
    //recursion for readability.
    private void genPathExpression(AST ast, StringBuffer buf) {
        if (ast == null) {
            return;
        }
        switch (ast.getType()) {
            case CMP_FIELD_ACCESS:
            case COLLECTION_CMR_FIELD_ACCESS:
            case SINGLE_CMR_FIELD_ACCESS:
                AST left = ast.getFirstChild();
                AST right = left.getNextSibling();
                genPathExpression(left, buf);
                buf.append('.');
                genPathExpression(right, buf);
                break;
            default:
                buf.append(ast.getText());
                break;
        }
    }

    /**
     * Analyses a logical operation AND, OR
     * @param op the logical operator
     * @param leftAST left operand 
     * @param rightAST right operand
     * @return the type info of the operator 
     */
    private Object analyseConditionalExpr(EJBQLAST op, EJBQLAST leftAST, EJBQLAST rightAST)
    {
        Object left = leftAST.getTypeInfo();
        Object right = rightAST.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(left) || typeSupport.isErrorType(right))
            return typeSupport.errorType;
        
        if (typeSupport.isBooleanType(left) && typeSupport.isBooleanType(right)) {
            Object common = typeSupport.booleanType;
            return common;
        }

        // if this code is reached a bitwise operator was used with invalid arguments
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_InvalidArguments",  op.getText())); //NOI18N
        return typeSupport.errorType;
    }
    
    /** 
     * Analyses a equality operation (==, <>)
     * @param op the relational operator
     * @param leftAST left operand 
     * @param rightAST right operand
     * @return the type info of the operator
     */
    private Object analyseEqualityExpr(EJBQLAST op, EJBQLAST leftAST, EJBQLAST rightAST)
    {
        Object left = leftAST.getTypeInfo();
        Object right = rightAST.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(left) || typeSupport.isErrorType(right)) {
            return typeSupport.errorType;
        }

        // check left hand side for literals and input params 
        if (isLiteral(leftAST)) {
            ErrorMsg.error(leftAST.getLine(), leftAST.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_InvalidLHSLiteral", //NOI18N 
                    leftAST.getText(), op.getText()));
            return typeSupport.errorType;
        }
        else if (isInputParameter(leftAST)) {
            ErrorMsg.error(leftAST.getLine(), leftAST.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_InvalidLHSParameter", //NOI18N 
                    leftAST.getText(), op.getText()));
            return typeSupport.errorType;
        }
        
        // check operand types 
        if (typeSupport.isNumberType(left) && typeSupport.isNumberType(right)) {
            return typeSupport.booleanType;
        }
        else if (typeSupport.isStringType(left) && typeSupport.isStringType(right)) {
            return typeSupport.booleanType;
        }
        else if (typeSupport.isDateTimeType(left) && typeSupport.isDateTimeType(right)) {
            return typeSupport.booleanType;
        }
        else if (isEntityBeanValue(leftAST) && isEntityBeanValue(rightAST) && 
                 (typeSupport.isCompatibleWith(left, right) ||
                  typeSupport.isCompatibleWith(right, left))) {
            String leftEjbName = (String)leftAST.getTypeInfo();
            // the input parameter must be on right hand side of an equality
            // expression ('?1' = e.department is not supported)
            return analyseParameterEjbName(rightAST, leftEjbName);
        }
        else if (typeSupport.isBooleanType(left) && typeSupport.isBooleanType(right)) {
            return typeSupport.booleanType;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_InvalidArguments",  op.getText())); //NOI18N 
        return typeSupport.errorType;
    }
    
    /**
     * Analyses a relational operation (<, <=, >, >=)
     * @param op the relational operator
     * @param leftAST left operand 
     * @param rightAST right operand
     * @return the type info of the operator
     */
    private Object analyseRelationalExpr(EJBQLAST op, EJBQLAST leftAST, EJBQLAST rightAST)
    {
        Object left = leftAST.getTypeInfo();
        Object right = rightAST.getTypeInfo();

        // handle error type
        if (typeSupport.isErrorType(left) || typeSupport.isErrorType(right)) {
            return typeSupport.errorType;
        }

        // check left hand side for literals and input params 
        if (isLiteral(leftAST)) {
            ErrorMsg.error(leftAST.getLine(), leftAST.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_InvalidLHSLiteral", //NOI18N 
                    leftAST.getText(), op.getText()));
            return typeSupport.errorType;
        }
        else if (isInputParameter(leftAST)) {
            ErrorMsg.error(leftAST.getLine(), leftAST.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_InvalidLHSParameter", //NOI18N 
                    leftAST.getText(), op.getText()));
            return typeSupport.errorType;
        }
        
        // check operand types
        if ((typeSupport.isNumberType(left) && typeSupport.isNumberType(right)) ||
            (typeSupport.isDateTimeType(left) && typeSupport.isDateTimeType(right)) ||
            (typeSupport.isStringType(left) && typeSupport.isStringType(right))) {
            return typeSupport.booleanType;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_InvalidArguments",  op.getText())); //NOI18N 
        return typeSupport.errorType;
    }
    
    /**
     * Analyses a binary arithmetic expression +, -, *, /.
     * @param op the  operator
     * @param leftAST left operand 
     * @param rightAST right operand
     * @return the type info of the operator
     */
    private Object analyseBinaryArithmeticExpr(EJBQLAST op, EJBQLAST leftAST, EJBQLAST rightAST)
    {
        Object left = leftAST.getTypeInfo();
        Object right = rightAST.getTypeInfo();

        // handle error type
        if (typeSupport.isErrorType(left) || typeSupport.isErrorType(right)) {
            return typeSupport.errorType;
        }

        if (typeSupport.isNumberType(left) && typeSupport.isNumberType(right)) {
            Object common = getCommonOperandType(left, right);
            if (!typeSupport.isErrorType(common))
                return common;
        }

        // if this code is reached a conditional operator was used with invalid arguments
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_InvalidArguments",  op.getText())); //NOI18N
        return typeSupport.errorType;
    }

    /**
     * Returns the common type info for the specified operand types. 
     * This includes binary numeric promotion as specified in Java.
     * @param left type info of left operand 
     * @param right type info of right operand
     * @return the type info of the operator
     */
    private Object getCommonOperandType(Object left, Object right)
    {
        if (typeSupport.isNumberType(left) && typeSupport.isNumberType(right)) {
            boolean wrapper = false;

            // handle java.math.BigDecimal:
            if (typeSupport.bigDecimalType.equals(left)) {
                return left;
            }
            if (typeSupport.bigDecimalType.equals(right)) {
                return right;
            }
            
            // handle java.math.BigInteger
            if (typeSupport.bigIntegerType.equals(left)) {
                // if right is floating point return BigDecimal, 
                // otherwise return BigInteger
                return typeSupport.isFloatingPointType(right) ? 
                       typeSupport.bigDecimalType : left;
            }
            if (typeSupport.bigIntegerType.equals(right)) {
                // if left is floating point return BigDecimal, 
                // otherwise return BigInteger
                return typeSupport.isFloatingPointType(left) ? 
                       typeSupport.bigDecimalType : right;
            }       

            if (typeSupport.isNumericWrapperType(left)) {
                left = typeSupport.getPrimitiveType(left);
                wrapper = true;
            }
            if (typeSupport.isNumericWrapperType(right)) {
                right = typeSupport.getPrimitiveType(right);
                wrapper = true;
            }
            
            // handle numeric types with arbitrary arithmetic operator
            if (typeSupport.isNumericType(left) && typeSupport.isNumericType(right)) {
                Object promotedType = typeSupport.binaryNumericPromotion(left, right);
                if (wrapper) 
                    promotedType = typeSupport.getWrapperType(promotedType);
                return promotedType;
            }
        }
        else if (typeSupport.isBooleanType(left) && typeSupport.isBooleanType(right)) {
            // check for boolean wrapper class: if one of the operands has the 
            // type Boolean return Boolean, otherwise return boolean.
            if (left.equals(typeSupport.booleanClassType) || 
                right.equals(typeSupport.booleanClassType))
                return typeSupport.booleanClassType;
            else
                return typeSupport.booleanType;
        }
        else if (typeSupport.isCompatibleWith(left, right)) {
            return right;
        }
        else if (typeSupport.isCompatibleWith(right, left)) {
            return left;
        }

        // not compatible types => return errorType
        return typeSupport.errorType;
    }

    /**
     * Analyses a unary expression (+ and -).
     * @param op the operator
     * @param argASTleftAST left operand 
     * @param rightAST right operand
     * @return the type info of the operator 
     */
    private Object analyseUnaryArithmeticExpr(EJBQLAST op, EJBQLAST argAST)
    {
        Object arg = argAST.getTypeInfo();

        // handle error type
        if (typeSupport.isErrorType(arg))
            return arg;
        
        if (typeSupport.isNumberType(arg)) {
            boolean wrapper = false;
            if (typeSupport.isNumericWrapperType(arg)) {
                arg = typeSupport.getPrimitiveType(arg);
                wrapper = true;
            }

            Object promotedType = typeSupport.unaryNumericPromotion(arg);
            if (wrapper)
                promotedType = typeSupport.getWrapperType(promotedType);
            return promotedType;
        }
        
        // if this code is reached a conditional operator was used with invalid arguments
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_InvalidArguments",  op.getText())); //NOI18N
        return typeSupport.errorType;
    }
    
    /** 
     * Analyses a expression node that is expected to access a collection 
     * valued CMR field. It returns the element type of the collection valued 
     * CMR field. 
     * @param fieldAccess the field access node
     * @return the type info of the operator 
     */
    private Object analyseCollectionValuedCMRField(EJBQLAST fieldAccess)
    {
        if (fieldAccess.getType() != COLLECTION_CMR_FIELD_ACCESS) {
            ErrorMsg.fatal(I18NHelper.getMessage(msgs, "ERR_InvalidPathExpr")); //NOI18N
            return typeSupport.errorType;
        }

        EJBQLAST classExpr = (EJBQLAST)fieldAccess.getFirstChild();
        EJBQLAST field = (EJBQLAST)classExpr.getNextSibling();
        Object fieldInfo = 
            typeSupport.getFieldInfo(classExpr.getTypeInfo(), field.getText());
        return typeSupport.getElementType(fieldInfo);
    }

    /**
     * Analyses a MEMBER OF operation. 
     * @param op the MEMBER OF operator
     * @param value node representing the value to be tested
     * @param col the collection
     * @return the type info of the operator
     */
    private Object analyseMemberExpr(EJBQLAST op, EJBQLAST value, EJBQLAST col)
    {
        Object valueTypeInfo = value.getTypeInfo();
        Object elementTypeInfo = analyseCollectionValuedCMRField(col);

        // handle error type
        if (typeSupport.isErrorType(valueTypeInfo) || 
            typeSupport.isErrorType(elementTypeInfo)) {
            return typeSupport.errorType;
        }

        // check compatibility
        if (typeSupport.isCompatibleWith(valueTypeInfo, elementTypeInfo) ||
            typeSupport.isCompatibleWith(elementTypeInfo, valueTypeInfo)) {

            return analyseParameterEjbName(value, (String)elementTypeInfo);
        }
        
        // if this code is reached there is a compatibility problem 
        // with the value and the collection expr
        ErrorMsg.error(op.getLine(), op.getColumn(), 
            I18NHelper.getMessage(msgs, "EXC_CollectionElementTypeMismatch", //NOI18N
                typeSupport.getTypeName(elementTypeInfo), 
                typeSupport.getTypeName(valueTypeInfo)));
        return typeSupport.errorType;
    }

    /**
     * Analyses the type of the element to be compatible with the type of the
     * value expression in the sense that element type can be cast into value
     * type without losing precision.
     * For instance, element type can be a double and value type can be an
     * integer.
     * @param elementAST given element
     * @param valueTypeInfo the type to be check for compatibility
     * @return the type info of the elementAST or typeSupport.errorType
     */
    private Object analyseInCollectionElement(EJBQLAST elementAST,
            Object valueTypeInfo)
    {
        Object elementTypeInfo = elementAST.getTypeInfo();

        // handle error type
        if (typeSupport.isErrorType(valueTypeInfo) || 
            typeSupport.isErrorType(elementTypeInfo)) {
            return typeSupport.errorType;
        }

        Object common = getCommonOperandType(elementTypeInfo, valueTypeInfo);
        if (!typeSupport.isErrorType(common) &&
                elementTypeInfo.equals(common)) {
            return common;
        }

        // if this code is reached there is a compatibility problem
        // with the value and the collection expr
        ErrorMsg.error(elementAST.getLine(), elementAST.getColumn(),
            I18NHelper.getMessage(msgs, "EXC_CollectionElementTypeMismatch", //NOI18N
            typeSupport.getTypeName(valueTypeInfo),
            typeSupport.getTypeName(elementTypeInfo)));
        return typeSupport.errorType;
    }

    /**
     * Analyses whether paramAST can be associated to a ejbName.
     * @param paramAST AST node corresponds to a PARAMETER
     * @param ejbName name to be check with paramAST
     * @return the type info of typeSupport.booleanType or typeSupport.errorType
     */
    private Object analyseParameterEjbName(EJBQLAST paramAST, String ejbName)
    {
        if (isInputParameter(paramAST)) {
            String paramName = paramAST.getText();
            String paramEjbName = paramSupport.getParameterEjbName(paramName);
            if (paramEjbName != null && !paramEjbName.equals(ejbName)) {
                ErrorMsg.error(paramAST.getLine(), paramAST.getColumn(),
                    I18NHelper.getMessage(msgs,
                    "EXC_MultipleEJBNameParameter", // NOI18N
                    paramName, ejbName, paramEjbName));
                return typeSupport.errorType;
            } else {
                paramSupport.setParameterEjbName(paramName, ejbName);
            }
        }
        return typeSupport.booleanType;
    }
    
    /** 
     * Returns <code>true</code> if ast denotes a entity bena value.
     */
    private boolean isEntityBeanValue(EJBQLAST ast)
    {
        switch(ast.getType()) {
        case SINGLE_CMR_FIELD_ACCESS:
        case IDENTIFICATION_VAR:
            return true;
        case INPUT_PARAMETER:
            Object typeInfo = ast.getTypeInfo();
            return typeSupport.isEjbOrInterfaceName(typeInfo);
        }
        return false;
    }

    /** 
     * Returns <code>true</code> if ast denotes a literal.
     */
    private boolean isLiteral(EJBQLAST ast)
    {
        int tokenType = ast.getType();
        return ((tokenType == INT_LITERAL) || 
                (tokenType == LONG_LITERAL) ||
                (tokenType == STRING_LITERAL) || 
                (tokenType == FLOAT_LITERAL) || 
                (tokenType == DOUBLE_LITERAL) ||
                (tokenType == TRUE) || 
                (tokenType == FALSE));
    }

    /** 
     * Returns <code>true</code> if ast denotes a input parameter access.
     */
    private boolean isInputParameter(EJBQLAST ast)
    {
        return ast.getType() == INPUT_PARAMETER;
    }
    
    /**
     * The method checks the specified node being an expression of type String. 
     * @param expr the expression to be checked
     * @return <code>true</code> if the specified expression has the type String.
     */
    private boolean isStringExpr(EJBQLAST expr)
    {
        Object exprType = expr.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(exprType))
            return true;
        
        // expr must have the type String
        if (!typeSupport.isStringType(exprType)) {
            ErrorMsg.error(expr.getLine(), expr.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_StringExprExpected", //NOI18N
                    typeSupport.getTypeName(exprType)));
            return false;
        }
        
        // everything is ok => return true;
        return true;
    }

    /**
     * The method checks the specified node being an expression of 
     * type int or java.lang.Integer.
     * @param expr the expression to be checked
     * @return <code>true</code> if the specified expression has the type 
     * int or java.lang.Integer.
     */
    private boolean isIntExpr(EJBQLAST expr)
    {
        Object exprType = expr.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(exprType))
            return true;
        
        // expr must have the type int or Integer
        if (!typeSupport.isIntType(exprType)) {
            ErrorMsg.error(expr.getLine(), expr.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_IntExprExpected", //NOI18N
                    typeSupport.getTypeName(exprType)));
            return false;
        }
        
        // everything is ok => return true;
        return true;
    }

    /**
     * The method checks the specified node being an expression of 
     * type double or java.lang.Double.
     * @param expr the expression to be checked
     * @return <code>true</code> if the specified expression has the type 
     * double or java.lang.Double.
     */
    private boolean isDoubleExpr(EJBQLAST expr)
    {
        Object exprType = expr.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(exprType))
            return true;
        
        // expr must have the type double or Double
        if (!typeSupport.isDoubleType(exprType)) {
            ErrorMsg.error(expr.getLine(), expr.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_DoubleExprExpected", //NOI18N
                    typeSupport.getTypeName(exprType)));
            return false;
        }
        
        // everything is ok => return true;
        return true;
    }

    /**
     * The method checks the specified node being an expression of a number type
     * (a numeric type or a number wrapper class).
     * @param expr the expression to be checked
     * @return <code>true</code> if the specified expression has a number type.
     */
    private boolean isNumberExpr(EJBQLAST expr)
    {
        Object exprType = expr.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(exprType))
            return true;
        
        // expr must have a number type
        if (!typeSupport.isNumberType(exprType)) {
            ErrorMsg.error(expr.getLine(), expr.getColumn(), 
                I18NHelper.getMessage(msgs, "EXC_NumberExprExpected", //NOI18N
                    typeSupport.getTypeName(exprType)));
            return false;
        }
        
        // everything is ok => return true;
        return true;
    }

    /**
     * The method checks the specified node being an expression of a number type
     * (a numeric type or a number wrapper class).
     * @param expr the expression to be checked
     * @return <code>true</code> if the specified expression has a number or
     * String type 
     */
    private boolean isNumberOrStringExpr(EJBQLAST expr)
    {
        Object exprType = expr.getTypeInfo();
        
        // handle error type
        if (typeSupport.isErrorType(exprType))
            return true;
        
        // expr must have a number type
        if (!typeSupport.isNumberType(exprType) &&
                !typeSupport.isStringType(exprType)) {
            ErrorMsg.error(expr.getLine(), expr.getColumn(), 
                I18NHelper.getMessage(msgs,
                    "EXC_NumberOrStringExprExpected", //NOI18N
                    typeSupport.getTypeName(exprType)));
            return false;
        }
        
        // everything is ok => return true;
        return true;
    }

    /** 
     * The method checks whether the specified node denotes a valid abstract 
     * schema type.
     * @param ident the node to be checked
     * @return the type info for the abstract bean class of the specified 
     * abstract schema type.
     */
    private Object checkAbstractSchemaType(EJBQLAST ident)
    {
        String name = ident.getText();
        Object typeInfo = 
            typeSupport.getTypeInfoForAbstractSchema(name);
        if (typeInfo == null) {
            ErrorMsg.error(ident.getLine(), ident.getColumn(), 
                I18NHelper.getMessage(msgs, 
                    "EXC_AbstractSchemNameExpected", name)); //NOI18N
            typeInfo = typeSupport.errorType;
        }
        return typeInfo;
    }

    /**
     * Returns true if the specified text is a string literal consisting of a
     * single char. Escaped chars are counted as a single char such as \ uxxxx.
     */
    private boolean isSingleCharacterStringLiteral(String text)
    {
        int i = 0;
        int length = text.length();
        if (length == 0) {
            // empty string
            return false;
        }
        if (text.charAt(i) == '\\')
        {
            i++;
            if (i == length) {
                // string literal was '\'
                return true;
            }
            // escaped char => check the next char
            if (text.charAt(i) == 'u') {
                // unicode
                i +=5;
            }
            else if (('0' <= text.charAt(i)) && (text.charAt(i) <= '3')) {
                i++;
                if ((i < length) && isOctalDigit(text.charAt(i))) {
                    i++;
                    if ((i < length) && isOctalDigit(text.charAt(i))) {
                        i++;
                    }
                }
            }
            else if (isOctalDigit(text.charAt(i))) {
                i++;
                if ((i < length) && isOctalDigit(text.charAt(i))) {
                    i++;
                }
            }
            else {
                i++;
            }
        }
        else if (text.charAt(i) == '\''){
            // check special EJBQL single quote char
            i++;
            if ((i < length) && (text.charAt(i) == '\'')) {
                i++;
            }
        }
        else {
            i++;
        }
        // reached end of text?
        return (i == length);
    }

    /** Returns true if the specified char is an octal digit */
    private boolean isOctalDigit(char c)
    {
        return ('0' <= c && c <= '7');
    }

public Semantic() {
	tokenNames = _tokenNames;
}

	public final void query(AST _t) throws RecognitionException {
		
		EJBQLAST query_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST query_AST = null;
		EJBQLAST s_AST = null;
		EJBQLAST s = null;
		EJBQLAST o_AST = null;
		EJBQLAST o = null;
		
		AST __t2 = _t;
		EJBQLAST tmp1_AST = null;
		EJBQLAST tmp1_AST_in = null;
		tmp1_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp1_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp1_AST);
		ASTPair __currentAST2 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,QUERY);
		_t = _t.getFirstChild();
		fromClause(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		s = _t==ASTNULL ? null : (EJBQLAST)_t;
		selectClause(_t);
		_t = _retTree;
		s_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		whereClause(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		o = _t==ASTNULL ? null : (EJBQLAST)_t;
		orderbyClause(_t);
		_t = _retTree;
		o_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST2;
		_t = __t2;
		_t = _t.getNextSibling();
		
		checkSelectOrderbyClause(s_AST, o_AST);
		
		query_AST = (EJBQLAST)currentAST.root;
		returnAST = query_AST;
		_retTree = _t;
	}
	
	public final void fromClause(AST _t) throws RecognitionException {
		
		EJBQLAST fromClause_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST fromClause_AST = null;
		
		AST __t4 = _t;
		EJBQLAST tmp2_AST = null;
		EJBQLAST tmp2_AST_in = null;
		tmp2_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp2_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp2_AST);
		ASTPair __currentAST4 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,FROM);
		_t = _t.getFirstChild();
		{
		int _cnt6=0;
		_loop6:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_t.getType()==IN||_t.getType()==RANGE)) {
				identificationVarDecl(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				if ( _cnt6>=1 ) { break _loop6; } else {throw new NoViableAltException(_t);}
			}
			
			_cnt6++;
		} while (true);
		}
		currentAST = __currentAST4;
		_t = __t4;
		_t = _t.getNextSibling();
		fromClause_AST = (EJBQLAST)currentAST.root;
		returnAST = fromClause_AST;
		_retTree = _t;
	}
	
	public final void selectClause(AST _t) throws RecognitionException {
		
		EJBQLAST selectClause_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST selectClause_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		AST __t13 = _t;
		EJBQLAST tmp3_AST = null;
		EJBQLAST tmp3_AST_in = null;
		tmp3_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp3_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp3_AST);
		ASTPair __currentAST13 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,SELECT);
		_t = _t.getFirstChild();
		distinct(_t);
		_t = _retTree;
		astFactory.addASTChild(currentAST, returnAST);
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		projection(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST13;
		_t = __t13;
		_t = _t.getNextSibling();
		
		Object selectClauseTypeInfo = p_AST.getTypeInfo();
		Class returnType = method.getReturnType();
		if (finderNotSelector) {
		checkFinderReturnType(returnType, selectClauseTypeInfo);
		checkFinderResultTypeMapping();
		}
		else {
		checkSelectorReturnType(returnType, selectClauseTypeInfo);
		checkSelectorResultTypeMapping(returnType, 
		selectClauseTypeInfo);
		}
		
		selectClause_AST = (EJBQLAST)currentAST.root;
		returnAST = selectClause_AST;
		_retTree = _t;
	}
	
	public final void whereClause(AST _t) throws RecognitionException {
		
		EJBQLAST whereClause_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST whereClause_AST = null;
		EJBQLAST e_AST = null;
		EJBQLAST e = null;
		
		AST __t29 = _t;
		EJBQLAST tmp4_AST = null;
		EJBQLAST tmp4_AST_in = null;
		tmp4_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp4_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp4_AST);
		ASTPair __currentAST29 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,WHERE);
		_t = _t.getFirstChild();
		e = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		e_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST29;
		_t = __t29;
		_t = _t.getNextSibling();
		
		Object typeInfo = e_AST.getTypeInfo();
		if (!typeSupport.isBooleanType(typeInfo)) {
		ErrorMsg.error(e_AST.getLine(), e_AST.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_BooleanWhereClauseExpected",  //NOI18N
		typeSupport.getTypeName(typeInfo)));
		}
		
		whereClause_AST = (EJBQLAST)currentAST.root;
		returnAST = whereClause_AST;
		_retTree = _t;
	}
	
	public final void orderbyClause(AST _t) throws RecognitionException {
		
		EJBQLAST orderbyClause_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST orderbyClause_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ORDER:
		{
			AST __t31 = _t;
			EJBQLAST tmp5_AST = null;
			EJBQLAST tmp5_AST_in = null;
			tmp5_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp5_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp5_AST);
			ASTPair __currentAST31 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,ORDER);
			_t = _t.getFirstChild();
			{
			int _cnt33=0;
			_loop33:
			do {
				if (_t==null) _t=ASTNULL;
				if ((_t.getType()==DOT)) {
					orderbyItem(_t);
					_t = _retTree;
					astFactory.addASTChild(currentAST, returnAST);
				}
				else {
					if ( _cnt33>=1 ) { break _loop33; } else {throw new NoViableAltException(_t);}
				}
				
				_cnt33++;
			} while (true);
			}
			currentAST = __currentAST31;
			_t = __t31;
			_t = _t.getNextSibling();
			orderbyClause_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case 3:
		{
			orderbyClause_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = orderbyClause_AST;
		_retTree = _t;
	}
	
	public final void identificationVarDecl(AST _t) throws RecognitionException {
		
		EJBQLAST identificationVarDecl_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST identificationVarDecl_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IN:
		{
			collectionMemberDecl(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			identificationVarDecl_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case RANGE:
		{
			rangeVarDecl(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			identificationVarDecl_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = identificationVarDecl_AST;
		_retTree = _t;
	}
	
	public final void collectionMemberDecl(AST _t) throws RecognitionException {
		
		EJBQLAST collectionMemberDecl_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST collectionMemberDecl_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		EJBQLAST var = null;
		EJBQLAST var_AST = null;
		
		AST __t9 = _t;
		EJBQLAST tmp6_AST = null;
		EJBQLAST tmp6_AST_in = null;
		tmp6_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp6_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp6_AST);
		ASTPair __currentAST9 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,IN);
		_t = _t.getFirstChild();
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		collectionValuedPathExpression(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		var = (EJBQLAST)_t;
		EJBQLAST var_AST_in = null;
		var_AST = (EJBQLAST)astFactory.create(var);
		astFactory.addASTChild(currentAST, var_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST9;
		_t = __t9;
		_t = _t.getNextSibling();
		
		Object typeInfo = analyseCollectionValuedCMRField(p_AST);
		String name = var_AST.getText();
		Object identVar = new IdentificationVariable(name, typeInfo);
		if (symtab.declare(name, identVar) != null) {
		ErrorMsg.error(var_AST.getLine(), var_AST.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_MultipleDeclaration", name)); //NOI18N
		}
		var_AST.setType(IDENTIFICATION_VAR_DECL);
		var_AST.setTypeInfo(typeInfo);
		
		collectionMemberDecl_AST = (EJBQLAST)currentAST.root;
		returnAST = collectionMemberDecl_AST;
		_retTree = _t;
	}
	
	public final void rangeVarDecl(AST _t) throws RecognitionException {
		
		EJBQLAST rangeVarDecl_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST rangeVarDecl_AST = null;
		EJBQLAST abstractSchemaName = null;
		EJBQLAST abstractSchemaName_AST = null;
		EJBQLAST var = null;
		EJBQLAST var_AST = null;
		
		AST __t11 = _t;
		EJBQLAST tmp7_AST = null;
		EJBQLAST tmp7_AST_in = null;
		tmp7_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
		tmp7_AST_in = (EJBQLAST)_t;
		astFactory.addASTChild(currentAST, tmp7_AST);
		ASTPair __currentAST11 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,RANGE);
		_t = _t.getFirstChild();
		abstractSchemaName = (EJBQLAST)_t;
		EJBQLAST abstractSchemaName_AST_in = null;
		abstractSchemaName_AST = (EJBQLAST)astFactory.create(abstractSchemaName);
		astFactory.addASTChild(currentAST, abstractSchemaName_AST);
		match(_t,ABSTRACT_SCHEMA_NAME);
		_t = _t.getNextSibling();
		var = (EJBQLAST)_t;
		EJBQLAST var_AST_in = null;
		var_AST = (EJBQLAST)astFactory.create(var);
		astFactory.addASTChild(currentAST, var_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST11;
		_t = __t11;
		_t = _t.getNextSibling();
		
		// check abstract schema name
		Object typeInfo = 
		checkAbstractSchemaType(abstractSchemaName_AST);
		abstractSchemaName_AST.setTypeInfo(typeInfo);
		
		// check identification variable
		String name = var_AST.getText();
		Object identVar = new IdentificationVariable(name, typeInfo);
		if (symtab.declare(name, identVar) != null) {
		ErrorMsg.error(var_AST.getLine(), var_AST.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_MultipleDeclaration", name)); //NOI18N
		}
		var_AST.setType(IDENTIFICATION_VAR_DECL);
		var_AST.setTypeInfo(typeInfo);
		
		rangeVarDecl_AST = (EJBQLAST)currentAST.root;
		returnAST = rangeVarDecl_AST;
		_retTree = _t;
	}
	
	public final void collectionValuedPathExpression(AST _t) throws RecognitionException {
		
		EJBQLAST collectionValuedPathExpression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST collectionValuedPathExpression_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		pathExpression(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		
		int fieldTokenType = p_AST.getType();
		if (fieldTokenType != COLLECTION_CMR_FIELD_ACCESS) {
		EJBQLAST classExpr = (EJBQLAST)p_AST.getFirstChild();
		EJBQLAST field = (EJBQLAST)classExpr.getNextSibling();
		ErrorMsg.error(field.getLine(), field.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_CollectionValuedCMRFieldExpected", //NOI18N
		field.getText(), typeSupport.getTypeName(field.getTypeInfo())));
		p_AST.setType(COLLECTION_CMR_FIELD_ACCESS);
		}
		
		collectionValuedPathExpression_AST = (EJBQLAST)currentAST.root;
		returnAST = collectionValuedPathExpression_AST;
		_retTree = _t;
	}
	
	public final void distinct(AST _t) throws RecognitionException {
		
		EJBQLAST distinct_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST distinct_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case DISTINCT:
		{
			EJBQLAST tmp8_AST = null;
			EJBQLAST tmp8_AST_in = null;
			tmp8_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp8_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp8_AST);
			match(_t,DISTINCT);
			_t = _t.getNextSibling();
			distinct_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case OBJECT:
		case AVG:
		case MAX:
		case MIN:
		case SUM:
		case COUNT:
		case DOT:
		{
			distinct_AST = (EJBQLAST)currentAST.root;
			
			// Insert DISTINCT keyword, in the case of a multi-object selector 
			// having java.util.Set as return type
			if (!finderNotSelector && 
			(method.getReturnType() == java.util.Set.class)) {
			distinct_AST = (EJBQLAST)astFactory.create(DISTINCT,"distinct");
			}
			
			currentAST.root = distinct_AST;
			currentAST.child = distinct_AST!=null &&distinct_AST.getFirstChild()!=null ?
				distinct_AST.getFirstChild() : distinct_AST;
			currentAST.advanceChildToEnd();
			distinct_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = distinct_AST;
		_retTree = _t;
	}
	
	public final void projection(AST _t) throws RecognitionException {
		
		EJBQLAST projection_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST projection_AST = null;
		EJBQLAST o = null;
		EJBQLAST o_AST = null;
		EJBQLAST var = null;
		EJBQLAST var_AST = null;
		EJBQLAST sum = null;
		EJBQLAST sum_AST = null;
		EJBQLAST sumExpr_AST = null;
		EJBQLAST sumExpr = null;
		EJBQLAST avg = null;
		EJBQLAST avg_AST = null;
		EJBQLAST avgExpr_AST = null;
		EJBQLAST avgExpr = null;
		EJBQLAST min = null;
		EJBQLAST min_AST = null;
		EJBQLAST minExpr_AST = null;
		EJBQLAST minExpr = null;
		EJBQLAST max = null;
		EJBQLAST max_AST = null;
		EJBQLAST maxExpr_AST = null;
		EJBQLAST maxExpr = null;
		EJBQLAST c = null;
		EJBQLAST c_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case DOT:
		{
			singleValuedPathExpression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case OBJECT:
		{
			AST __t16 = _t;
			o = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST o_AST_in = null;
			o_AST = (EJBQLAST)astFactory.create(o);
			astFactory.addASTChild(currentAST, o_AST);
			ASTPair __currentAST16 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OBJECT);
			_t = _t.getFirstChild();
			var = (EJBQLAST)_t;
			EJBQLAST var_AST_in = null;
			var_AST = (EJBQLAST)astFactory.create(var);
			astFactory.addASTChild(currentAST, var_AST);
			match(_t,IDENT);
			_t = _t.getNextSibling();
			currentAST = __currentAST16;
			_t = __t16;
			_t = _t.getNextSibling();
			
			String name = var_AST.getText();
			Object decl = symtab.getDeclaration(name);
			Object typeInfo = null;
			if ((decl != null) && 
			(decl instanceof IdentificationVariable)) {
			var_AST.setType(IDENTIFICATION_VAR);
			typeInfo = ((IdentificationVariable)decl).getTypeInfo();
			}
			else {
			ErrorMsg.error(var_AST.getLine(), var_AST.getColumn(), 
			I18NHelper.getMessage(msgs, 
			"EXC_IdentificationVariableExcepted", name)); //NOI18N
			}
			var_AST.setTypeInfo(typeInfo);
			o_AST.setTypeInfo(typeInfo);
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case SUM:
		{
			AST __t17 = _t;
			sum = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST sum_AST_in = null;
			sum_AST = (EJBQLAST)astFactory.create(sum);
			astFactory.addASTChild(currentAST, sum_AST);
			ASTPair __currentAST17 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,SUM);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DISTINCT:
			{
				EJBQLAST tmp9_AST = null;
				EJBQLAST tmp9_AST_in = null;
				tmp9_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
				tmp9_AST_in = (EJBQLAST)_t;
				astFactory.addASTChild(currentAST, tmp9_AST);
				match(_t,DISTINCT);
				_t = _t.getNextSibling();
				break;
			}
			case DOT:
			{
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			sumExpr = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			sumExpr_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST17;
			_t = __t17;
			_t = _t.getNextSibling();
			
			// check numeric type
			Object typeInfo = sumExpr_AST.getTypeInfo();
			if (!typeSupport.isNumberType(typeInfo) ||
			typeSupport.isCharType(typeInfo)) {
			ErrorMsg.error(sumExpr_AST.getLine(), sumExpr_AST.getColumn(),
			I18NHelper.getMessage(msgs,
			"EXC_NumberExprExpected", //NO18N
			typeSupport.getTypeName(typeInfo)));
			}
			sum_AST.setTypeInfo(typeSupport.getSumReturnType(typeInfo));
			isAggregate = true;
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case AVG:
		{
			AST __t19 = _t;
			avg = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST avg_AST_in = null;
			avg_AST = (EJBQLAST)astFactory.create(avg);
			astFactory.addASTChild(currentAST, avg_AST);
			ASTPair __currentAST19 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AVG);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DISTINCT:
			{
				EJBQLAST tmp10_AST = null;
				EJBQLAST tmp10_AST_in = null;
				tmp10_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
				tmp10_AST_in = (EJBQLAST)_t;
				astFactory.addASTChild(currentAST, tmp10_AST);
				match(_t,DISTINCT);
				_t = _t.getNextSibling();
				break;
			}
			case DOT:
			{
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			avgExpr = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			avgExpr_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST19;
			_t = __t19;
			_t = _t.getNextSibling();
			
			// check numeric type
			Object typeInfo = avgExpr_AST.getTypeInfo();
			if (!typeSupport.isNumberType(typeInfo) ||
			typeSupport.isCharType(typeInfo)) {
			ErrorMsg.error(avgExpr_AST.getLine(), avgExpr_AST.getColumn(),
			I18NHelper.getMessage(msgs,
			"EXC_NumberExprExpected", //NO18N
			typeSupport.getTypeName(typeInfo)));
			}
			avg_AST.setTypeInfo(typeSupport.getAvgReturnType(typeInfo));
			isAggregate = true;
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case MIN:
		{
			AST __t21 = _t;
			min = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST min_AST_in = null;
			min_AST = (EJBQLAST)astFactory.create(min);
			astFactory.addASTChild(currentAST, min_AST);
			ASTPair __currentAST21 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MIN);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DISTINCT:
			{
				EJBQLAST tmp11_AST = null;
				EJBQLAST tmp11_AST_in = null;
				tmp11_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
				tmp11_AST_in = (EJBQLAST)_t;
				astFactory.addASTChild(currentAST, tmp11_AST);
				match(_t,DISTINCT);
				_t = _t.getNextSibling();
				break;
			}
			case DOT:
			{
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			minExpr = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			minExpr_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST21;
			_t = __t21;
			_t = _t.getNextSibling();
			
			// check orderable type
			Object typeInfo = minExpr_AST.getTypeInfo();
			if (!typeSupport.isOrderableType(typeInfo)) {
			ErrorMsg.error(minExpr_AST.getLine(), minExpr_AST.getColumn(),
			I18NHelper.getMessage(msgs,
			"EXC_OrderableExpected", //NO18N
			typeSupport.getTypeName(typeInfo)));
			}
			min_AST.setTypeInfo(typeSupport.getMinMaxReturnType(typeInfo));
			isAggregate = true;
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case MAX:
		{
			AST __t23 = _t;
			max = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST max_AST_in = null;
			max_AST = (EJBQLAST)astFactory.create(max);
			astFactory.addASTChild(currentAST, max_AST);
			ASTPair __currentAST23 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MAX);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DISTINCT:
			{
				EJBQLAST tmp12_AST = null;
				EJBQLAST tmp12_AST_in = null;
				tmp12_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
				tmp12_AST_in = (EJBQLAST)_t;
				astFactory.addASTChild(currentAST, tmp12_AST);
				match(_t,DISTINCT);
				_t = _t.getNextSibling();
				break;
			}
			case DOT:
			{
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			maxExpr = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			maxExpr_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST23;
			_t = __t23;
			_t = _t.getNextSibling();
			
			// check orderable type
			Object typeInfo = maxExpr_AST.getTypeInfo();
			if (!typeSupport.isOrderableType(typeInfo)) {
			ErrorMsg.error(maxExpr_AST.getLine(), maxExpr_AST.getColumn(),
			I18NHelper.getMessage(msgs,
			"EXC_OrderableExpected", //NO18N
			typeSupport.getTypeName(typeInfo)));
			}
			max_AST.setTypeInfo(typeSupport.getMinMaxReturnType(typeInfo));
			isAggregate = true;
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case COUNT:
		{
			AST __t25 = _t;
			c = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST c_AST_in = null;
			c_AST = (EJBQLAST)astFactory.create(c);
			astFactory.addASTChild(currentAST, c_AST);
			ASTPair __currentAST25 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,COUNT);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DISTINCT:
			{
				EJBQLAST tmp13_AST = null;
				EJBQLAST tmp13_AST_in = null;
				tmp13_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
				tmp13_AST_in = (EJBQLAST)_t;
				astFactory.addASTChild(currentAST, tmp13_AST);
				match(_t,DISTINCT);
				_t = _t.getNextSibling();
				break;
			}
			case IDENT:
			case DOT:
			{
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			countExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST25;
			_t = __t25;
			_t = _t.getNextSibling();
			
			c_AST.setTypeInfo(typeSupport.longClassType);
			isAggregate = true;
			
			projection_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = projection_AST;
		_retTree = _t;
	}
	
	public final void singleValuedPathExpression(AST _t) throws RecognitionException {
		
		EJBQLAST singleValuedPathExpression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST singleValuedPathExpression_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		pathExpression(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		
		int fieldTokenType = p_AST.getType();
		if ((fieldTokenType != SINGLE_CMR_FIELD_ACCESS) && 
		(fieldTokenType != CMP_FIELD_ACCESS)) {
		EJBQLAST classExpr = (EJBQLAST)p_AST.getFirstChild();
		EJBQLAST field = (EJBQLAST)classExpr.getNextSibling();
		ErrorMsg.error(field.getLine(), field.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_SingleValuedCMROrCMPFieldExpected", //NOI18N
		field.getText(), typeSupport.getTypeName(field.getTypeInfo())));
		p_AST.setType(SINGLE_CMR_FIELD_ACCESS);
		}
		
		singleValuedPathExpression_AST = (EJBQLAST)currentAST.root;
		returnAST = singleValuedPathExpression_AST;
		_retTree = _t;
	}
	
	public final void cmpPathExpression(AST _t) throws RecognitionException {
		
		EJBQLAST cmpPathExpression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST cmpPathExpression_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		pathExpression(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		
		int fieldTokenType = p_AST.getType();
		if ((fieldTokenType != CMP_FIELD_ACCESS)) {
		EJBQLAST classExpr = (EJBQLAST)p_AST.getFirstChild();
		EJBQLAST field = (EJBQLAST)classExpr.getNextSibling();
		ErrorMsg.error(field.getLine(), field.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_CMPFieldExpected", //NOI18N
		field.getText(), typeSupport.getTypeName(field.getTypeInfo())));
		p_AST.setType(CMP_FIELD_ACCESS);
		}
		
		cmpPathExpression_AST = (EJBQLAST)currentAST.root;
		returnAST = cmpPathExpression_AST;
		_retTree = _t;
	}
	
	public final void countExpr(AST _t) throws RecognitionException {
		
		EJBQLAST countExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST countExpr_AST = null;
		EJBQLAST v = null;
		EJBQLAST v_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IDENT:
		{
			v = (EJBQLAST)_t;
			EJBQLAST v_AST_in = null;
			v_AST = (EJBQLAST)astFactory.create(v);
			astFactory.addASTChild(currentAST, v_AST);
			match(_t,IDENT);
			_t = _t.getNextSibling();
			
			String name = v_AST.getText();
			Object decl = symtab.getDeclaration(name);
			Object typeInfo = null;
			if ((decl != null) && 
			(decl instanceof IdentificationVariable)) {
			v_AST.setType(IDENTIFICATION_VAR);
			typeInfo = ((IdentificationVariable)decl).getTypeInfo();
			}
			else {
			ErrorMsg.error(v_AST.getLine(), v_AST.getColumn(), 
			I18NHelper.getMessage(msgs, 
			"EXC_IdentificationVariableExcepted", name)); //NOI18N
			}
			v_AST.setTypeInfo(typeInfo);
			
			countExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			singleValuedPathExpression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			countExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = countExpr_AST;
		_retTree = _t;
	}
	
	public final void expression(AST _t) throws RecognitionException {
		
		EJBQLAST expression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST expression_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case AND:
		case OR:
		{
			conditionalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case EQUAL:
		case NOT_EQUAL:
		case GE:
		case GT:
		case LE:
		case LT:
		{
			relationalExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case PLUS:
		case MINUS:
		case STAR:
		case DIV:
		{
			binaryArithmeticExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT:
		case UNARY_MINUS:
		case UNARY_PLUS:
		{
			unaryExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case BETWEEN:
		case NOT_BETWEEN:
		{
			betweenExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LIKE:
		case NOT_LIKE:
		{
			likeExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case IN:
		case NOT_IN:
		{
			inExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NULL:
		case NOT_NULL:
		{
			nullComparisonExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case EMPTY:
		case NOT_EMPTY:
		{
			emptyCollectionComparisonExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case MEMBER:
		case NOT_MEMBER:
		{
			collectionMemberExpr(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case CONCAT:
		case SUBSTRING:
		case LOCATE:
		case LENGTH:
		case ABS:
		case SQRT:
		case MOD:
		{
			function(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case TRUE:
		case FALSE:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		case IDENT:
		case DOT:
		case INPUT_PARAMETER:
		{
			primary(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			expression_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = expression_AST;
		_retTree = _t;
	}
	
	public final void orderbyItem(AST _t) throws RecognitionException {
		
		EJBQLAST orderbyItem_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST orderbyItem_AST = null;
		EJBQLAST expr_AST = null;
		EJBQLAST expr = null;
		
		expr = _t==ASTNULL ? null : (EJBQLAST)_t;
		cmpPathExpression(_t);
		_t = _retTree;
		expr_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ASC:
		{
			EJBQLAST tmp14_AST = null;
			EJBQLAST tmp14_AST_in = null;
			tmp14_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp14_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp14_AST);
			match(_t,ASC);
			_t = _t.getNextSibling();
			break;
		}
		case DESC:
		{
			EJBQLAST tmp15_AST = null;
			EJBQLAST tmp15_AST_in = null;
			tmp15_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp15_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp15_AST);
			match(_t,DESC);
			_t = _t.getNextSibling();
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		
		// check orderable type
		Object typeInfo = expr_AST.getTypeInfo();
		if (!typeSupport.isOrderableType(typeInfo)) {
		ErrorMsg.error(expr_AST.getLine(), expr_AST.getColumn(),
		I18NHelper.getMessage(msgs,
		"EXC_OrderableOrderbyClauseExpected", //NO18N
		typeSupport.getTypeName(typeInfo)));
		}
		
		orderbyItem_AST = (EJBQLAST)currentAST.root;
		returnAST = orderbyItem_AST;
		_retTree = _t;
	}
	
	public final void conditionalExpr(AST _t) throws RecognitionException {
		
		EJBQLAST conditionalExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST conditionalExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST left1_AST = null;
		EJBQLAST left1 = null;
		EJBQLAST right1_AST = null;
		EJBQLAST right1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST left2_AST = null;
		EJBQLAST left2 = null;
		EJBQLAST right2_AST = null;
		EJBQLAST right2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case AND:
		{
			AST __t38 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST38 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,AND);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST38;
			_t = __t38;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(analyseConditionalExpr(op1_AST, left1_AST, right1_AST));
			
			conditionalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case OR:
		{
			AST __t39 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST39 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,OR);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST39;
			_t = __t39;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(analyseConditionalExpr(op2_AST, left2_AST, right2_AST));
			
			conditionalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = conditionalExpr_AST;
		_retTree = _t;
	}
	
	public final void relationalExpr(AST _t) throws RecognitionException {
		
		EJBQLAST relationalExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST relationalExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST left1_AST = null;
		EJBQLAST left1 = null;
		EJBQLAST right1_AST = null;
		EJBQLAST right1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST left2_AST = null;
		EJBQLAST left2 = null;
		EJBQLAST right2_AST = null;
		EJBQLAST right2 = null;
		EJBQLAST op3 = null;
		EJBQLAST op3_AST = null;
		EJBQLAST left3_AST = null;
		EJBQLAST left3 = null;
		EJBQLAST right3_AST = null;
		EJBQLAST right3 = null;
		EJBQLAST op4 = null;
		EJBQLAST op4_AST = null;
		EJBQLAST left4_AST = null;
		EJBQLAST left4 = null;
		EJBQLAST right4_AST = null;
		EJBQLAST right4 = null;
		EJBQLAST op5 = null;
		EJBQLAST op5_AST = null;
		EJBQLAST left5_AST = null;
		EJBQLAST left5 = null;
		EJBQLAST right5_AST = null;
		EJBQLAST right5 = null;
		EJBQLAST op6 = null;
		EJBQLAST op6_AST = null;
		EJBQLAST left6_AST = null;
		EJBQLAST left6 = null;
		EJBQLAST right6_AST = null;
		EJBQLAST right6 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case EQUAL:
		{
			AST __t41 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST41 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,EQUAL);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST41;
			_t = __t41;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(analyseEqualityExpr(op1_AST, left1_AST, right1_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_EQUAL:
		{
			AST __t42 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST42 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_EQUAL);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST42;
			_t = __t42;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(analyseEqualityExpr(op2_AST, left2_AST, right2_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LT:
		{
			AST __t43 = _t;
			op3 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op3_AST_in = null;
			op3_AST = (EJBQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST43 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LT);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST43;
			_t = __t43;
			_t = _t.getNextSibling();
			
			op3_AST.setTypeInfo(analyseRelationalExpr(op3_AST, left3_AST, right3_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LE:
		{
			AST __t44 = _t;
			op4 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op4_AST_in = null;
			op4_AST = (EJBQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST44 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LE);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST44;
			_t = __t44;
			_t = _t.getNextSibling();
			
			op4_AST.setTypeInfo(analyseRelationalExpr(op4_AST, left4_AST, right4_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case GT:
		{
			AST __t45 = _t;
			op5 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op5_AST_in = null;
			op5_AST = (EJBQLAST)astFactory.create(op5);
			astFactory.addASTChild(currentAST, op5_AST);
			ASTPair __currentAST45 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GT);
			_t = _t.getFirstChild();
			left5 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left5_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right5 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right5_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST45;
			_t = __t45;
			_t = _t.getNextSibling();
			
			op5_AST.setTypeInfo(analyseRelationalExpr(op5_AST, left5_AST, right5_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case GE:
		{
			AST __t46 = _t;
			op6 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op6_AST_in = null;
			op6_AST = (EJBQLAST)astFactory.create(op6);
			astFactory.addASTChild(currentAST, op6_AST);
			ASTPair __currentAST46 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,GE);
			_t = _t.getFirstChild();
			left6 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left6_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right6 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right6_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST46;
			_t = __t46;
			_t = _t.getNextSibling();
			
			op6_AST.setTypeInfo(analyseRelationalExpr(op6_AST, left6_AST, right6_AST));
			
			relationalExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = relationalExpr_AST;
		_retTree = _t;
	}
	
	public final void binaryArithmeticExpr(AST _t) throws RecognitionException {
		
		EJBQLAST binaryArithmeticExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST binaryArithmeticExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST left1_AST = null;
		EJBQLAST left1 = null;
		EJBQLAST right1_AST = null;
		EJBQLAST right1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST left2_AST = null;
		EJBQLAST left2 = null;
		EJBQLAST right2_AST = null;
		EJBQLAST right2 = null;
		EJBQLAST op3 = null;
		EJBQLAST op3_AST = null;
		EJBQLAST left3_AST = null;
		EJBQLAST left3 = null;
		EJBQLAST right3_AST = null;
		EJBQLAST right3 = null;
		EJBQLAST op4 = null;
		EJBQLAST op4_AST = null;
		EJBQLAST left4_AST = null;
		EJBQLAST left4 = null;
		EJBQLAST right4_AST = null;
		EJBQLAST right4 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case PLUS:
		{
			AST __t48 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST48 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,PLUS);
			_t = _t.getFirstChild();
			left1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST48;
			_t = __t48;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(analyseBinaryArithmeticExpr(op1_AST, left1_AST, right1_AST));
			
			binaryArithmeticExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case MINUS:
		{
			AST __t49 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST49 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MINUS);
			_t = _t.getFirstChild();
			left2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST49;
			_t = __t49;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(analyseBinaryArithmeticExpr(op2_AST, left2_AST, right2_AST));
			
			binaryArithmeticExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case STAR:
		{
			AST __t50 = _t;
			op3 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op3_AST_in = null;
			op3_AST = (EJBQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST50 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,STAR);
			_t = _t.getFirstChild();
			left3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST50;
			_t = __t50;
			_t = _t.getNextSibling();
			
			op3_AST.setTypeInfo(analyseBinaryArithmeticExpr(op3_AST, left3_AST, right3_AST));
			
			binaryArithmeticExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DIV:
		{
			AST __t51 = _t;
			op4 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op4_AST_in = null;
			op4_AST = (EJBQLAST)astFactory.create(op4);
			astFactory.addASTChild(currentAST, op4_AST);
			ASTPair __currentAST51 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,DIV);
			_t = _t.getFirstChild();
			left4 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			left4_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			right4 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			right4_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST51;
			_t = __t51;
			_t = _t.getNextSibling();
			
			op4_AST.setTypeInfo(analyseBinaryArithmeticExpr(op4_AST, left4_AST, right4_AST));
			
			binaryArithmeticExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = binaryArithmeticExpr_AST;
		_retTree = _t;
	}
	
	public final void unaryExpr(AST _t) throws RecognitionException {
		
		EJBQLAST unaryExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST unaryExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST arg1_AST = null;
		EJBQLAST arg1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST arg2_AST = null;
		EJBQLAST arg2 = null;
		EJBQLAST op3 = null;
		EJBQLAST op3_AST = null;
		EJBQLAST arg3_AST = null;
		EJBQLAST arg3 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case UNARY_PLUS:
		{
			AST __t53 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST53 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,UNARY_PLUS);
			_t = _t.getFirstChild();
			arg1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST53;
			_t = __t53;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(analyseUnaryArithmeticExpr(op1_AST, arg1_AST));
			
			unaryExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case UNARY_MINUS:
		{
			AST __t54 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST54 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,UNARY_MINUS);
			_t = _t.getFirstChild();
			arg2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST54;
			_t = __t54;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(analyseUnaryArithmeticExpr(op2_AST, arg2_AST));
			
			unaryExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT:
		{
			AST __t55 = _t;
			op3 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op3_AST_in = null;
			op3_AST = (EJBQLAST)astFactory.create(op3);
			astFactory.addASTChild(currentAST, op3_AST);
			ASTPair __currentAST55 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT);
			_t = _t.getFirstChild();
			arg3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST55;
			_t = __t55;
			_t = _t.getNextSibling();
			
			Object typeInfo = typeSupport.errorType;
			Object arg = arg3_AST.getTypeInfo();
			if (typeSupport.isErrorType(arg))
			typeInfo = typeSupport.errorType;
			else if (typeSupport.isBooleanType(arg))
			typeInfo = arg;
			else {
			ErrorMsg.error(op3_AST.getLine(), op3_AST.getColumn(), 
			I18NHelper.getMessage(msgs, "EXC_InvalidArguments", //NOI18N
			op3_AST.getText())); 
			}
			op3_AST.setTypeInfo(typeInfo);
			
			unaryExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = unaryExpr_AST;
		_retTree = _t;
	}
	
	public final void betweenExpr(AST _t) throws RecognitionException {
		
		EJBQLAST betweenExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST betweenExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST expr1_AST = null;
		EJBQLAST expr1 = null;
		EJBQLAST lower1_AST = null;
		EJBQLAST lower1 = null;
		EJBQLAST upper1_AST = null;
		EJBQLAST upper1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST expr2_AST = null;
		EJBQLAST expr2 = null;
		EJBQLAST lower2_AST = null;
		EJBQLAST lower2 = null;
		EJBQLAST upper2_AST = null;
		EJBQLAST upper2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case BETWEEN:
		{
			AST __t57 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST57 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,BETWEEN);
			_t = _t.getFirstChild();
			expr1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			expr1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			lower1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			lower1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			upper1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			upper1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST57;
			_t = __t57;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo((isNumberExpr(expr1_AST) && isNumberExpr(lower1_AST) && isNumberExpr(upper1_AST)) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			betweenExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_BETWEEN:
		{
			AST __t58 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST58 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_BETWEEN);
			_t = _t.getFirstChild();
			expr2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			expr2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			lower2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			lower2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			upper2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			upper2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST58;
			_t = __t58;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo((isNumberExpr(expr2_AST) && isNumberExpr(lower2_AST) && isNumberExpr(upper2_AST)) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			betweenExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = betweenExpr_AST;
		_retTree = _t;
	}
	
	public final void likeExpr(AST _t) throws RecognitionException {
		
		EJBQLAST likeExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST likeExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST expr1_AST = null;
		EJBQLAST expr1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST expr2_AST = null;
		EJBQLAST expr2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case LIKE:
		{
			AST __t60 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST60 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,LIKE);
			_t = _t.getFirstChild();
			expr1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			expr1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			pattern(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			escape(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST60;
			_t = __t60;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(isStringExpr(expr1_AST) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			likeExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_LIKE:
		{
			AST __t61 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST61 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_LIKE);
			_t = _t.getFirstChild();
			expr2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			expr2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			pattern(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			escape(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST61;
			_t = __t61;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(isStringExpr(expr2_AST) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			likeExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = likeExpr_AST;
		_retTree = _t;
	}
	
	public final void inExpr(AST _t) throws RecognitionException {
		
		EJBQLAST inExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST inExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST expr1_AST = null;
		EJBQLAST expr1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST expr2_AST = null;
		EJBQLAST expr2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IN:
		{
			AST __t67 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST67 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,IN);
			_t = _t.getFirstChild();
			expr1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			expr1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			inCollection(_t,expr1_AST.getTypeInfo());
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST67;
			_t = __t67;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(isNumberOrStringExpr(expr1_AST) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			inExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_IN:
		{
			AST __t68 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST68 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_IN);
			_t = _t.getFirstChild();
			expr2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			cmpPathExpression(_t);
			_t = _retTree;
			expr2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			inCollection(_t,expr2_AST.getTypeInfo());
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST68;
			_t = __t68;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(isNumberOrStringExpr(expr2_AST) ? 
			typeSupport.booleanType : typeSupport.errorType);
			
			inExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = inExpr_AST;
		_retTree = _t;
	}
	
	public final void nullComparisonExpr(AST _t) throws RecognitionException {
		
		EJBQLAST nullComparisonExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST nullComparisonExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case NULL:
		{
			AST __t70 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST70 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NULL);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DOT:
			{
				singleValuedPathExpression(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
				break;
			}
			case INPUT_PARAMETER:
			{
				inputParameter(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			currentAST = __currentAST70;
			_t = __t70;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(typeSupport.booleanType);
			
			nullComparisonExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_NULL:
		{
			AST __t72 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST72 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_NULL);
			_t = _t.getFirstChild();
			{
			if (_t==null) _t=ASTNULL;
			switch ( _t.getType()) {
			case DOT:
			{
				singleValuedPathExpression(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
				break;
			}
			case INPUT_PARAMETER:
			{
				inputParameter(_t);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
				break;
			}
			default:
			{
				throw new NoViableAltException(_t);
			}
			}
			}
			currentAST = __currentAST72;
			_t = __t72;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(typeSupport.booleanType);
			
			nullComparisonExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = nullComparisonExpr_AST;
		_retTree = _t;
	}
	
	public final void emptyCollectionComparisonExpr(AST _t) throws RecognitionException {
		
		EJBQLAST emptyCollectionComparisonExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST emptyCollectionComparisonExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST col1_AST = null;
		EJBQLAST col1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST col2_AST = null;
		EJBQLAST col2 = null;
		
		Object elementTypeInfo = null; 
		
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case EMPTY:
		{
			AST __t75 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST75 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,EMPTY);
			_t = _t.getFirstChild();
			col1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			collectionValuedPathExpression(_t);
			_t = _retTree;
			col1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST75;
			_t = __t75;
			_t = _t.getNextSibling();
			
			elementTypeInfo = analyseCollectionValuedCMRField(col1_AST);
			op1_AST.setTypeInfo(typeSupport.isErrorType(elementTypeInfo) ? 
			typeSupport.errorType : typeSupport.booleanType );
			
			emptyCollectionComparisonExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_EMPTY:
		{
			AST __t76 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST76 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_EMPTY);
			_t = _t.getFirstChild();
			col2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			collectionValuedPathExpression(_t);
			_t = _retTree;
			col2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST76;
			_t = __t76;
			_t = _t.getNextSibling();
			
			elementTypeInfo = analyseCollectionValuedCMRField(col2_AST);
			op2_AST.setTypeInfo(typeSupport.isErrorType(elementTypeInfo) ? 
			typeSupport.errorType : typeSupport.booleanType );
			
			emptyCollectionComparisonExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = emptyCollectionComparisonExpr_AST;
		_retTree = _t;
	}
	
	public final void collectionMemberExpr(AST _t) throws RecognitionException {
		
		EJBQLAST collectionMemberExpr_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST collectionMemberExpr_AST = null;
		EJBQLAST op1 = null;
		EJBQLAST op1_AST = null;
		EJBQLAST value1_AST = null;
		EJBQLAST value1 = null;
		EJBQLAST col1_AST = null;
		EJBQLAST col1 = null;
		EJBQLAST op2 = null;
		EJBQLAST op2_AST = null;
		EJBQLAST value2_AST = null;
		EJBQLAST value2 = null;
		EJBQLAST col2_AST = null;
		EJBQLAST col2 = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case MEMBER:
		{
			AST __t78 = _t;
			op1 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op1_AST_in = null;
			op1_AST = (EJBQLAST)astFactory.create(op1);
			astFactory.addASTChild(currentAST, op1_AST);
			ASTPair __currentAST78 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,MEMBER);
			_t = _t.getFirstChild();
			value1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			member(_t);
			_t = _retTree;
			value1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			col1 = _t==ASTNULL ? null : (EJBQLAST)_t;
			collectionValuedPathExpression(_t);
			_t = _retTree;
			col1_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST78;
			_t = __t78;
			_t = _t.getNextSibling();
			
			op1_AST.setTypeInfo(analyseMemberExpr(op1_AST, value1_AST, col1_AST));
			
			collectionMemberExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case NOT_MEMBER:
		{
			AST __t79 = _t;
			op2 = _t==ASTNULL ? null :(EJBQLAST)_t;
			EJBQLAST op2_AST_in = null;
			op2_AST = (EJBQLAST)astFactory.create(op2);
			astFactory.addASTChild(currentAST, op2_AST);
			ASTPair __currentAST79 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,NOT_MEMBER);
			_t = _t.getFirstChild();
			value2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			member(_t);
			_t = _retTree;
			value2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			col2 = _t==ASTNULL ? null : (EJBQLAST)_t;
			collectionValuedPathExpression(_t);
			_t = _retTree;
			col2_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST79;
			_t = __t79;
			_t = _t.getNextSibling();
			
			op2_AST.setTypeInfo(analyseMemberExpr(op2_AST, value2_AST, col2_AST));
			
			collectionMemberExpr_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = collectionMemberExpr_AST;
		_retTree = _t;
	}
	
	public final void function(AST _t) throws RecognitionException {
		
		EJBQLAST function_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST function_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case CONCAT:
		{
			concat(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case SUBSTRING:
		{
			substring(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LENGTH:
		{
			length(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LOCATE:
		{
			locate(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case ABS:
		{
			abs(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case SQRT:
		{
			sqrt(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case MOD:
		{
			mod(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			function_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = function_AST;
		_retTree = _t;
	}
	
	public final void primary(AST _t) throws RecognitionException {
		
		EJBQLAST primary_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST primary_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TRUE:
		case FALSE:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			literal(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			singleValuedPathExpression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case IDENT:
		{
			identificationVariable(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INPUT_PARAMETER:
		{
			inputParameter(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			primary_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = primary_AST;
		_retTree = _t;
	}
	
	public final void pattern(AST _t) throws RecognitionException {
		
		EJBQLAST pattern_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST pattern_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case STRING_LITERAL:
		{
			EJBQLAST tmp16_AST = null;
			EJBQLAST tmp16_AST_in = null;
			tmp16_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp16_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp16_AST);
			match(_t,STRING_LITERAL);
			_t = _t.getNextSibling();
			pattern_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INPUT_PARAMETER:
		{
			p = _t==ASTNULL ? null : (EJBQLAST)_t;
			inputParameter(_t);
			_t = _retTree;
			p_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			if (!typeSupport.isStringType(p_AST.getTypeInfo())) {
			ErrorMsg.error(p_AST.getLine(), p_AST.getColumn(),
			I18NHelper.getMessage(msgs, "EXC_InvalidPatternDefinition",
			p_AST.getText())); //NOI18N
			}
			
			pattern_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = pattern_AST;
		_retTree = _t;
	}
	
	public final void escape(AST _t) throws RecognitionException {
		
		EJBQLAST escape_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST escape_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case ESCAPE:
		{
			AST __t64 = _t;
			EJBQLAST tmp17_AST = null;
			EJBQLAST tmp17_AST_in = null;
			tmp17_AST = (EJBQLAST)astFactory.create((EJBQLAST)_t);
			tmp17_AST_in = (EJBQLAST)_t;
			astFactory.addASTChild(currentAST, tmp17_AST);
			ASTPair __currentAST64 = currentAST.copy();
			currentAST.root = currentAST.child;
			currentAST.child = null;
			match(_t,ESCAPE);
			_t = _t.getFirstChild();
			escapeCharacter(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			currentAST = __currentAST64;
			_t = __t64;
			_t = _t.getNextSibling();
			escape_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case 3:
		{
			escape_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = escape_AST;
		_retTree = _t;
	}
	
	public final void inputParameter(AST _t) throws RecognitionException {
		
		EJBQLAST inputParameter_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST inputParameter_AST = null;
		EJBQLAST param = null;
		EJBQLAST param_AST = null;
		
		param = (EJBQLAST)_t;
		EJBQLAST param_AST_in = null;
		param_AST = (EJBQLAST)astFactory.create(param);
		astFactory.addASTChild(currentAST, param_AST);
		match(_t,INPUT_PARAMETER);
		_t = _t.getNextSibling();
		
		Object typeInfo = typeSupport.getTypeInfo(
		paramSupport.getParameterType(param_AST.getText()));
		param_AST.setTypeInfo(typeInfo);
		
		inputParameter_AST = (EJBQLAST)currentAST.root;
		returnAST = inputParameter_AST;
		_retTree = _t;
	}
	
	public final void escapeCharacter(AST _t) throws RecognitionException {
		
		EJBQLAST escapeCharacter_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST escapeCharacter_AST = null;
		EJBQLAST s = null;
		EJBQLAST s_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case STRING_LITERAL:
		{
			s = (EJBQLAST)_t;
			EJBQLAST s_AST_in = null;
			s_AST = (EJBQLAST)astFactory.create(s);
			astFactory.addASTChild(currentAST, s_AST);
			match(_t,STRING_LITERAL);
			_t = _t.getNextSibling();
			
			String literal = s_AST.getText();
			// String must be single charater string literal =>
			// either '<char>' or ''''
			if (!isSingleCharacterStringLiteral(s_AST.getText())) {
			ErrorMsg.error(s_AST.getLine(), s_AST.getColumn(),
			I18NHelper.getMessage(msgs, 
			"EXC_InvalidEscapeDefinition", s_AST.getText())); //NOI18N
			}
			
			escapeCharacter_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INPUT_PARAMETER:
		{
			p = _t==ASTNULL ? null : (EJBQLAST)_t;
			inputParameter(_t);
			_t = _retTree;
			p_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			Object paramType = p_AST.getTypeInfo();
			if (!typeSupport.isCharType(paramType)) {
			ErrorMsg.error(p_AST.getLine(), p_AST.getColumn(),
			I18NHelper.getMessage(msgs, 
			"EXC_InvalidEscapeParameterDefinition", p_AST.getText())); //NOI18N
			}
			
			escapeCharacter_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = escapeCharacter_AST;
		_retTree = _t;
	}
	
	public final void inCollection(AST _t,
		Object valueExprTypeInfo
	) throws RecognitionException {
		
		EJBQLAST inCollection_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST inCollection_AST = null;
		
		{
		int _cnt109=0;
		_loop109:
		do {
			if (_t==null) _t=ASTNULL;
			if ((_tokenSet_0.member(_t.getType()))) {
				inCollectionElement(_t,valueExprTypeInfo);
				_t = _retTree;
				astFactory.addASTChild(currentAST, returnAST);
			}
			else {
				if ( _cnt109>=1 ) { break _loop109; } else {throw new NoViableAltException(_t);}
			}
			
			_cnt109++;
		} while (true);
		}
		inCollection_AST = (EJBQLAST)currentAST.root;
		returnAST = inCollection_AST;
		_retTree = _t;
	}
	
	public final void member(AST _t) throws RecognitionException {
		
		EJBQLAST member_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST member_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IDENT:
		{
			identificationVariable(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			member_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INPUT_PARAMETER:
		{
			inputParameter(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			member_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			singleValuedCmrPathExpression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			member_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = member_AST;
		_retTree = _t;
	}
	
	public final void identificationVariable(AST _t) throws RecognitionException {
		
		EJBQLAST identificationVariable_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST identificationVariable_AST = null;
		EJBQLAST i = null;
		EJBQLAST i_AST = null;
		
		i = (EJBQLAST)_t;
		EJBQLAST i_AST_in = null;
		i_AST = (EJBQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		
		String name = i_AST.getText();
		Object decl = symtab.getDeclaration(name);
		// check for identification variables
		if ((decl != null) && (decl instanceof IdentificationVariable)) {
		i_AST.setType(IDENTIFICATION_VAR);
		i_AST.setTypeInfo(((IdentificationVariable)decl).getTypeInfo());
		}
		else {
		i_AST.setTypeInfo(typeSupport.errorType);
		ErrorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_UndefinedIdentifier", name)); //NOI18N
		
		}
		
		identificationVariable_AST = (EJBQLAST)currentAST.root;
		returnAST = identificationVariable_AST;
		_retTree = _t;
	}
	
	public final void singleValuedCmrPathExpression(AST _t) throws RecognitionException {
		
		EJBQLAST singleValuedCmrPathExpression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST singleValuedCmrPathExpression_AST = null;
		EJBQLAST p_AST = null;
		EJBQLAST p = null;
		
		p = _t==ASTNULL ? null : (EJBQLAST)_t;
		pathExpression(_t);
		_t = _retTree;
		p_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		
		int fieldTokenType = p_AST.getType();
		if (fieldTokenType != SINGLE_CMR_FIELD_ACCESS) {
		EJBQLAST classExpr = (EJBQLAST)p_AST.getFirstChild();
		EJBQLAST field = (EJBQLAST)classExpr.getNextSibling();
		ErrorMsg.error(field.getLine(), field.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_SingleValuedCMRFieldExpected", //NOI18N
		field.getText(), typeSupport.getTypeName(field.getTypeInfo())));
		p_AST.setType(COLLECTION_CMR_FIELD_ACCESS);
		}
		
		singleValuedCmrPathExpression_AST = (EJBQLAST)currentAST.root;
		returnAST = singleValuedCmrPathExpression_AST;
		_retTree = _t;
	}
	
	public final void concat(AST _t) throws RecognitionException {
		
		EJBQLAST concat_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST concat_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST arg1_AST = null;
		EJBQLAST arg1 = null;
		EJBQLAST arg2_AST = null;
		EJBQLAST arg2 = null;
		
		AST __t83 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST83 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,CONCAT);
		_t = _t.getFirstChild();
		arg1 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg1_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		arg2 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg2_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST83;
		_t = __t83;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo((isStringExpr(arg1_AST) && isStringExpr(arg2_AST)) ?
		typeSupport.stringType : typeSupport.errorType);
		
		concat_AST = (EJBQLAST)currentAST.root;
		returnAST = concat_AST;
		_retTree = _t;
	}
	
	public final void substring(AST _t) throws RecognitionException {
		
		EJBQLAST substring_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST substring_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST arg1_AST = null;
		EJBQLAST arg1 = null;
		EJBQLAST arg2_AST = null;
		EJBQLAST arg2 = null;
		EJBQLAST arg3_AST = null;
		EJBQLAST arg3 = null;
		
		AST __t85 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST85 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,SUBSTRING);
		_t = _t.getFirstChild();
		arg1 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg1_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		arg2 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg2_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		arg3 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg3_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST85;
		_t = __t85;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo((isStringExpr(arg1_AST) && isIntExpr(arg2_AST) && isIntExpr(arg3_AST)) ? 
		typeSupport.stringType : typeSupport.errorType);
		
		substring_AST = (EJBQLAST)currentAST.root;
		returnAST = substring_AST;
		_retTree = _t;
	}
	
	public final void length(AST _t) throws RecognitionException {
		
		EJBQLAST length_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST length_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST arg_AST = null;
		EJBQLAST arg = null;
		
		AST __t87 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST87 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,LENGTH);
		_t = _t.getFirstChild();
		arg = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST87;
		_t = __t87;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo(isStringExpr(arg_AST) ? 
		typeSupport.intType : typeSupport.errorType);
		
		length_AST = (EJBQLAST)currentAST.root;
		returnAST = length_AST;
		_retTree = _t;
	}
	
	public final void locate(AST _t) throws RecognitionException {
		
		EJBQLAST locate_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST locate_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST arg1_AST = null;
		EJBQLAST arg1 = null;
		EJBQLAST arg2_AST = null;
		EJBQLAST arg2 = null;
		EJBQLAST arg3_AST = null;
		EJBQLAST arg3 = null;
		
		AST __t89 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST89 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,LOCATE);
		_t = _t.getFirstChild();
		arg1 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg1_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		arg2 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg2_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		{
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case NULL:
		case TRUE:
		case FALSE:
		case NOT:
		case AND:
		case OR:
		case BETWEEN:
		case LIKE:
		case IN:
		case EMPTY:
		case MEMBER:
		case CONCAT:
		case SUBSTRING:
		case LOCATE:
		case LENGTH:
		case ABS:
		case SQRT:
		case MOD:
		case EQUAL:
		case NOT_EQUAL:
		case GE:
		case GT:
		case LE:
		case LT:
		case PLUS:
		case MINUS:
		case STAR:
		case DIV:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		case IDENT:
		case DOT:
		case INPUT_PARAMETER:
		case UNARY_MINUS:
		case UNARY_PLUS:
		case NOT_BETWEEN:
		case NOT_LIKE:
		case NOT_IN:
		case NOT_NULL:
		case NOT_EMPTY:
		case NOT_MEMBER:
		{
			arg3 = _t==ASTNULL ? null : (EJBQLAST)_t;
			expression(_t);
			_t = _retTree;
			arg3_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			break;
		}
		case 3:
		{
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		}
		currentAST = __currentAST89;
		_t = __t89;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo((isStringExpr(arg1_AST) && isStringExpr(arg2_AST) && 
		((arg3_AST == null) || isIntExpr(arg3_AST))) ?
		typeSupport.intType : typeSupport.errorType);
		
		locate_AST = (EJBQLAST)currentAST.root;
		returnAST = locate_AST;
		_retTree = _t;
	}
	
	public final void abs(AST _t) throws RecognitionException {
		
		EJBQLAST abs_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST abs_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST expr_AST = null;
		EJBQLAST expr = null;
		
		AST __t92 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST92 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,ABS);
		_t = _t.getFirstChild();
		expr = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		expr_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST92;
		_t = __t92;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo(isNumberExpr(expr_AST) ? 
		expr_AST.getTypeInfo() : typeSupport.errorType);
		
		abs_AST = (EJBQLAST)currentAST.root;
		returnAST = abs_AST;
		_retTree = _t;
	}
	
	public final void sqrt(AST _t) throws RecognitionException {
		
		EJBQLAST sqrt_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST sqrt_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST expr_AST = null;
		EJBQLAST expr = null;
		
		AST __t94 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST94 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,SQRT);
		_t = _t.getFirstChild();
		expr = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		expr_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST94;
		_t = __t94;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo(isDoubleExpr(expr_AST) ? 
		expr_AST.getTypeInfo() : typeSupport.errorType);
		
		sqrt_AST = (EJBQLAST)currentAST.root;
		returnAST = sqrt_AST;
		_retTree = _t;
	}
	
	public final void mod(AST _t) throws RecognitionException {
		
		EJBQLAST mod_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST mod_AST = null;
		EJBQLAST op = null;
		EJBQLAST op_AST = null;
		EJBQLAST arg1_AST = null;
		EJBQLAST arg1 = null;
		EJBQLAST arg2_AST = null;
		EJBQLAST arg2 = null;
		
		AST __t96 = _t;
		op = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST op_AST_in = null;
		op_AST = (EJBQLAST)astFactory.create(op);
		astFactory.addASTChild(currentAST, op_AST);
		ASTPair __currentAST96 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,MOD);
		_t = _t.getFirstChild();
		arg1 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg1_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		arg2 = _t==ASTNULL ? null : (EJBQLAST)_t;
		expression(_t);
		_t = _retTree;
		arg2_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		currentAST = __currentAST96;
		_t = __t96;
		_t = _t.getNextSibling();
		
		op_AST.setTypeInfo((isIntExpr(arg1_AST) && isIntExpr(arg2_AST)) ? 
		typeSupport.intType : typeSupport.errorType);
		
		mod_AST = (EJBQLAST)currentAST.root;
		returnAST = mod_AST;
		_retTree = _t;
	}
	
	public final void literal(AST _t) throws RecognitionException {
		
		EJBQLAST literal_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST literal_AST = null;
		EJBQLAST b1 = null;
		EJBQLAST b1_AST = null;
		EJBQLAST b2 = null;
		EJBQLAST b2_AST = null;
		EJBQLAST s = null;
		EJBQLAST s_AST = null;
		EJBQLAST i = null;
		EJBQLAST i_AST = null;
		EJBQLAST l = null;
		EJBQLAST l_AST = null;
		EJBQLAST f = null;
		EJBQLAST f_AST = null;
		EJBQLAST d = null;
		EJBQLAST d_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TRUE:
		{
			b1 = (EJBQLAST)_t;
			EJBQLAST b1_AST_in = null;
			b1_AST = (EJBQLAST)astFactory.create(b1);
			astFactory.addASTChild(currentAST, b1_AST);
			match(_t,TRUE);
			_t = _t.getNextSibling();
			b1_AST.setTypeInfo(typeSupport.booleanType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case FALSE:
		{
			b2 = (EJBQLAST)_t;
			EJBQLAST b2_AST_in = null;
			b2_AST = (EJBQLAST)astFactory.create(b2);
			astFactory.addASTChild(currentAST, b2_AST);
			match(_t,FALSE);
			_t = _t.getNextSibling();
			b2_AST.setTypeInfo(typeSupport.booleanType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case STRING_LITERAL:
		{
			s = (EJBQLAST)_t;
			EJBQLAST s_AST_in = null;
			s_AST = (EJBQLAST)astFactory.create(s);
			astFactory.addASTChild(currentAST, s_AST);
			match(_t,STRING_LITERAL);
			_t = _t.getNextSibling();
			s_AST.setTypeInfo(typeSupport.stringType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INT_LITERAL:
		{
			i = (EJBQLAST)_t;
			EJBQLAST i_AST_in = null;
			i_AST = (EJBQLAST)astFactory.create(i);
			astFactory.addASTChild(currentAST, i_AST);
			match(_t,INT_LITERAL);
			_t = _t.getNextSibling();
			i_AST.setTypeInfo(typeSupport.intType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case LONG_LITERAL:
		{
			l = (EJBQLAST)_t;
			EJBQLAST l_AST_in = null;
			l_AST = (EJBQLAST)astFactory.create(l);
			astFactory.addASTChild(currentAST, l_AST);
			match(_t,LONG_LITERAL);
			_t = _t.getNextSibling();
			l_AST.setTypeInfo(typeSupport.longType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case FLOAT_LITERAL:
		{
			f = (EJBQLAST)_t;
			EJBQLAST f_AST_in = null;
			f_AST = (EJBQLAST)astFactory.create(f);
			astFactory.addASTChild(currentAST, f_AST);
			match(_t,FLOAT_LITERAL);
			_t = _t.getNextSibling();
			f_AST.setTypeInfo(typeSupport.floatType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DOUBLE_LITERAL:
		{
			d = (EJBQLAST)_t;
			EJBQLAST d_AST_in = null;
			d_AST = (EJBQLAST)astFactory.create(d);
			astFactory.addASTChild(currentAST, d_AST);
			match(_t,DOUBLE_LITERAL);
			_t = _t.getNextSibling();
			d_AST.setTypeInfo(typeSupport.doubleType);
			literal_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = literal_AST;
		_retTree = _t;
	}
	
	public final void pathExpression(AST _t) throws RecognitionException {
		
		EJBQLAST pathExpression_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST pathExpression_AST = null;
		EJBQLAST dot = null;
		EJBQLAST dot_AST = null;
		EJBQLAST o_AST = null;
		EJBQLAST o = null;
		EJBQLAST i = null;
		EJBQLAST i_AST = null;
		
		AST __t100 = _t;
		dot = _t==ASTNULL ? null :(EJBQLAST)_t;
		EJBQLAST dot_AST_in = null;
		dot_AST = (EJBQLAST)astFactory.create(dot);
		astFactory.addASTChild(currentAST, dot_AST);
		ASTPair __currentAST100 = currentAST.copy();
		currentAST.root = currentAST.child;
		currentAST.child = null;
		match(_t,DOT);
		_t = _t.getFirstChild();
		o = _t==ASTNULL ? null : (EJBQLAST)_t;
		objectDenoter(_t);
		_t = _retTree;
		o_AST = (EJBQLAST)returnAST;
		astFactory.addASTChild(currentAST, returnAST);
		i = (EJBQLAST)_t;
		EJBQLAST i_AST_in = null;
		i_AST = (EJBQLAST)astFactory.create(i);
		astFactory.addASTChild(currentAST, i_AST);
		match(_t,IDENT);
		_t = _t.getNextSibling();
		currentAST = __currentAST100;
		_t = __t100;
		_t = _t.getNextSibling();
		
		String fieldName = i_AST.getText();
		Object typeInfo = o_AST.getTypeInfo();
		Object fieldTypeInfo = 
		typeSupport.getFieldType(typeInfo, fieldName);
		if (fieldTypeInfo == null) {
		// field is not known
		ErrorMsg.error(i_AST.getLine(), i_AST.getColumn(),
		I18NHelper.getMessage(msgs, "EXC_UnknownField", fieldName, //NOI18N
		typeSupport.getAbstractSchemaForTypeInfo(typeInfo)));
		fieldTypeInfo = typeSupport.errorType;
		}
		else {
		Object fieldInfo = typeSupport.getFieldInfo(typeInfo, fieldName);
		if (fieldInfo == null) {
		ErrorMsg.fatal(I18NHelper.getMessage(msgs, 
		"ERR_MissingFieldInfo",  //NOI18N
		fieldName, typeSupport.getTypeName(typeInfo)));
		}
		if (!typeSupport.isRelationship(fieldInfo)) {
		// field is not a relationship => cmp field
		i_AST.setType(CMP_FIELD);
		dot_AST.setType(CMP_FIELD_ACCESS);
		}
		else if (typeSupport.isCollectionType(fieldTypeInfo)) {
		// field is a relationship of a collection type =>
		// collection valued cmr field
		i_AST.setType(COLLECTION_CMR_FIELD);
		dot_AST.setType(COLLECTION_CMR_FIELD_ACCESS);
		}
		else {
		// field is a relationship of a non collection type =>
		// single valued cmr field
		i_AST.setType(SINGLE_CMR_FIELD);
		dot_AST.setType(SINGLE_CMR_FIELD_ACCESS);
		}
		}
		dot_AST.setTypeInfo(fieldTypeInfo);
		i_AST.setTypeInfo(fieldTypeInfo);
		
		pathExpression_AST = (EJBQLAST)currentAST.root;
		returnAST = pathExpression_AST;
		_retTree = _t;
	}
	
	public final void objectDenoter(AST _t) throws RecognitionException {
		
		EJBQLAST objectDenoter_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST objectDenoter_AST = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case IDENT:
		{
			identificationVariable(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			objectDenoter_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case DOT:
		{
			singleValuedCmrPathExpression(_t);
			_t = _retTree;
			astFactory.addASTChild(currentAST, returnAST);
			objectDenoter_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = objectDenoter_AST;
		_retTree = _t;
	}
	
	public final void inCollectionElement(AST _t,
		Object valueExprTypeInfo
	) throws RecognitionException {
		
		EJBQLAST inCollectionElement_AST_in = (_t == ASTNULL) ? null : (EJBQLAST)_t;
		returnAST = null;
		ASTPair currentAST = new ASTPair();
		EJBQLAST inCollectionElement_AST = null;
		EJBQLAST l_AST = null;
		EJBQLAST l = null;
		EJBQLAST i_AST = null;
		EJBQLAST i = null;
		
		if (_t==null) _t=ASTNULL;
		switch ( _t.getType()) {
		case TRUE:
		case FALSE:
		case STRING_LITERAL:
		case INT_LITERAL:
		case LONG_LITERAL:
		case FLOAT_LITERAL:
		case DOUBLE_LITERAL:
		{
			l = _t==ASTNULL ? null : (EJBQLAST)_t;
			literal(_t);
			_t = _retTree;
			l_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			l.setTypeInfo(analyseInCollectionElement(l_AST, valueExprTypeInfo));
			
			inCollectionElement_AST = (EJBQLAST)currentAST.root;
			break;
		}
		case INPUT_PARAMETER:
		{
			i = _t==ASTNULL ? null : (EJBQLAST)_t;
			inputParameter(_t);
			_t = _retTree;
			i_AST = (EJBQLAST)returnAST;
			astFactory.addASTChild(currentAST, returnAST);
			
			i.setTypeInfo(analyseInCollectionElement(i_AST, valueExprTypeInfo));
			
			inCollectionElement_AST = (EJBQLAST)currentAST.root;
			break;
		}
		default:
		{
			throw new NoViableAltException(_t);
		}
		}
		returnAST = inCollectionElement_AST;
		_retTree = _t;
	}
	
	
	public static final String[] _tokenNames = {
		"<0>",
		"EOF",
		"<2>",
		"NULL_TREE_LOOKAHEAD",
		"\"select\"",
		"\"from\"",
		"\"where\"",
		"\"distinct\"",
		"\"object\"",
		"\"null\"",
		"\"true\"",
		"\"false\"",
		"\"not\"",
		"\"and\"",
		"\"or\"",
		"\"between\"",
		"\"like\"",
		"\"in\"",
		"\"as\"",
		"\"unknown\"",
		"\"empty\"",
		"\"member\"",
		"\"of\"",
		"\"is\"",
		"\"escape\"",
		"\"concat\"",
		"\"substring\"",
		"\"locate\"",
		"\"length\"",
		"\"abs\"",
		"\"sqrt\"",
		"\"mod\"",
		"\"avg\"",
		"\"max\"",
		"\"min\"",
		"\"sum\"",
		"\"count\"",
		"\"order\"",
		"\"by\"",
		"\"asc\"",
		"\"desc\"",
		"EQUAL",
		"NOT_EQUAL",
		"GE",
		"GT",
		"LE",
		"LT",
		"PLUS",
		"MINUS",
		"STAR",
		"DIV",
		"STRING_LITERAL",
		"INT_LITERAL",
		"LONG_LITERAL",
		"FLOAT_LITERAL",
		"DOUBLE_LITERAL",
		"an identifier",
		"DOT",
		"INPUT_PARAMETER",
		"LPAREN",
		"RPAREN",
		"COMMA",
		"WS",
		"HEX_DIGIT",
		"EXPONENT",
		"FLOAT_SUFFIX",
		"UNICODE_DIGIT",
		"UNICODE_STR",
		"NEWLINE",
		"ESC",
		"FLOATINGPOINT_SUFFIX",
		"UNICODE_ESCAPE",
		"QUERY",
		"RANGE",
		"CMP_FIELD_ACCESS",
		"SINGLE_CMR_FIELD_ACCESS",
		"COLLECTION_CMR_FIELD_ACCESS",
		"IDENTIFICATION_VAR",
		"IDENTIFICATION_VAR_DECL",
		"ABSTRACT_SCHEMA_NAME",
		"CMP_FIELD",
		"SINGLE_CMR_FIELD",
		"COLLECTION_CMR_FIELD",
		"UNARY_MINUS",
		"UNARY_PLUS",
		"NOT_BETWEEN",
		"NOT_LIKE",
		"NOT_IN",
		"NOT_NULL",
		"NOT_EMPTY",
		"NOT_MEMBER"
	};
	
	private static final long[] mk_tokenSet_0() {
		long[] data = { 358036170375957504L, 0L};
		return data;
	}
	public static final BitSet _tokenSet_0 = new BitSet(mk_tokenSet_0());
	}
	
