/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.glowroot.config;

import javax.annotation.Nullable;

import org.glowroot.shaded.fasterxml.jackson.annotation.JsonProperty;
import org.glowroot.shaded.fasterxml.jackson.databind.annotation.JsonSerialize;
import org.immutables.value.Value;
import org.glowroot.shaded.slf4j.Logger;
import org.glowroot.shaded.slf4j.LoggerFactory;

@Value.Immutable
@JsonSerialize
public abstract class PropertyDescriptorBase {

    private static final Logger logger = LoggerFactory.getLogger(PropertyDescriptorBase.class);

    public abstract String name();

    public abstract PropertyType type();

    @JsonProperty("default")
    abstract @Nullable PropertyValue defaultValue();

    @Value.Default
    public boolean hidden() {
        return false;
    }

    public abstract String label();

    @Value.Default
    public String checkboxLabel() {
        return "";
    }

    @Value.Default
    public String description() {
        return "";
    }

    PropertyValue getValidatedNonNullDefaultValue() {
        PropertyValue defaultValue = defaultValue();
        if (defaultValue == null) {
            return PropertyValue.getDefaultValue(type());
        }
        Object value = defaultValue.value();
        if (value == null) {
            // this actually shouldn't occur since immutables unmarshals null defaultValue as null
            // as opposed to new PropertyValue(null)
            return PropertyValue.getDefaultValue(type());
        }
        if (isValidType(value, type())) {
            return new PropertyValue(value);
        } else {
            logger.warn("invalid default value for plugin property: {}", name());
            return PropertyValue.getDefaultValue(type());
        }
    }

    static boolean isValidType(Object value, PropertyType type) {
        switch (type) {
            case BOOLEAN:
                return value instanceof Boolean;
            case STRING:
                return value instanceof String;
            case DOUBLE:
                return value instanceof Double;
            default:
                throw new AssertionError("Unexpected property type: " + type);
        }
    }
}
