/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.glowroot.weaving;

import java.util.List;

import javax.annotation.Nullable;

import org.glowroot.shaded.google.common.annotations.VisibleForTesting;
import org.glowroot.shaded.google.common.collect.Lists;
import org.glowroot.shaded.slf4j.Logger;
import org.glowroot.shaded.slf4j.LoggerFactory;

// Class loading is also a bad idea inside of JVM shutdown hooks, see
// https://bugs.openjdk.java.net/browse/JDK-7142035
public class PreInitializeStorageShutdownClasses {

    private static final Logger logger =
            LoggerFactory.getLogger(PreInitializeStorageShutdownClasses.class);

    private PreInitializeStorageShutdownClasses() {}

    public static void preInitializeClasses() {
        ClassLoader loader = PreInitializeStorageShutdownClasses.class.getClassLoader();
        for (String type : usedTypes()) {
            initialize(type, loader);
        }
        for (String type : maybeUsedTypes()) {
            if (exists(type)) {
                initialize(type, loader);
            }
        }
    }

    private static void initialize(String type, @Nullable ClassLoader loader) {
        if (type.equals("org.glowroot.shaded.h2.value.ValueGeometry")) {
            // this type depends on an optional third party library and is not really used
            // (initializing the class throws an error due to the dependency)
            return;
        }
        try {
            Class.forName(type, true, loader);
        } catch (ClassNotFoundException e) {
            logger.warn("class not found: {}", type);
            // log exception at trace level
            logger.trace(e.getMessage(), e);
        }
    }

    @VisibleForTesting
    static List<String> usedTypes() {
        List<String> types = Lists.newArrayList();
        types.addAll(getGuavaUsedTypes());
        types.addAll(getGlowrootUsedTypes());
        types.addAll(getH2UsedTypes());
        return types;

    }

    private static List<String> getGuavaUsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.shaded.google.common.base.Ascii");
        types.add("org.glowroot.shaded.google.common.base.Equivalence");
        types.add("org.glowroot.shaded.google.common.base.Equivalence$Equals");
        types.add("org.glowroot.shaded.google.common.base.Equivalence$Identity");
        types.add("org.glowroot.shaded.google.common.base.Function");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$1");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$ToStringHelper");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$ToStringHelper$ValueHolder");
        types.add("org.glowroot.shaded.google.common.base.Objects");
        types.add("org.glowroot.shaded.google.common.base.Platform");
        types.add("org.glowroot.shaded.google.common.base.Preconditions");
        types.add("org.glowroot.shaded.google.common.base.Stopwatch");
        types.add("org.glowroot.shaded.google.common.base.Stopwatch$1");
        types.add("org.glowroot.shaded.google.common.base.Supplier");
        types.add("org.glowroot.shaded.google.common.base.Suppliers");
        types.add("org.glowroot.shaded.google.common.base.Suppliers$SupplierOfInstance");
        types.add("org.glowroot.shaded.google.common.base.Ticker");
        types.add("org.glowroot.shaded.google.common.base.Ticker$1");
        types.add("org.glowroot.shaded.google.common.cache.AbstractCache$SimpleStatsCounter");
        types.add("org.glowroot.shaded.google.common.cache.AbstractCache$StatsCounter");
        types.add("org.glowroot.shaded.google.common.cache.Cache");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$1");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$2");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$3");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$NullListener");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$OneWeigher");
        types.add("org.glowroot.shaded.google.common.cache.CacheLoader");
        types.add("org.glowroot.shaded.google.common.cache.CacheLoader$InvalidCacheLoadException");
        types.add("org.glowroot.shaded.google.common.cache.CacheStats");
        types.add("org.glowroot.shaded.google.common.cache.LoadingCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AbstractCacheSet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AbstractReferenceEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$3");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$4");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$5");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$6");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$7");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$8");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntrySet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$HashIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$KeyIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$KeySet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LoadingValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LoadingValueReference$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LocalLoadingCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LocalManualCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$NullEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ReferenceEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Segment");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Segment$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$SoftValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$3");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongAccessEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongAccessWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ValueIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Values");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakAccessEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakAccessWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedSoftValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedStrongValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedWeakValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteThroughEntry");
        types.add("org.glowroot.shaded.google.common.cache.LongAddable");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$1");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$2");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$PureJavaLongAddable");
        types.add("org.glowroot.shaded.google.common.cache.LongAdder");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$1");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$2");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$3");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$4");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$5");
        types.add("org.glowroot.shaded.google.common.cache.RemovalListener");
        types.add("org.glowroot.shaded.google.common.cache.RemovalNotification");
        types.add("org.glowroot.shaded.google.common.cache.Striped64");
        types.add("org.glowroot.shaded.google.common.cache.Striped64$1");
        types.add("org.glowroot.shaded.google.common.cache.Striped64$Cell");
        types.add("org.glowroot.shaded.google.common.cache.Weigher");
        types.add("org.glowroot.shaded.google.common.collect.AbstractSequentialIterator");
        types.add("org.glowroot.shaded.google.common.collect.ByFunctionOrdering");
        types.add("org.glowroot.shaded.google.common.collect.CollectPreconditions");
        types.add("org.glowroot.shaded.google.common.collect.EmptyImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableCollection");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.Iterators");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$1");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$2");
        types.add("org.glowroot.shaded.google.common.collect.NaturalOrdering");
        types.add("org.glowroot.shaded.google.common.collect.ObjectArrays");
        types.add("org.glowroot.shaded.google.common.collect.Ordering");
        types.add("org.glowroot.shaded.google.common.collect.Platform");
        types.add("org.glowroot.shaded.google.common.collect.ReverseNaturalOrdering");
        types.add("org.glowroot.shaded.google.common.collect.ReverseOrdering");
        types.add("org.glowroot.shaded.google.common.collect.Sets");
        types.add("org.glowroot.shaded.google.common.collect.UnmodifiableIterator");
        types.add("org.glowroot.shaded.google.common.collect.UnmodifiableListIterator");
        types.add("org.glowroot.shaded.google.common.primitives.Ints");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AbstractFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AbstractFuture$Sync");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AsyncFunction");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionError");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionList");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionList$RunnableExecutorPair");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$1$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$2");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$4");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$7");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ChainingListenableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ChainingListenableFuture$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateFailedFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateSuccessfulFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ListenableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.MoreExecutors");
        types.add("org.glowroot.shaded.google.common.util.concurrent.MoreExecutors$DirectExecutor");
        types.add("org.glowroot.shaded.google.common.util.concurrent.SettableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.UncheckedExecutionException");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Uninterruptibles");
        return types;
    }

    private static List<String> getGlowrootUsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.local.store.CappedDatabase");
        types.add("org.glowroot.local.store.CappedDatabase$ShutdownHookThread");
        types.add("org.glowroot.local.store.CappedDatabaseOutputStream");
        types.add("org.glowroot.local.store.DataSource");
        types.add("org.glowroot.local.store.DataSource$1");
        types.add("org.glowroot.local.store.DataSource$ShutdownHookThread");
        return types;
    }

    private static List<String> getH2UsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.shaded.h2.api.DatabaseEventListener");
        types.add("org.glowroot.shaded.h2.api.ErrorCode");
        types.add("org.glowroot.shaded.h2.api.JavaObjectSerializer");
        types.add("org.glowroot.shaded.h2.command.CommandInterface");
        types.add("org.glowroot.shaded.h2.command.CommandRemote");
        types.add("org.glowroot.shaded.h2.command.dml.SetTypes");
        types.add("org.glowroot.shaded.h2.compress.CompressDeflate");
        types.add("org.glowroot.shaded.h2.compress.CompressLZF");
        types.add("org.glowroot.shaded.h2.compress.CompressNo");
        types.add("org.glowroot.shaded.h2.compress.Compressor");
        types.add("org.glowroot.shaded.h2.engine.ConnectionInfo");
        types.add("org.glowroot.shaded.h2.engine.Constants");
        types.add("org.glowroot.shaded.h2.engine.DbSettings");
        types.add("org.glowroot.shaded.h2.engine.SessionFactory");
        types.add("org.glowroot.shaded.h2.engine.SessionInterface");
        types.add("org.glowroot.shaded.h2.engine.SessionRemote");
        types.add("org.glowroot.shaded.h2.engine.SessionWithState");
        types.add("org.glowroot.shaded.h2.engine.SettingsBase");
        types.add("org.glowroot.shaded.h2.engine.SysProperties");
        types.add("org.glowroot.shaded.h2.expression.ParameterInterface");
        types.add("org.glowroot.shaded.h2.expression.ParameterRemote");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcArray");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcBatchUpdateException");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcBlob");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcBlob$1");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcBlob$2");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcCallableStatement");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcClob");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcClob$1");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcClob$2");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcConnection");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcDatabaseMetaData");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcParameterMetaData");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcPreparedStatement");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcResultSet");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcResultSetMetaData");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcSQLException");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcSavepoint");
        types.add("org.glowroot.shaded.h2.jdbc.JdbcStatement");
        types.add("org.glowroot.shaded.h2.message.DbException");
        types.add("org.glowroot.shaded.h2.message.Trace");
        types.add("org.glowroot.shaded.h2.message.TraceObject");
        types.add("org.glowroot.shaded.h2.message.TraceSystem");
        types.add("org.glowroot.shaded.h2.message.TraceWriter");
        types.add("org.glowroot.shaded.h2.mvstore.DataUtils");
        types.add("org.glowroot.shaded.h2.result.ResultColumn");
        types.add("org.glowroot.shaded.h2.result.ResultInterface");
        types.add("org.glowroot.shaded.h2.result.ResultRemote");
        types.add("org.glowroot.shaded.h2.result.UpdatableRow");
        types.add("org.glowroot.shaded.h2.security.AES");
        types.add("org.glowroot.shaded.h2.security.BlockCipher");
        types.add("org.glowroot.shaded.h2.security.CipherFactory");
        types.add("org.glowroot.shaded.h2.security.Fog");
        types.add("org.glowroot.shaded.h2.security.SHA256");
        types.add("org.glowroot.shaded.h2.security.SecureFileStore");
        types.add("org.glowroot.shaded.h2.security.XTEA");
        types.add("org.glowroot.shaded.h2.store.Data");
        types.add("org.glowroot.shaded.h2.store.DataHandler");
        types.add("org.glowroot.shaded.h2.store.DataReader");
        types.add("org.glowroot.shaded.h2.store.DataReader$FastEOFException");
        types.add("org.glowroot.shaded.h2.store.FileStore");
        types.add("org.glowroot.shaded.h2.store.FileStoreInputStream");
        types.add("org.glowroot.shaded.h2.store.FileStoreOutputStream");
        types.add("org.glowroot.shaded.h2.store.LobStorageFrontend");
        types.add("org.glowroot.shaded.h2.store.LobStorageInterface");
        types.add("org.glowroot.shaded.h2.store.LobStorageRemoteInputStream");
        types.add("org.glowroot.shaded.h2.store.fs.FileBase");
        types.add("org.glowroot.shaded.h2.store.fs.FileChannelInputStream");
        types.add("org.glowroot.shaded.h2.store.fs.FileChannelOutputStream");
        types.add("org.glowroot.shaded.h2.store.fs.FilePath");
        types.add("org.glowroot.shaded.h2.store.fs.FilePathEncrypt");
        types.add("org.glowroot.shaded.h2.store.fs.FilePathEncrypt$FileEncrypt");
        types.add("org.glowroot.shaded.h2.store.fs.FilePathEncrypt$XTS");
        types.add("org.glowroot.shaded.h2.store.fs.FilePathRec");
        types.add("org.glowroot.shaded.h2.store.fs.FilePathWrapper");
        types.add("org.glowroot.shaded.h2.store.fs.FileRec");
        types.add("org.glowroot.shaded.h2.store.fs.FileUtils");
        types.add("org.glowroot.shaded.h2.store.fs.Recorder");
        types.add("org.glowroot.shaded.h2.tools.CompressTool");
        types.add("org.glowroot.shaded.h2.tools.SimpleResultSet");
        types.add("org.glowroot.shaded.h2.tools.SimpleResultSet$Column");
        types.add("org.glowroot.shaded.h2.tools.SimpleResultSet$SimpleArray");
        types.add("org.glowroot.shaded.h2.tools.SimpleRowSource");
        types.add("org.glowroot.shaded.h2.util.BitField");
        types.add("org.glowroot.shaded.h2.util.CloseWatcher");
        types.add("org.glowroot.shaded.h2.util.DateTimeUtils");
        types.add("org.glowroot.shaded.h2.util.IOUtils");
        types.add("org.glowroot.shaded.h2.util.MathUtils");
        types.add("org.glowroot.shaded.h2.util.MathUtils$1");
        types.add("org.glowroot.shaded.h2.util.NetUtils");
        types.add("org.glowroot.shaded.h2.util.New");
        types.add("org.glowroot.shaded.h2.util.SmallLRUCache");
        types.add("org.glowroot.shaded.h2.util.SortedProperties");
        types.add("org.glowroot.shaded.h2.util.StatementBuilder");
        types.add("org.glowroot.shaded.h2.util.StringUtils");
        types.add("org.glowroot.shaded.h2.util.Task");
        types.add("org.glowroot.shaded.h2.util.TempFileDeleter");
        types.add("org.glowroot.shaded.h2.util.Utils");
        types.add("org.glowroot.shaded.h2.util.Utils$1");
        types.add("org.glowroot.shaded.h2.util.Utils$ClassFactory");
        types.add("org.glowroot.shaded.h2.value.CompareMode");
        types.add("org.glowroot.shaded.h2.value.CompareModeDefault");
        types.add("org.glowroot.shaded.h2.value.CompareModeIcu4J");
        types.add("org.glowroot.shaded.h2.value.DataType");
        types.add("org.glowroot.shaded.h2.value.Transfer");
        types.add("org.glowroot.shaded.h2.value.Value");
        types.add("org.glowroot.shaded.h2.value.Value$ValueBlob");
        types.add("org.glowroot.shaded.h2.value.Value$ValueClob");
        types.add("org.glowroot.shaded.h2.value.ValueArray");
        types.add("org.glowroot.shaded.h2.value.ValueBoolean");
        types.add("org.glowroot.shaded.h2.value.ValueByte");
        types.add("org.glowroot.shaded.h2.value.ValueBytes");
        types.add("org.glowroot.shaded.h2.value.ValueDate");
        types.add("org.glowroot.shaded.h2.value.ValueDecimal");
        types.add("org.glowroot.shaded.h2.value.ValueDouble");
        types.add("org.glowroot.shaded.h2.value.ValueFloat");
        types.add("org.glowroot.shaded.h2.value.ValueGeometry");
        types.add("org.glowroot.shaded.h2.value.ValueInt");
        types.add("org.glowroot.shaded.h2.value.ValueJavaObject");
        types.add("org.glowroot.shaded.h2.value.ValueJavaObject$NotSerialized");
        types.add("org.glowroot.shaded.h2.value.ValueLobDb");
        types.add("org.glowroot.shaded.h2.value.ValueLong");
        types.add("org.glowroot.shaded.h2.value.ValueNull");
        types.add("org.glowroot.shaded.h2.value.ValueResultSet");
        types.add("org.glowroot.shaded.h2.value.ValueShort");
        types.add("org.glowroot.shaded.h2.value.ValueString");
        types.add("org.glowroot.shaded.h2.value.ValueStringFixed");
        types.add("org.glowroot.shaded.h2.value.ValueStringIgnoreCase");
        types.add("org.glowroot.shaded.h2.value.ValueTime");
        types.add("org.glowroot.shaded.h2.value.ValueTimestamp");
        types.add("org.glowroot.shaded.h2.value.ValueUuid");
        return types;
    }

    @VisibleForTesting
    static List<String> maybeUsedTypes() {
        List<String> types = Lists.newArrayList();
        // these are special classes generated by javac (but not by the eclipse compiler) to handle
        // accessing the private constructor in an enclosed type
        // (see http://stackoverflow.com/questions/2883181)
        types.add("org.glowroot.common.Reflections$1");
        types.add("org.glowroot.trace.model.TraceGcInfo$1");
        types.add("org.glowroot.weaving.AnalyzedClass$1");
        types.add("org.glowroot.weaving.Weaver$1");
        types.add("org.glowroot.weaving.WeavingMethodVisitor$1");
        // this is a special class generated by javac (but not by the eclipse compiler) to handle
        // enum switch statements
        // (see http://stackoverflow.com/questions/1834632/java-enum-and-additional-class-files)
        types.add("org.glowroot.weaving.AdviceMatcher$1");
        return types;
    }

    private static boolean exists(String type) {
        try {
            Class.forName(type);
            return true;
        } catch (ClassNotFoundException e) {
            // log exception at trace level
            logger.trace(e.getMessage(), e);
            return false;
        }
    }
}
