/*
 * Copyright 2014 The Netty Project
 *
 * The Netty Project licenses this file to you under the Apache License,
 * version 2.0 (the "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at:
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations
 * under the License.
 */
package org.glowroot.shaded.netty.example.spdy.server;

import org.glowroot.shaded.netty.bootstrap.ServerBootstrap;
import org.glowroot.shaded.netty.channel.Channel;
import org.glowroot.shaded.netty.channel.ChannelOption;
import org.glowroot.shaded.netty.channel.EventLoopGroup;
import org.glowroot.shaded.netty.channel.nio.NioEventLoopGroup;
import org.glowroot.shaded.netty.channel.socket.nio.NioServerSocketChannel;
import org.glowroot.shaded.netty.handler.logging.LogLevel;
import org.glowroot.shaded.netty.handler.logging.LoggingHandler;
import org.glowroot.shaded.netty.handler.ssl.ApplicationProtocolConfig;
import org.glowroot.shaded.netty.handler.ssl.ApplicationProtocolConfig.Protocol;
import org.glowroot.shaded.netty.handler.ssl.ApplicationProtocolConfig.SelectedListenerFailureBehavior;
import org.glowroot.shaded.netty.handler.ssl.ApplicationProtocolConfig.SelectorFailureBehavior;
import org.glowroot.shaded.netty.handler.ssl.ApplicationProtocolNames;
import org.glowroot.shaded.netty.handler.ssl.SslContext;
import org.glowroot.shaded.netty.handler.ssl.SslContextBuilder;
import org.glowroot.shaded.netty.handler.ssl.util.SelfSignedCertificate;

/**
 * A SPDY Server that responds to a GET request with a Hello World.
 * <p>
 * This class must be run with the JVM parameter: {@code java -Xbootclasspath/p:<path_to_npn_boot_jar> ...}.
 * The "path_to_npn_boot_jar" is the path on the file system for the NPN Boot Jar file which can be downloaded from
 * Maven at coordinates org.mortbay.jetty.npn:npn-boot. Different versions applies to different OpenJDK versions.
 * See <a href="http://www.eclipse.org/jetty/documentation/current/npn-chapter.html">Jetty docs</a> for more
 * information.
 * <p>
 * You may also use the {@code run-example.sh} script to start the server from the command line:
 * <pre>
 *     ./run-example.sh spdy-server
 * </pre>
 * <p>
 * Once started, you can test the server with your
 * <a href="http://en.wikipedia.org/wiki/SPDY#Browser_support_and_usage">SPDY enabled web browser</a> by navigating
 * to <a href="https://localhost:8443/">https://localhost:8443/</a>
 */
public final class SpdyServer {

    static final int PORT = Integer.parseInt(System.getProperty("port", "8443"));

    public static void main(String[] args) throws Exception {
        // Configure SSL.
        SelfSignedCertificate ssc = new SelfSignedCertificate();
        SslContext sslCtx = SslContextBuilder.forServer(ssc.certificate(), ssc.privateKey())
            .applicationProtocolConfig(new ApplicationProtocolConfig(
                        Protocol.NPN,
                        // NO_ADVERTISE is currently the only mode supported by both OpenSsl and JDK providers.
                        SelectorFailureBehavior.NO_ADVERTISE,
                        // ACCEPT is currently the only mode supported by both OpenSsl and JDK providers.
                        SelectedListenerFailureBehavior.ACCEPT,
                        ApplicationProtocolNames.SPDY_3_1,
                        ApplicationProtocolNames.HTTP_1_1))
            .build();

        // Configure the server.
        EventLoopGroup bossGroup = new NioEventLoopGroup(1);
        EventLoopGroup workerGroup = new NioEventLoopGroup();
        try {
            ServerBootstrap b = new ServerBootstrap();
            b.option(ChannelOption.SO_BACKLOG, 1024);
            b.group(bossGroup, workerGroup)
             .channel(NioServerSocketChannel.class)
             .handler(new LoggingHandler(LogLevel.INFO))
             .childHandler(new SpdyServerInitializer(sslCtx));

            Channel ch = b.bind(PORT).sync().channel();

            System.err.println("Open your SPDY-enabled web browser and navigate to https://127.0.0.1:" + PORT + '/');
            System.err.println("If using Chrome browser, check your SPDY sessions at chrome://net-internals/#spdy");

            ch.closeFuture().sync();
        } finally {
            bossGroup.shutdownGracefully();
            workerGroup.shutdownGracefully();
        }
    }
}
