/*
 * Copyright 2012-2015 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.glowroot.weaving;

import java.util.List;

import javax.annotation.Nullable;

import org.glowroot.shaded.google.common.annotations.VisibleForTesting;
import org.glowroot.shaded.google.common.collect.Lists;
import org.glowroot.shaded.slf4j.Logger;
import org.glowroot.shaded.slf4j.LoggerFactory;

// "There are some things that agents are allowed to do that simply should not be permitted"
//
// -- http://mail.openjdk.java.net/pipermail/hotspot-dev/2012-March/005464.html
//
// in particular (at least prior to parallel class loading in JDK 7) initializing other classes
// inside of a ClassFileTransformer.transform() method occasionally leads to deadlocks
//
// to avoid initializing other classes inside of the transform() method, all classes referenced from
// WeavingClassFileTransformer are pre-initialized (and all classes referenced from those classes,
// etc)
public class PreInitializeWeavingClasses {

    private static final Logger logger = LoggerFactory.getLogger(PreInitializeWeavingClasses.class);

    private PreInitializeWeavingClasses() {}

    public static void preInitializeClasses() {
        ClassLoader loader = PreInitializeWeavingClasses.class.getClassLoader();
        for (String type : usedTypes()) {
            initialize(type, loader, true);
        }
        for (String type : maybeUsedTypes()) {
            initialize(type, loader, false);
        }
        for (String type : javaUsedTypes()) {
            // passing warnOnNotExists=false since ThreadLocalRandom only exists in jdk 1.7+
            initialize(type, loader, false);
        }
    }

    private static void initialize(String type, @Nullable ClassLoader loader,
            boolean warnOnNotExists) {
        try {
            Class.forName(type, true, loader);
        } catch (ClassNotFoundException e) {
            if (warnOnNotExists) {
                logger.warn("class not found: {}", type);
            }
            // log exception at trace level
            logger.trace(e.getMessage(), e);
        }
    }

    @VisibleForTesting
    static List<String> usedTypes() {
        List<String> types = Lists.newArrayList();
        types.addAll(getGuavaUsedTypes());
        types.addAll(getGlowrootUsedTypes());
        types.addAll(getAsmUsedTypes());
        return types;
    }

    private static List<String> getGuavaUsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.shaded.google.common.base.Ascii");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$1");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$10");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$13");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$15");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$2");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$3");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$4");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$5");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$6");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$7");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$8");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$9");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$FastMatcher");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$NegatedFastMatcher");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$NegatedMatcher");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$Or");
        types.add("org.glowroot.shaded.google.common.base.CharMatcher$RangesMatcher");
        types.add("org.glowroot.shaded.google.common.base.Equivalence");
        types.add("org.glowroot.shaded.google.common.base.Equivalence$Equals");
        types.add("org.glowroot.shaded.google.common.base.Equivalence$Identity");
        types.add("org.glowroot.shaded.google.common.base.Function");
        types.add("org.glowroot.shaded.google.common.base.Joiner");
        types.add("org.glowroot.shaded.google.common.base.Joiner$1");
        types.add("org.glowroot.shaded.google.common.base.Joiner$MapJoiner");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$1");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$ToStringHelper");
        types.add("org.glowroot.shaded.google.common.base.MoreObjects$ToStringHelper$ValueHolder");
        types.add("org.glowroot.shaded.google.common.base.Objects");
        types.add("org.glowroot.shaded.google.common.base.Platform");
        types.add("org.glowroot.shaded.google.common.base.Preconditions");
        types.add("org.glowroot.shaded.google.common.base.Predicate");
        types.add("org.glowroot.shaded.google.common.base.Stopwatch");
        types.add("org.glowroot.shaded.google.common.base.Stopwatch$1");
        types.add("org.glowroot.shaded.google.common.base.Supplier");
        types.add("org.glowroot.shaded.google.common.base.Suppliers");
        types.add("org.glowroot.shaded.google.common.base.Suppliers$SupplierOfInstance");
        types.add("org.glowroot.shaded.google.common.base.Throwables");
        types.add("org.glowroot.shaded.google.common.base.Ticker");
        types.add("org.glowroot.shaded.google.common.base.Ticker$1");
        types.add("org.glowroot.shaded.google.common.cache.AbstractCache$SimpleStatsCounter");
        types.add("org.glowroot.shaded.google.common.cache.AbstractCache$StatsCounter");
        types.add("org.glowroot.shaded.google.common.cache.Cache");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$1");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$2");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$3");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$NullListener");
        types.add("org.glowroot.shaded.google.common.cache.CacheBuilder$OneWeigher");
        types.add("org.glowroot.shaded.google.common.cache.CacheLoader");
        types.add("org.glowroot.shaded.google.common.cache.CacheLoader$InvalidCacheLoadException");
        types.add("org.glowroot.shaded.google.common.cache.CacheStats");
        types.add("org.glowroot.shaded.google.common.cache.LoadingCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AbstractCacheSet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AbstractReferenceEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$AccessQueue$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$3");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$4");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$5");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$6");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$7");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryFactory$8");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntryIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$EntrySet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$HashIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$KeyIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$KeySet");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LoadingValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LoadingValueReference$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LocalLoadingCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$LocalManualCache");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$NullEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ReferenceEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Segment");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Segment$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$SoftValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Strength$3");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongAccessEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongAccessWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$StrongWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ValueIterator");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$ValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$Values");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakAccessEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakAccessWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeakWriteEntry");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedSoftValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedStrongValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WeightedWeakValueReference");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue$1");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteQueue$2");
        types.add("org.glowroot.shaded.google.common.cache.LocalCache$WriteThroughEntry");
        types.add("org.glowroot.shaded.google.common.cache.LongAddable");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$1");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$2");
        types.add("org.glowroot.shaded.google.common.cache.LongAddables$PureJavaLongAddable");
        types.add("org.glowroot.shaded.google.common.cache.LongAdder");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$1");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$2");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$3");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$4");
        types.add("org.glowroot.shaded.google.common.cache.RemovalCause$5");
        types.add("org.glowroot.shaded.google.common.cache.RemovalListener");
        types.add("org.glowroot.shaded.google.common.cache.RemovalNotification");
        types.add("org.glowroot.shaded.google.common.cache.Striped64");
        types.add("org.glowroot.shaded.google.common.cache.Striped64$1");
        types.add("org.glowroot.shaded.google.common.cache.Striped64$Cell");
        types.add("org.glowroot.shaded.google.common.cache.Weigher");
        types.add("org.glowroot.shaded.google.common.collect.AbstractIndexedListIterator");
        types.add("org.glowroot.shaded.google.common.collect.AbstractSequentialIterator");
        types.add("org.glowroot.shaded.google.common.collect.ByFunctionOrdering");
        types.add("org.glowroot.shaded.google.common.collect.CollectPreconditions");
        types.add("org.glowroot.shaded.google.common.collect.Collections2");
        types.add("org.glowroot.shaded.google.common.collect.EmptyImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.Hashing");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableAsList");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableCollection");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableCollection$ArrayBasedBuilder");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableCollection$Builder");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableList");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableList$1");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableList$Builder");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableList$ReverseImmutableList");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableList$SubList");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.ImmutableSet$Builder");
        types.add("org.glowroot.shaded.google.common.collect.Iterables");
        types.add("org.glowroot.shaded.google.common.collect.Iterators");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$1");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$11");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$12");
        types.add("org.glowroot.shaded.google.common.collect.Iterators$2");
        types.add("org.glowroot.shaded.google.common.collect.Lists");
        types.add("org.glowroot.shaded.google.common.collect.Lists$RandomAccessReverseList");
        types.add("org.glowroot.shaded.google.common.collect.Lists$ReverseList");
        types.add("org.glowroot.shaded.google.common.collect.Lists$ReverseList$1");
        types.add("org.glowroot.shaded.google.common.collect.Maps");
        types.add("org.glowroot.shaded.google.common.collect.NaturalOrdering");
        types.add("org.glowroot.shaded.google.common.collect.ObjectArrays");
        types.add("org.glowroot.shaded.google.common.collect.Ordering");
        types.add("org.glowroot.shaded.google.common.collect.Platform");
        types.add("org.glowroot.shaded.google.common.collect.RegularImmutableAsList");
        types.add("org.glowroot.shaded.google.common.collect.RegularImmutableList");
        types.add("org.glowroot.shaded.google.common.collect.RegularImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.ReverseNaturalOrdering");
        types.add("org.glowroot.shaded.google.common.collect.ReverseOrdering");
        types.add("org.glowroot.shaded.google.common.collect.Sets");
        types.add("org.glowroot.shaded.google.common.collect.SingletonImmutableList");
        types.add("org.glowroot.shaded.google.common.collect.SingletonImmutableSet");
        types.add("org.glowroot.shaded.google.common.collect.UnmodifiableIterator");
        types.add("org.glowroot.shaded.google.common.collect.UnmodifiableListIterator");
        types.add("org.glowroot.shaded.google.common.io.ByteSource");
        types.add("org.glowroot.shaded.google.common.io.ByteStreams");
        types.add("org.glowroot.shaded.google.common.io.ByteStreams$1");
        types.add("org.glowroot.shaded.google.common.io.Closeables");
        types.add("org.glowroot.shaded.google.common.io.Closer");
        types.add("org.glowroot.shaded.google.common.io.Closer$LoggingSuppressor");
        types.add("org.glowroot.shaded.google.common.io.Closer$SuppressingSuppressor");
        types.add("org.glowroot.shaded.google.common.io.Closer$Suppressor");
        types.add("org.glowroot.shaded.google.common.io.LineProcessor");
        types.add("org.glowroot.shaded.google.common.io.Resources");
        types.add("org.glowroot.shaded.google.common.io.Resources$1");
        types.add("org.glowroot.shaded.google.common.io.Resources$UrlByteSource");
        types.add("org.glowroot.shaded.google.common.primitives.Booleans");
        types.add("org.glowroot.shaded.google.common.primitives.Ints");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AbstractFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AbstractFuture$Sync");
        types.add("org.glowroot.shaded.google.common.util.concurrent.AsyncFunction");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionError");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionList");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ExecutionList$RunnableExecutorPair");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$1$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$2");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$4");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$7");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ChainingListenableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ChainingListenableFuture$1");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateFailedFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Futures$ImmediateSuccessfulFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.ListenableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.MoreExecutors");
        types.add("org.glowroot.shaded.google.common.util.concurrent.MoreExecutors$DirectExecutor");
        types.add("org.glowroot.shaded.google.common.util.concurrent.SettableFuture");
        types.add("org.glowroot.shaded.google.common.util.concurrent.UncheckedExecutionException");
        types.add("org.glowroot.shaded.google.common.util.concurrent.Uninterruptibles");
        return types;
    }

    private static List<String> getGlowrootUsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.common.ClassNames");
        types.add("org.glowroot.common.Reflections");
        types.add("org.glowroot.common.Tickers");
        types.add("org.glowroot.common.Tickers$DummyTicker");
        types.add("org.glowroot.plugin.api.transaction.Timer");
        types.add("org.glowroot.plugin.api.transaction.TimerName");
        types.add("org.glowroot.plugin.api.util.FastThreadLocal");
        types.add("org.glowroot.plugin.api.util.FastThreadLocal$1");
        types.add("org.glowroot.plugin.api.util.Holder");
        types.add("org.glowroot.plugin.api.weaving.BindParameter");
        types.add("org.glowroot.plugin.api.weaving.BindTraveler");
        types.add("org.glowroot.plugin.api.weaving.IsEnabled");
        types.add("org.glowroot.plugin.api.weaving.MethodModifier");
        types.add("org.glowroot.plugin.api.weaving.OnAfter");
        types.add("org.glowroot.plugin.api.weaving.OnBefore");
        types.add("org.glowroot.plugin.api.weaving.OnReturn");
        types.add("org.glowroot.plugin.api.weaving.OnThrow");
        types.add("org.glowroot.plugin.api.weaving.Pointcut");
        types.add("org.glowroot.plugin.api.weaving.Shim");
        types.add("org.glowroot.transaction.TransactionRegistry");
        types.add("org.glowroot.transaction.WeavingTimerServiceImpl");
        types.add("org.glowroot.transaction.WeavingTimerServiceImpl$2");
        types.add("org.glowroot.transaction.WeavingTimerServiceImpl$NopWeavingTimer");
        types.add("org.glowroot.transaction.model.NestedTimerMap");
        types.add("org.glowroot.transaction.model.NestedTimerMap$Entry");
        types.add("org.glowroot.transaction.model.TimerImpl");
        types.add("org.glowroot.transaction.model.TimerNameImpl");
        types.add("org.glowroot.transaction.model.TimerNameImplBase");
        types.add("org.glowroot.transaction.model.Transaction");
        types.add("org.glowroot.weaving.Advice");
        types.add("org.glowroot.weaving.AdviceBase");
        types.add("org.glowroot.weaving.AdviceBase$AdviceOrdering");
        types.add("org.glowroot.weaving.AdviceBase$AdviceParameterBase");
        types.add("org.glowroot.weaving.AdviceFlowOuterHolder");
        types.add("org.glowroot.weaving.AdviceFlowOuterHolder$1");
        types.add("org.glowroot.weaving.AdviceFlowOuterHolder$AdviceFlowHolder");
        types.add("org.glowroot.weaving.AdviceMatcher");
        types.add("org.glowroot.weaving.AdviceMatcherBase");
        types.add("org.glowroot.weaving.AdviceParameter");
        types.add("org.glowroot.weaving.AnalyzedClass");
        types.add("org.glowroot.weaving.AnalyzedClass$Builder");
        types.add("org.glowroot.weaving.AnalyzedClassBase");
        types.add("org.glowroot.weaving.AnalyzedMethod");
        types.add("org.glowroot.weaving.AnalyzedMethod$Builder");
        types.add("org.glowroot.weaving.AnalyzedMethodBase");
        types.add("org.glowroot.weaving.AnalyzedMethodKey");
        types.add("org.glowroot.weaving.AnalyzedMethodKey$Builder");
        types.add("org.glowroot.weaving.AnalyzedWorld");
        types.add("org.glowroot.weaving.AnalyzedWorld$1");
        types.add("org.glowroot.weaving.AnalyzedWorld$ParseContextBase");
        types.add("org.glowroot.weaving.AnalyzingClassVisitor");
        types.add("org.glowroot.weaving.BootstrapMetaHolders");
        types.add("org.glowroot.weaving.BootstrapMetaHolders$ClassMetaHolder");
        types.add("org.glowroot.weaving.BootstrapMetaHolders$MethodMetaHolder");
        types.add("org.glowroot.weaving.CatchHandler");
        types.add("org.glowroot.weaving.ClassLoaders");
        types.add("org.glowroot.weaving.ExtraBootResourceFinder");
        types.add("org.glowroot.weaving.GeneratedBytecodeUtil");
        types.add("org.glowroot.weaving.MethodMetaGroup");
        types.add("org.glowroot.weaving.MethodMetaGroup$Builder");
        types.add("org.glowroot.weaving.MixinType");
        types.add("org.glowroot.weaving.MixinTypeBase");
        types.add("org.glowroot.weaving.ParameterKind");
        types.add("org.glowroot.weaving.ParseContext");
        types.add("org.glowroot.weaving.PointcutClassVisitor");
        types.add("org.glowroot.weaving.ShimType");
        types.add("org.glowroot.weaving.ShimTypeBase");
        types.add("org.glowroot.weaving.Weaver");
        types.add("org.glowroot.weaving.Weaver$ComputeFramesClassWriter");
        types.add("org.glowroot.weaving.Weaver$JSRInlinerClassVisitor");
        types.add("org.glowroot.weaving.WeavingClassFileTransformer");
        types.add("org.glowroot.weaving.WeavingClassVisitor");
        types.add("org.glowroot.weaving.WeavingClassVisitor$AnalyzedMethodKeyBase");
        types.add("org.glowroot.weaving.WeavingClassVisitor$InitMixins");
        types.add("org.glowroot.weaving.WeavingClassVisitor$MethodMetaGroupBase");
        types.add("org.glowroot.weaving.WeavingClassVisitor$PointcutClassFoundException");
        types.add("org.glowroot.weaving.WeavingClassVisitor$ShortCircuitException");
        types.add("org.glowroot.weaving.WeavingMethodVisitor");
        types.add("org.glowroot.weaving.WeavingMethodVisitor$CatchHandlerBase");
        types.add("org.glowroot.weaving.WeavingTimerService");
        types.add("org.glowroot.weaving.WeavingTimerService$WeavingTimer");
        return types;
    }

    private static List<String> getAsmUsedTypes() {
        List<String> types = Lists.newArrayList();
        types.add("org.glowroot.shaded.objectweb.asm.AnnotationVisitor");
        types.add("org.glowroot.shaded.objectweb.asm.AnnotationWriter");
        types.add("org.glowroot.shaded.objectweb.asm.Attribute");
        types.add("org.glowroot.shaded.objectweb.asm.ByteVector");
        types.add("org.glowroot.shaded.objectweb.asm.ClassReader");
        types.add("org.glowroot.shaded.objectweb.asm.ClassVisitor");
        types.add("org.glowroot.shaded.objectweb.asm.ClassWriter");
        types.add("org.glowroot.shaded.objectweb.asm.Context");
        types.add("org.glowroot.shaded.objectweb.asm.Edge");
        types.add("org.glowroot.shaded.objectweb.asm.FieldVisitor");
        types.add("org.glowroot.shaded.objectweb.asm.FieldWriter");
        types.add("org.glowroot.shaded.objectweb.asm.Frame");
        types.add("org.glowroot.shaded.objectweb.asm.Handle");
        types.add("org.glowroot.shaded.objectweb.asm.Handler");
        types.add("org.glowroot.shaded.objectweb.asm.Item");
        types.add("org.glowroot.shaded.objectweb.asm.Label");
        types.add("org.glowroot.shaded.objectweb.asm.MethodVisitor");
        types.add("org.glowroot.shaded.objectweb.asm.MethodWriter");
        types.add("org.glowroot.shaded.objectweb.asm.Opcodes");
        types.add("org.glowroot.shaded.objectweb.asm.Type");
        types.add("org.glowroot.shaded.objectweb.asm.TypePath");
        types.add("org.glowroot.shaded.objectweb.asm.commons.AdviceAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.GeneratorAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.JSRInlinerAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.JSRInlinerAdapter$Instantiation");
        types.add("org.glowroot.shaded.objectweb.asm.commons.LocalVariablesSorter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.Method");
        types.add("org.glowroot.shaded.objectweb.asm.commons.Remapper");
        types.add("org.glowroot.shaded.objectweb.asm.commons.RemappingAnnotationAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.RemappingMethodAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.RemappingSignatureAdapter");
        types.add("org.glowroot.shaded.objectweb.asm.commons.SimpleRemapper");
        types.add("org.glowroot.shaded.objectweb.asm.signature.SignatureReader");
        types.add("org.glowroot.shaded.objectweb.asm.signature.SignatureVisitor");
        types.add("org.glowroot.shaded.objectweb.asm.signature.SignatureWriter");
        types.add("org.glowroot.shaded.objectweb.asm.tree.AbstractInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.AnnotationNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.ClassNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.FieldInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.FieldNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.FrameNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.IincInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.InnerClassNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.InsnList");
        types.add("org.glowroot.shaded.objectweb.asm.tree.InsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.IntInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.InvokeDynamicInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.JumpInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LabelNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LdcInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LineNumberNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LocalVariableAnnotationNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LocalVariableNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.LookupSwitchInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.MethodInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.MethodNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.MethodNode$1");
        types.add("org.glowroot.shaded.objectweb.asm.tree.MultiANewArrayInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.ParameterNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.TableSwitchInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.TryCatchBlockNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.TypeAnnotationNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.TypeInsnNode");
        types.add("org.glowroot.shaded.objectweb.asm.tree.VarInsnNode");
        return types;
    }

    @VisibleForTesting
    static List<String> maybeUsedTypes() {
        List<String> types = Lists.newArrayList();
        // these are special classes generated by javac (but not by the eclipse compiler) to handle
        // accessing the private constructor in an enclosed type
        // (see http://stackoverflow.com/questions/2883181)
        types.add("org.glowroot.common.Tickers$1");
        types.add("org.glowroot.transaction.model.NestedTimerMap$1");
        types.add("org.glowroot.weaving.AdviceBase$1");
        types.add("org.glowroot.weaving.AnalyzedClass$1");
        types.add("org.glowroot.weaving.AnalyzedMethod$1");
        types.add("org.glowroot.weaving.AnalyzedMethodKey$1");
        types.add("org.glowroot.weaving.BootstrapMetaHolders$1");
        types.add("org.glowroot.weaving.MethodMetaGroup$1");
        types.add("org.glowroot.weaving.Weaver$1");
        types.add("org.glowroot.weaving.WeavingClassVisitor$1");
        // this is a special class generated by javac (but not by the eclipse compiler) to handle
        // enum switch statements
        // (see http://stackoverflow.com/questions/1834632/java-enum-and-additional-class-files)
        types.add("org.glowroot.weaving.AdviceMatcherBase$1");
        types.add("org.glowroot.weaving.WeavingMethodVisitor$1");
        return types;
    }

    // for the most part, adding used java types is not needed and will just slow down startup
    // exceptions can be added here
    private static List<String> javaUsedTypes() {
        List<String> types = Lists.newArrayList();
        // pre-initialize ThreadLocalRandom to avoid this error that occurred once during
        // integration tests (ClassLoaderLeakTest):
        //
        // java.lang.ClassCircularityError: sun/nio/ch/Interruptible
        //
        // java.lang.Class.getDeclaredFields0(Native Method)[na:1.8.0_20]
        // java.lang.Class.privateGetDeclaredFields(Class.java:2570)[na:1.8.0_20]
        // java.lang.Class.getDeclaredField(Class.java:2055)[na:1.8.0_20]
        // java.util.concurrent.ThreadLocalRandom.<clinit>(ThreadLocalRandom.java:1092)~[na:1.8.0_20]
        // java.util.concurrent.ConcurrentHashMap.fullAddCount(ConcurrentHashMap.java:2526)~[na:1.8.0_20]
        // java.util.concurrent.ConcurrentHashMap.addCount(ConcurrentHashMap.java:2266)~[na:1.8.0_20]
        // java.util.concurrent.ConcurrentHashMap.putVal(ConcurrentHashMap.java:1070)~[na:1.8.0_20]
        // java.util.concurrent.ConcurrentHashMap.put(ConcurrentHashMap.java:1006)~[na:1.8.0_20]
        // org.glowroot.weaving.AnalyzedWorld.add(AnalyzedWorld.java:156)~[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.AnalyzingClassVisitor.visitEndReturningAnalyzedClass(AnalyzingClassVisitor.java:160)~[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.WeavingClassVisitor.visitEnd(WeavingClassVisitor.java:229)~[na:0.5-SNAPSHOT]
        // org.glowroot.shaded.objectweb.asm.ClassVisitor.visitEnd(Unknown Source)~[na:0.5-SNAPSHOT]
        // org.glowroot.shaded.objectweb.asm.ClassReader.accept(Unknown Source)~[na:0.5-SNAPSHOT]
        // org.glowroot.shaded.objectweb.asm.ClassReader.accept(Unknown Source)~[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.Weaver.weaveInternal(Weaver.java:115)[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.Weaver.weave$glowroot$timer$glowroot$weaving$0(Weaver.java:88)[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.Weaver.weave(Weaver.java:78)[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.WeavingClassFileTransformer.transformInternal(WeavingClassFileTransformer.java:113)[na:0.5-SNAPSHOT]
        // org.glowroot.weaving.WeavingClassFileTransformer.transform(WeavingClassFileTransformer.java:76)[na:0.5-SNAPSHOT]
        // sun.instrument.TransformerManager.transform(TransformerManager.java:188)[na:1.8.0_20]
        // sun.instrument.InstrumentationImpl.transform(InstrumentationImpl.java:428)[na:1.8.0_20]
        // java.lang.Class.getDeclaredFields0(Native Method)[na:1.8.0_20]
        // java.lang.Class.privateGetDeclaredFields(Class.java:2570)[na:1.8.0_20]
        // java.lang.Class.getDeclaredField(Class.java:2055)[na:1.8.0_20]
        // java.util.concurrent.locks.LockSupport.<clinit>(LockSupport.java:404)[na:1.8.0_20]
        // java.util.concurrent.locks.AbstractQueuedSynchronizer$ConditionObject.await(AbstractQueuedSynchronizer.java:2039)[na:1.8.0_20]
        // java.util.concurrent.ScheduledThreadPoolExecutor$DelayedWorkQueue.take(ScheduledThreadPoolExecutor.java:1088)[na:1.8.0_20]
        // java.util.concurrent.ScheduledThreadPoolExecutor$DelayedWorkQueue.take(ScheduledThreadPoolExecutor.java:809)[na:1.8.0_20]
        // java.util.concurrent.ThreadPoolExecutor.getTask(ThreadPoolExecutor.java:1067)[na:1.8.0_20]
        // java.util.concurrent.ThreadPoolExecutor.runWorker(ThreadPoolExecutor.java:1127)[na:1.8.0_20]
        // java.util.concurrent.ThreadPoolExecutor$Worker.run(ThreadPoolExecutor.java:617)[na:1.8.0_20]
        // java.lang.Thread.run(Thread.java:745)[na:1.8.0_20]
        types.add("java.util.concurrent.ThreadLocalRandom");
        return types;
    }
}
