/*
 * Decompiled with CFR 0.152.
 */
package org.glowroot.local.store;

import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.text.DecimalFormat;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.TimeUnit;
import javax.crypto.SecretKey;
import org.glowroot.collector.Aggregate;
import org.glowroot.collector.LazyHistogram;
import org.glowroot.common.Encryption;
import org.glowroot.config.AlertConfig;
import org.glowroot.config.ConfigService;
import org.glowroot.config.SmtpConfig;
import org.glowroot.local.store.AggregateDao;
import org.glowroot.local.store.MailService;
import org.glowroot.local.store.TriggeredAlertDao;
import org.glowroot.shaded.google.common.annotations.VisibleForTesting;
import org.glowroot.shaded.google.common.collect.ImmutableList;
import org.glowroot.shaded.google.common.collect.ImmutableMap;
import org.glowroot.shaded.javax.mail.Address;
import org.glowroot.shaded.javax.mail.Authenticator;
import org.glowroot.shaded.javax.mail.Message;
import org.glowroot.shaded.javax.mail.PasswordAuthentication;
import org.glowroot.shaded.javax.mail.Session;
import org.glowroot.shaded.javax.mail.internet.InternetAddress;
import org.glowroot.shaded.javax.mail.internet.MimeMessage;
import org.glowroot.shaded.slf4j.Logger;
import org.glowroot.shaded.slf4j.LoggerFactory;

public class AlertingService {
    private static final Logger logger = LoggerFactory.getLogger(AlertingService.class);
    private final ConfigService configService;
    private final TriggeredAlertDao triggeredAlertDao;
    private final AggregateDao aggregateDao;
    private final MailService mailService;

    AlertingService(ConfigService configService, TriggeredAlertDao triggeredAlertDao, AggregateDao aggregateDao, MailService mailService) {
        this.configService = configService;
        this.triggeredAlertDao = triggeredAlertDao;
        this.aggregateDao = aggregateDao;
        this.mailService = mailService;
    }

    void checkAlerts(long endTime) {
        for (AlertConfig alertConfig : this.configService.getAlertConfigs()) {
            try {
                this.checkAlert(alertConfig, endTime);
            }
            catch (Exception e) {
                logger.error(e.getMessage(), e);
            }
        }
    }

    private void checkAlert(AlertConfig alertConfig, long endTime) throws Exception {
        boolean currentlyTriggered;
        long startTime = endTime - TimeUnit.MINUTES.toMillis(alertConfig.timePeriodMinutes());
        int rollupLevel = AlertingService.getRollupLevel(++startTime, endTime);
        ImmutableList<Aggregate> aggregates = this.aggregateDao.readOverallAggregates(alertConfig.transactionType(), startTime, endTime, rollupLevel);
        long transactionCount = 0L;
        LazyHistogram histogram = new LazyHistogram();
        for (Aggregate aggregate : aggregates) {
            transactionCount += aggregate.transactionCount();
            histogram.decodeFromByteBuffer(ByteBuffer.wrap(aggregate.histogram()));
        }
        if (transactionCount < (long)alertConfig.minTransactionCount()) {
            return;
        }
        boolean previouslyTriggered = this.triggeredAlertDao.exists(alertConfig.version());
        long valueAtPercentile = histogram.getValueAtPercentile(alertConfig.percentile());
        boolean bl = currentlyTriggered = valueAtPercentile >= TimeUnit.MILLISECONDS.toMicros(alertConfig.thresholdMillis());
        if (previouslyTriggered && !currentlyTriggered) {
            this.triggeredAlertDao.delete(alertConfig.version());
            this.sendAlert(alertConfig, valueAtPercentile, transactionCount, true);
        } else if (!previouslyTriggered && currentlyTriggered) {
            this.triggeredAlertDao.insert(alertConfig.version(), endTime);
            this.sendAlert(alertConfig, valueAtPercentile, transactionCount, false);
        }
    }

    private void sendAlert(AlertConfig alertConfig, long valueAtPercentile, long transactionCount, boolean ok) throws Exception {
        String fromDisplayName;
        SmtpConfig smtpConfig = this.configService.getSmtpConfig();
        Session session = AlertingService.createMailSession(smtpConfig, this.configService.getSecretKey());
        MimeMessage message = new MimeMessage(session);
        String fromEmailAddress = smtpConfig.fromEmailAddress();
        if (fromEmailAddress.isEmpty()) {
            String localServerName = InetAddress.getLocalHost().getHostName();
            fromEmailAddress = "glowroot@" + localServerName;
        }
        if ((fromDisplayName = smtpConfig.fromDisplayName()).isEmpty()) {
            fromDisplayName = "Glowroot";
        }
        ((Message)message).setFrom(new InternetAddress(fromEmailAddress, fromDisplayName));
        Address[] emailAddresses = new Address[alertConfig.emailAddresses().size()];
        for (int i = 0; i < alertConfig.emailAddresses().size(); ++i) {
            emailAddresses[i] = new InternetAddress((String)alertConfig.emailAddresses().get(i));
        }
        ((Message)message).setRecipients(Message.RecipientType.TO, emailAddresses);
        String subject = "Glowroot alert";
        ImmutableList<String> allTransactionTypes = this.configService.getAllTransactionTypes();
        if (allTransactionTypes.size() != 1 || !((String)allTransactionTypes.get(0)).equals(alertConfig.transactionType())) {
            subject = subject + " - " + alertConfig.transactionType();
        }
        if (ok) {
            subject = subject + " - OK";
        }
        ((Message)message).setSubject(subject);
        StringBuilder sb = new StringBuilder();
        sb.append(AlertingService.getPercentileWithSuffix(alertConfig.percentile()));
        sb.append(" percentile over the last ");
        sb.append(alertConfig.timePeriodMinutes());
        sb.append(" minutes was ");
        sb.append(Math.round((double)valueAtPercentile / 1000.0));
        sb.append(" milliseconds.\n\nTotal transaction count over the last ");
        sb.append(alertConfig.timePeriodMinutes());
        sb.append(" minutes was ");
        sb.append(transactionCount);
        sb.append(".");
        message.setText(sb.toString());
        this.mailService.send(message);
    }

    public static Session createMailSession(SmtpConfig smtpConfig, SecretKey secretKey) throws Exception {
        Properties props = new Properties();
        props.put("mail.smtp.host", smtpConfig.host());
        Integer port = smtpConfig.port();
        if (port == null) {
            port = 25;
        }
        props.put("mail.smtp.port", port);
        if (smtpConfig.ssl()) {
            props.put("mail.smtp.socketFactory.port", port);
            props.put("mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory");
        }
        for (Map.Entry entry : ((ImmutableMap)smtpConfig.additionalProperties()).entrySet()) {
            props.put(entry.getKey(), entry.getValue());
        }
        Authenticator authenticator = null;
        if (!smtpConfig.encryptedPassword().isEmpty()) {
            props.put("mail.smtp.auth", "true");
            final String username = smtpConfig.username();
            final String password = Encryption.decrypt(smtpConfig.encryptedPassword(), secretKey);
            authenticator = new Authenticator(){

                @Override
                protected PasswordAuthentication getPasswordAuthentication() {
                    return new PasswordAuthentication(username, password);
                }
            };
        }
        return Session.getInstance(props, authenticator);
    }

    private static int getRollupLevel(long from, long to) {
        if (to - from <= AggregateDao.ROLLUP_THRESHOLD_MILLIS) {
            return 0;
        }
        return 1;
    }

    @VisibleForTesting
    static String getPercentileWithSuffix(double percentile) {
        String percentileText = new DecimalFormat("0.#########").format(percentile);
        return percentileText + AlertingService.getPercentileSuffix(percentileText);
    }

    private static String getPercentileSuffix(String percentileText) {
        if (percentileText.equals("11") || percentileText.endsWith(".11")) {
            return "th";
        }
        if (percentileText.equals("12") || percentileText.endsWith(".12")) {
            return "th";
        }
        if (percentileText.equals("13") || percentileText.endsWith(".13")) {
            return "th";
        }
        switch (percentileText.charAt(percentileText.length() - 1)) {
            case '1': {
                return "st";
            }
            case '2': {
                return "nd";
            }
            case '3': {
                return "rd";
            }
        }
        return "th";
    }
}

