package org.gnit.lucenekmp.codecs.lucene99


import org.gnit.lucenekmp.codecs.KnnVectorsFormat
import org.gnit.lucenekmp.codecs.KnnVectorsReader
import org.gnit.lucenekmp.codecs.KnnVectorsWriter
import org.gnit.lucenekmp.codecs.hnsw.FlatVectorScorerUtil
import org.gnit.lucenekmp.codecs.hnsw.FlatVectorsFormat
import org.gnit.lucenekmp.index.SegmentReadState
import org.gnit.lucenekmp.index.SegmentWriteState
import org.gnit.lucenekmp.search.TaskExecutor
import org.gnit.lucenekmp.util.hnsw.HnswGraphBuilder
import okio.IOException
import org.gnit.lucenekmp.jdkport.ExecutorService
import kotlin.jvm.JvmOverloads

/**
 * Lucene 9.9 vector format, which encodes numeric vector values into an associated graph connecting
 * the documents having values. The graph is used to power HNSW search. The format consists of two
 * files, and requires a [FlatVectorsFormat] to store the actual vectors:
 *
 * <h2>.vex (vector index)</h2>
 *
 *
 * Stores graphs connecting the documents for each field organized as a list of nodes' neighbours
 * as following:
 *
 *
 *  * For each level:
 *
 *  * For each node:
 *
 *  * **[vint]** the number of neighbor nodes
 *  * **array[vint]** the delta encoded neighbor ordinals
 *
 *
 *  * After all levels are encoded, memory offsets for each node's neighbor nodes are appended to
 * the end of the file. The offsets are encoded by [       ].
 *
 *
 * <h2>.vem (vector metadata) file</h2>
 *
 *
 * For each field:
 *
 *
 *  * **[int32]** field number
 *  * **[int32]** vector similarity function ordinal
 *  * **[vlong]** offset to this field's index in the .vex file
 *  * **[vlong]** length of this field's index data, in bytes
 *  * **[vint]** dimension of this field's vectors
 *  * **[int]** the number of documents having values for this field
 *  * **[vint]** the maximum number of connections (neighbours) that each node can have
 *  * **[vint]** number of levels in the graph
 *  * Graph nodes by level. For each level
 *
 *  * **[vint]** the number of nodes on this level
 *  * **array[vint]** for levels greater than 0 list of nodes on this level, stored as
 * the level 0th delta encoded nodes' ordinals.
 *
 *
 *
 * @lucene.experimental
 */
class Lucene99HnswVectorsFormat @JvmOverloads constructor(
    maxConn: Int = DEFAULT_MAX_CONN,
    beamWidth: Int = DEFAULT_BEAM_WIDTH,
    numMergeWorkers: Int = DEFAULT_NUM_MERGE_WORKER,
    mergeExec: ExecutorService? = null
) : KnnVectorsFormat("Lucene99HnswVectorsFormat") {
    /**
     * Controls how many of the nearest neighbor candidates are connected to the new node. Defaults to
     * [Lucene99HnswVectorsFormat.DEFAULT_MAX_CONN]. See [HnswGraph] for more details.
     */
    private val maxConn: Int

    /**
     * The number of candidate neighbors to track while searching the graph for each newly inserted
     * node. Defaults to [Lucene99HnswVectorsFormat.DEFAULT_BEAM_WIDTH]. See [HnswGraph]
     * for details.
     */
    private val beamWidth: Int

    private val numMergeWorkers: Int
    private val mergeExec: TaskExecutor?

    /**
     * Constructs a format using the given graph construction parameters and scalar quantization.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     * @param numMergeWorkers number of workers (threads) that will be used when doing merge. If
     * larger than 1, a non-null [ExecutorService] must be passed as mergeExec
     * @param mergeExec the [ExecutorService] that will be used by ALL vector writers that are
     * generated by this format to do the merge. If null, the configured [     ][MergeScheduler.getIntraMergeExecutor] is used.
     */
    /** Constructs a format using default graph construction parameters  */
    /**
     * Constructs a format using the given graph construction parameters.
     *
     * @param maxConn the maximum number of connections to a node in the HNSW graph
     * @param beamWidth the size of the queue maintained during graph construction.
     */
    init {
        require(!(maxConn <= 0 || maxConn > MAXIMUM_MAX_CONN)) {
            ("maxConn must be positive and less than or equal to "
                    + MAXIMUM_MAX_CONN
                    + "; maxConn="
                    + maxConn)
        }
        require(!(beamWidth <= 0 || beamWidth > MAXIMUM_BEAM_WIDTH)) {
            ("beamWidth must be positive and less than or equal to "
                    + MAXIMUM_BEAM_WIDTH
                    + "; beamWidth="
                    + beamWidth)
        }
        this.maxConn = maxConn
        this.beamWidth = beamWidth
        require(!(numMergeWorkers == 1 && mergeExec != null)) { "No executor service is needed as we'll use single thread to merge" }
        this.numMergeWorkers = numMergeWorkers
        if (mergeExec != null) {
            this.mergeExec = TaskExecutor(mergeExec)
        } else {
            this.mergeExec = null
        }
    }

    @Throws(IOException::class)
    override fun fieldsWriter(state: SegmentWriteState): KnnVectorsWriter {
        return Lucene99HnswVectorsWriter(
            state,
            maxConn,
            beamWidth,
            flatVectorsFormat.fieldsWriter(state),
            numMergeWorkers,
            mergeExec
        )
    }

    @Throws(IOException::class)
    override fun fieldsReader(state: SegmentReadState): KnnVectorsReader {
        return Lucene99HnswVectorsReader(state, flatVectorsFormat.fieldsReader(state))
    }

    override fun getMaxDimensions(fieldName: String): Int {
        return 1024
    }

    override fun toString(): String {
        return ("Lucene99HnswVectorsFormat(name=Lucene99HnswVectorsFormat, maxConn="
                + maxConn
                + ", beamWidth="
                + beamWidth
                + ", flatVectorFormat="
                + flatVectorsFormat
                + ")")
    }

    companion object {
        const val META_CODEC_NAME: String = "Lucene99HnswVectorsFormatMeta"
        const val VECTOR_INDEX_CODEC_NAME: String = "Lucene99HnswVectorsFormatIndex"
        const val META_EXTENSION: String = "vem"
        const val VECTOR_INDEX_EXTENSION: String = "vex"

        const val VERSION_START: Int = 0
        const val VERSION_CURRENT: Int = VERSION_START

        /**
         * A maximum configurable maximum max conn.
         *
         *
         * NOTE: We eagerly populate `float[MAX_CONN*2]` and `int[MAX_CONN*2]`, so exceptionally large
         * numbers here will use an inordinate amount of heap
         */
        const val MAXIMUM_MAX_CONN: Int = 512

        /** Default number of maximum connections per node  */
        const val DEFAULT_MAX_CONN: Int = HnswGraphBuilder.DEFAULT_MAX_CONN

        /**
         * The maximum size of the queue to maintain while searching during graph construction. This
         * maximum value preserves the ratio of the `DEFAULT_BEAM_WIDTH`/`DEFAULT_MAX_CONN` (i.e. `6.25 *
         * 16 = 3200`).
         */
        const val MAXIMUM_BEAM_WIDTH: Int = 3200

        /**
         * Default number of the size of the queue maintained while searching during a graph construction.
         */
        const val DEFAULT_BEAM_WIDTH: Int = HnswGraphBuilder.DEFAULT_BEAM_WIDTH

        /** Default to use single thread merge  */
        const val DEFAULT_NUM_MERGE_WORKER: Int = 1

        const val DIRECT_MONOTONIC_BLOCK_SHIFT: Int = 16

        /** The format for storing, reading, and merging vectors on disk.  */
        private val flatVectorsFormat: FlatVectorsFormat =
            Lucene99FlatVectorsFormat(FlatVectorScorerUtil.getLucene99FlatVectorsScorer())
    }
}
