package org.gnit.lucenekmp.jdkport


/**
 * port of java.lang.CharacterData01
 *
 * The CharacterData class encapsulates the large tables once found in
 * Character.
 */
internal class CharacterData01 private constructor() // The X table has 2048 entries for a total of 4096 bytes.
    : CharacterData() {
    /* The character properties are currently encoded into 32 bits in the following manner:
            1 bit   mirrored property
            4 bits  directionality property
            9 bits  signed offset used for converting case
            1 bit   if 1, adding the signed offset converts the character to lowercase
            1 bit   if 1, subtracting the signed offset converts the character to uppercase
            1 bit   if 1, this character has a titlecase equivalent (possibly itself)
            3 bits  0  may not be part of an identifier
                    1  ignorable control; may continue a Unicode identifier or Java identifier
                    2  may continue a Java identifier but not a Unicode identifier (unused)
                    3  may continue a Unicode identifier or Java identifier
                    4  is a Java whitespace character
                    5  may start or continue a Java identifier;
                       may continue but not start a Unicode identifier (underscores)
                    6  may start or continue a Java identifier but not a Unicode identifier ($)
                    7  may start or continue a Unicode identifier or Java identifier
                    Thus:
                       5, 6, 7 may start a Java identifier
                       1, 2, 3, 5, 6, 7 may continue a Java identifier
                       7 may start a Unicode identifier
                       1, 3, 5, 7 may continue a Unicode identifier
                       1 is ignorable within an identifier
                       4 is Java whitespace
            2 bits  0  this character has no numeric property
                    1  adding the digit offset to the character code and then
                       masking with 0x1F will produce the desired numeric value
                    2  this character has a "strange" numeric value
                    3  a Java supradecimal digit: adding the digit offset to the
                       character code, then masking with 0x1F, then adding 10
                       will produce the desired numeric value
            5 bits  digit offset
            5 bits  character type

            The encoding of character properties is subject to change at any time.
         */
    override fun getProperties(ch: Int): Int {
        val offset = ch.toChar()
        val props = A[Y[X[offset.code shr 5].code or ((offset.code shr 1) and 0xF)].code or (offset.code and 0x1)]
        return props
    }

    fun getPropertiesEx(ch: Int): Int {
        val offset = ch.toChar()
        val props =
            B[Y[X[offset.code shr 5].code or ((offset.code shr 1) and 0xF)].code or (offset.code and 0x1)].code
        return props
    }

    override fun getType(ch: Int): Int {
        val props = getProperties(ch)
        return (props and 0x1F)
    }

    override fun isOtherAlphabetic(ch: Int): Boolean {
        val props = getPropertiesEx(ch)
        return (props and 0x0004) != 0
    }

    override fun isIdeographic(ch: Int): Boolean {
        val props = getPropertiesEx(ch)
        return (props and 0x0008) != 0
    }

    override fun isJavaIdentifierStart(ch: Int): Boolean {
        val props = getProperties(ch)
        return ((props and 0x00007000) >= 0x00005000)
    }

    override fun isJavaIdentifierPart(ch: Int): Boolean {
        val props = getProperties(ch)
        return ((props and 0x00003000) != 0)
    }

    override fun isUnicodeIdentifierStart(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0010) != 0
    }

    override fun isUnicodeIdentifierPart(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0020) != 0 ||
                isIdentifierIgnorable(ch)
    }

    override fun isIdentifierIgnorable(ch: Int): Boolean {
        val props = getProperties(ch)
        return ((props and 0x00007000) == 0x00001000)
    }

    override fun isEmoji(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0040) != 0
    }

    override fun isEmojiPresentation(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0080) != 0
    }

    override fun isEmojiModifier(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0100) != 0
    }

    override fun isEmojiModifierBase(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0200) != 0
    }

    override fun isEmojiComponent(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0400) != 0
    }

    override fun isExtendedPictographic(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0800) != 0
    }

    override fun toLowerCase(ch: Int): Int {
        var mapChar = ch
        val `val` = getProperties(ch)

        if ((`val` and 0x00020000) != 0) {
            val offset = `val` shl 5 shr (5 + 18)
            mapChar = ch + offset
        }
        return mapChar
    }

    override fun toUpperCase(ch: Int): Int {
        var mapChar = ch
        val `val` = getProperties(ch)

        if ((`val` and 0x00010000) != 0) {
            val offset = `val` shl 5 shr (5 + 18)
            mapChar = ch - offset
        }
        return mapChar
    }

    override fun toTitleCase(ch: Int): Int {
        var mapChar = ch
        val `val` = getProperties(ch)

        if ((`val` and 0x00008000) != 0) {
            // There is a titlecase equivalent.  Perform further checks:
            if ((`val` and 0x00010000) == 0) {
                // The character does not have an uppercase equivalent, so it must
                // already be uppercase; so add 1 to get the titlecase form.
                mapChar = ch + 1
            } else if ((`val` and 0x00020000) == 0) {
                // The character does not have a lowercase equivalent, so it must
                // already be lowercase; so subtract 1 to get the titlecase form.
                mapChar = ch - 1
            }
            // else {
            // The character has both an uppercase equivalent and a lowercase
            // equivalent, so it must itself be a titlecase form; return it.
            // return ch;
            //}
        } else if ((`val` and 0x00010000) != 0) {
            // This character has no titlecase equivalent but it does have an
            // uppercase equivalent, so use that (subtract the signed case offset).
            mapChar = toUpperCase(ch)
        }
        return mapChar
    }

    override fun digit(ch: Int, radix: Int): Int {
        var value = -1
        if (radix >= Character.MIN_RADIX && radix <= Character.MAX_RADIX) {
            val `val` = getProperties(ch)
            val kind = `val` and 0x1F
            if (kind == Character.DECIMAL_DIGIT_NUMBER.toInt()) {
                value = ch + ((`val` and 0x3E0) shr 5) and 0x1F
            } else if ((`val` and 0xC00) == 0x00000C00) {
                // Java supradecimal digit
                value = (ch + ((`val` and 0x3E0) shr 5) and 0x1F) + 10
            }
        }
        return if (value < radix) value else -1
    }

    override fun getNumericValue(ch: Int): Int {
        val `val` = getProperties(ch)
        var retval: Int

        when (`val` and 0xC00) {
            (0x00000000) -> retval = -1
            (0x00000400) -> retval = ch + ((`val` and 0x3E0) shr 5) and 0x1F
            (0x00000800) -> when (ch) {
                0x10113 -> retval = 40
                0x10114 -> retval = 50
                0x10115 -> retval = 60
                0x10116 -> retval = 70
                0x10117 -> retval = 80
                0x10118 -> retval = 90
                0x10119 -> retval = 100
                0x1011A -> retval = 200
                0x1011B -> retval = 300
                0x1011C -> retval = 400
                0x1011D -> retval = 500
                0x1011E -> retval = 600
                0x1011F -> retval = 700
                0x10120 -> retval = 800
                0x10121 -> retval = 900
                0x10122 -> retval = 1000
                0x10123 -> retval = 2000
                0x10124 -> retval = 3000
                0x10125 -> retval = 4000
                0x10126 -> retval = 5000
                0x10127 -> retval = 6000
                0x10128 -> retval = 7000
                0x10129 -> retval = 8000
                0x1012A -> retval = 9000
                0x1012B -> retval = 10000
                0x1012C -> retval = 20000
                0x1012D -> retval = 30000
                0x1012E -> retval = 40000
                0x1012F -> retval = 50000
                0x10130 -> retval = 60000
                0x10131 -> retval = 70000
                0x10132 -> retval = 80000
                0x10133 -> retval = 90000
                0x10144 -> retval = 50
                0x10145 -> retval = 500
                0x10146 -> retval = 5000
                0x10147 -> retval = 50000
                0x1014A -> retval = 50
                0x1014B -> retval = 100
                0x1014C -> retval = 500
                0x1014D -> retval = 1000
                0x1014E -> retval = 5000
                0x10151 -> retval = 50
                0x10152 -> retval = 100
                0x10153 -> retval = 500
                0x10154 -> retval = 1000
                0x10155 -> retval = 10000
                0x10156 -> retval = 50000
                0x10166 -> retval = 50
                0x10167 -> retval = 50
                0x10168 -> retval = 50
                0x10169 -> retval = 50
                0x1016A -> retval = 100
                0x1016B -> retval = 300
                0x1016C -> retval = 500
                0x1016D -> retval = 500
                0x1016E -> retval = 500
                0x1016F -> retval = 500
                0x10170 -> retval = 500
                0x10171 -> retval = 1000
                0x10172 -> retval = 5000
                0x10174 -> retval = 50
                0x102ED -> retval = 40
                0x102EE -> retval = 50
                0x102EF -> retval = 60
                0x102F0 -> retval = 70
                0x102F1 -> retval = 80
                0x102F2 -> retval = 90
                0x102F3 -> retval = 100
                0x102F4 -> retval = 200
                0x102F5 -> retval = 300
                0x102F6 -> retval = 400
                0x102F7 -> retval = 500
                0x102F8 -> retval = 600
                0x102F9 -> retval = 700
                0x102FA -> retval = 800
                0x102FB -> retval = 900
                0x10323 -> retval = 50
                0x10341 -> retval = 90
                0x1034A -> retval = 900
                0x103D5 -> retval = 100
                0x1085D -> retval = 100
                0x1085E -> retval = 1000
                0x1085F -> retval = 10000
                0x108AF -> retval = 100
                0x108FF -> retval = 100
                0x10919 -> retval = 100
                0x109CC -> retval = 40
                0x109CD -> retval = 50
                0x109CE -> retval = 60
                0x109CF -> retval = 70
                0x109D2 -> retval = 100
                0x109D3 -> retval = 200
                0x109D4 -> retval = 300
                0x109D5 -> retval = 400
                0x109D6 -> retval = 500
                0x109D7 -> retval = 600
                0x109D8 -> retval = 700
                0x109D9 -> retval = 800
                0x109DA -> retval = 900
                0x109DB -> retval = 1000
                0x109DC -> retval = 2000
                0x109DD -> retval = 3000
                0x109DE -> retval = 4000
                0x109DF -> retval = 5000
                0x109E0 -> retval = 6000
                0x109E1 -> retval = 7000
                0x109E2 -> retval = 8000
                0x109E3 -> retval = 9000
                0x109E4 -> retval = 10000
                0x109E5 -> retval = 20000
                0x109E6 -> retval = 30000
                0x109E7 -> retval = 40000
                0x109E8 -> retval = 50000
                0x109E9 -> retval = 60000
                0x109EA -> retval = 70000
                0x109EB -> retval = 80000
                0x109EC -> retval = 90000
                0x109ED -> retval = 100000
                0x109EE -> retval = 200000
                0x109EF -> retval = 300000
                0x109F0 -> retval = 400000
                0x109F1 -> retval = 500000
                0x109F2 -> retval = 600000
                0x109F3 -> retval = 700000
                0x109F4 -> retval = 800000
                0x109F5 -> retval = 900000
                0x10A46 -> retval = 100
                0x10A47 -> retval = 1000
                0x10A7E -> retval = 50
                0x10AEF -> retval = 100
                0x10B5E -> retval = 100
                0x10B5F -> retval = 1000
                0x10B7E -> retval = 100
                0x10B7F -> retval = 1000
                0x10BAF -> retval = 100
                0x10CFD -> retval = 50
                0x10CFE -> retval = 100
                0x10CFF -> retval = 1000
                0x10E6C -> retval = 40
                0x10E6D -> retval = 50
                0x10E6E -> retval = 60
                0x10E6F -> retval = 70
                0x10E70 -> retval = 80
                0x10E71 -> retval = 90
                0x10E72 -> retval = 100
                0x10E73 -> retval = 200
                0x10E74 -> retval = 300
                0x10E75 -> retval = 400
                0x10E76 -> retval = 500
                0x10E77 -> retval = 600
                0x10E78 -> retval = 700
                0x10E79 -> retval = 800
                0x10E7A -> retval = 900
                0x10F25 -> retval = 100
                0x10F54 -> retval = 100
                0x10FCB -> retval = 100
                0x1105E -> retval = 40
                0x1105F -> retval = 50
                0x11060 -> retval = 60
                0x11061 -> retval = 70
                0x11062 -> retval = 80
                0x11063 -> retval = 90
                0x11064 -> retval = 100
                0x11065 -> retval = 1000
                0x11C66 -> retval = 40
                0x11C67 -> retval = 50
                0x11C68 -> retval = 60
                0x11C69 -> retval = 70
                0x11C6A -> retval = 80
                0x11C6B -> retval = 90
                0x11C6C -> retval = 100
                0x111ED -> retval = 40
                0x111EE -> retval = 50
                0x111EF -> retval = 60
                0x111F0 -> retval = 70
                0x111F1 -> retval = 80
                0x111F2 -> retval = 90
                0x111F3 -> retval = 100
                0x111F4 -> retval = 1000
                0x118ED -> retval = 40
                0x118EE -> retval = 50
                0x118EF -> retval = 60
                0x118F0 -> retval = 70
                0x118F1 -> retval = 80
                0x118F2 -> retval = 90
                0x12432 -> retval = 216000
                0x12433 -> retval = 432000
                0x12467 -> retval = 40
                0x12468 -> retval = 50
                0x16B5C -> retval = 100
                0x16B5D -> retval = 10000
                0x16B5E -> retval = 1000000
                0x16B5F -> retval = 100000000
                0x1D36C -> retval = 40
                0x1D36D -> retval = 50
                0x1D36E -> retval = 60
                0x1D36F -> retval = 70
                0x1D370 -> retval = 80
                0x1D371 -> retval = 90
                0x1EC7D -> retval = 40
                0x1EC7E -> retval = 50
                0x1EC7F -> retval = 60
                0X1EC80 -> retval = 70
                0X1EC81 -> retval = 80
                0X1EC82 -> retval = 90
                0X1EC83 -> retval = 100
                0X1EC84 -> retval = 200
                0X1EC85 -> retval = 300
                0X1EC86 -> retval = 400
                0X1EC87 -> retval = 500
                0X1EC88 -> retval = 600
                0X1EC89 -> retval = 700
                0X1EC8A -> retval = 800
                0X1EC8B -> retval = 900
                0X1EC8C -> retval = 1000
                0X1EC8D -> retval = 2000
                0X1EC8E -> retval = 3000
                0X1EC8F -> retval = 4000
                0X1EC90 -> retval = 5000
                0X1EC91 -> retval = 6000
                0X1EC92 -> retval = 7000
                0X1EC93 -> retval = 8000
                0X1EC94 -> retval = 9000
                0X1EC95 -> retval = 10000
                0X1EC96 -> retval = 20000
                0X1EC97 -> retval = 30000
                0X1EC98 -> retval = 40000
                0X1EC99 -> retval = 50000
                0X1EC9A -> retval = 60000
                0X1EC9B -> retval = 70000
                0X1EC9C -> retval = 80000
                0X1EC9D -> retval = 90000
                0X1EC9E -> retval = 100000
                0X1EC9F -> retval = 200000
                0X1ECA0 -> retval = 100000
                0X1ECA1 -> retval = 10000000
                0X1ECA2 -> retval = 20000000
                0X1ECB3 -> retval = 10000
                0X1ECB4 -> retval = 100000
                0X1ED0D -> retval = 40
                0X1ED0E -> retval = 50
                0X1ED0F -> retval = 60
                0X1ED10 -> retval = 70
                0X1ED11 -> retval = 80
                0X1ED12 -> retval = 90
                0X1ED13 -> retval = 100
                0X1ED14 -> retval = 200
                0X1ED15 -> retval = 300
                0X1ED16 -> retval = 400
                0X1ED17 -> retval = 500
                0X1ED18 -> retval = 600
                0X1ED19 -> retval = 700
                0X1ED1A -> retval = 800
                0X1ED1B -> retval = 900
                0X1ED1C -> retval = 1000
                0X1ED1D -> retval = 2000
                0X1ED1E -> retval = 3000
                0X1ED1F -> retval = 4000
                0X1ED20 -> retval = 5000
                0X1ED21 -> retval = 6000
                0X1ED22 -> retval = 7000
                0X1ED23 -> retval = 8000
                0X1ED24 -> retval = 9000
                0X1ED25 -> retval = 10000
                0X1ED26 -> retval = 20000
                0X1ED27 -> retval = 30000
                0X1ED28 -> retval = 40000
                0X1ED29 -> retval = 50000
                0X1ED2A -> retval = 60000
                0X1ED2B -> retval = 70000
                0X1ED2C -> retval = 80000
                0X1ED2D -> retval = 90000
                0X1ED38 -> retval = 400
                0X1ED39 -> retval = 600
                0X1ED3A -> retval = 2000
                0X1ED3B -> retval = 10000
                else -> retval = -2
            }

            (0x00000C00) -> retval = (ch + ((`val` and 0x3E0) shr 5) and 0x1F) + 10
            else -> retval = -1
        }
        return retval
    }

    override fun isDigit(ch: Int): Boolean {
        val props = getProperties(ch)
        return (props and 0x1F) == Character.DECIMAL_DIGIT_NUMBER.toInt()
    }

    override fun isLowerCase(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0001) != 0
    }

    override fun isUpperCase(ch: Int): Boolean {
        return (getPropertiesEx(ch) and 0x0002) != 0
    }

    override fun isWhitespace(ch: Int): Boolean {
        val props = getProperties(ch)
        return ((props and 0x00007000) == 0x00004000)
    }

    override fun getDirectionality(ch: Int): Byte {
        val `val` = getProperties(ch)
        var directionality = ((`val` and 0x78000000) shr 27).toByte()
        if (directionality.toInt() == 0xF) {
            directionality = Character.DIRECTIONALITY_UNDEFINED
        }
        return directionality
    }

    override fun isMirrored(ch: Int): Boolean {
        val props = getProperties(ch)
        return ((props and -0x80000000) != 0)
    }

    companion object {
        val instance: CharacterData = CharacterData01()
        val X: CharArray =
            ("\u0000\u0010\u0020\u0030\u0040\u0040\u0040\u0050\u0060\u0070\u0080\u0090\u00a0\u00b0\u00c0\u00d0\u0030\u0030\u0030" +
                    "\u0030\u00e0\u0040\u00f0\u0100\u0040\u0110\u0120\u0130\u0140\u0040\u0150\u0030\u0160\u0170" +
                    "\u0180\u0040\u0190\u01A0\u01B0\u01C0\u0040\u01D0\u0040\u01E0\u01F0\u0200\u0040\u0210" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0220\u0230\u0240\u0250\u0260\u0030\u0030" +
                    "\u0270\u0280\u0290\u02A0\u02B0\u02C0\u0030\u02D0\u02E0\u02F0\u0030\u0030\u0300\u0310" +
                    "\u0320\u0330\u0340\u0350\u0360\u0370\u0380\u0030\u0390\u03A0\u0300\u03B0\u03C0" +
                    "\u03D0\u03E0\u03F0\u0030\u0030\u0300\u0300\u0400\u0030\u0410\u0420\u0430\u0440\u0450" +
                    "\u0460\u0470\u0480\u0490\u0030\u0030\u0030\u0030\u0030\u0030\u04A0\u0300\u04B0\u04C0\u04D0" +
                    "\u04E0\u04F0\u0500\u0510\u0520\u0510\u0530\u0540\u0550\u0560\u0570\u0580\u0590" +
                    "\u05A0\u05B0\u05C0\u05D0\u05E0\u05F0\u0600\u0590\u0610\u0620\u0630\u0640\u0650" +
                    "\u0660\u0030\u0670\u0680\u0690\u06A0\u06B0\u06C0\u06D0\u06E0\u06F0\u0700\u0710" +
                    "\u0720\u0040\u0730\u0740\u0750\u0040\u0760\u0770\u0030\u0030\u0030\u0030\u0030\u0040\u0780" +
                    "\u0790\u0030\u0040\u07A0\u07B0\u07C0\u0040\u07D0\u07E0\u07F0\u0800\u0810\u0820\u0030" +
                    "\u0030\u0030\u0030\u0030\u0040\u0830\u0030\u0030\u0030\u0840\u0850\u0860\u0870\u0880\u0890" +
                    "\u0030\u0030\u08A0\u08B0\u08C0\u08D0\u08E0\u08F0\u0040\u0900\u0910\u0040\u0920\u0930" +
                    "\u0030\u0030\u0030\u0030\u0030\u0040\u0940\u0950\u0960\u0970\u0980\u0990\u09A0\u0030\u0030" +
                    "\u09B0\u09C0\u09D0\u09E0\u09F0\u0A00\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0A10" +
                    "\u0A20\u0A30\u0A40\u0030\u0030\u0A50\u0A60\u0A70\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0A80\u0030\u0030\u0030\u0A90\u0AA0\u0AB0\u0AC0\u0040\u0040\u0040\u0040\u0040\u0040\u0AD0" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0AE0\u0040\u0040\u0AF0\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0B00\u0B10\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0050" +
                    "\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0820" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0B20\u0B30\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0920\u0B40\u0B50\u0040\u0B40\u0B60\u0B70\u0040\u0B80\u0B90\u0BA0\u0BB0" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0BC0\u0BD0\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0840\u0850\u0BE0\u0030\u0030\u0030\u0040\u0040\u0BF0\u0C00\u0C10\u0030\u0030" +
                    "\u0C20\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C40\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C50\u0C60\u0C70" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0C80\u0040\u0040\u0040\u0040\u0040" +
                    "\u0040\u0040\u0040\u0040\u0C90\u0CA0\u0CB0\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30\u0C30" +
                    "\u0C30\u0C30\u0C30\u0C30\u0CC0\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0040\u0040\u0040\u0CD0\u0CE0\u0CF0\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0D00\u0D00\u0D00\u0D00\u0D00\u0D00\u0D10\u0D20\u0D00\u0D00\u0D00\u0D00" +
                    "\u0D00\u0D00\u0D00\u0D00\u0D00\u0D00\u0D00\u0D00\u0D00\u0D30\u0030\u0030\u0D40" +
                    "\u0D50\u0D60\u0D70\u0D70\u0D70\u0D80\u0030\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70" +
                    "\u0D70\u0D90\u0D70\u0DA0\u0D70\u0DB0\u0DC0\u0DD0\u0D70\u0DE0\u0D00\u0D00\u0DF0" +
                    "\u0030\u0030\u0030\u0E00\u0E00\u0D00\u0D00\u0E10\u0E20\u0030\u0030\u0030\u0030\u0E30\u0E40" +
                    "\u0E50\u0E60\u0E70\u0E80\u0E90\u0EA0\u0EB0\u0EC0\u0ED0\u0EE0\u0EF0\u0E30\u0E40" +
                    "\u0F00\u0E60\u0F10\u0F20\u0F30\u0EA0\u0F40\u0F50\u0F60\u0F70\u0F80\u0F90\u0FA0" +
                    "\u0FB0\u0FC0\u0FD0\u0FE0\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70" +
                    "\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70\u0D70\u0D40\u0FF0\u0D40\u1000\u1010\u1020" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u1030\u1040\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u1050\u1060\u1070\u1080\u1090\u0030\u0030\u0030\u0040\u10A0\u10B0" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0AE0\u10C0\u0040\u10D0\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0AE0\u10E0\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0AE0\u10F0\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u1100" +
                    "\u0300\u0300\u0300\u0300\u0300\u0300\u1110\u0030\u1120\u1130\u1140\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u1150\u1160\u1170\u0030\u0030\u1180\u1190\u0030\u0030\u0030\u0030\u0030\u0030\u11A0" +
                    "\u11B0\u11C0\u11D0\u11E0\u11F0\u0030\u1200\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u1210" +
                    "\u1220\u1230\u1230\u1240\u1250\u1260\u1270\u1280\u1290\u12A0\u12B0\u12C0\u12D0" +
                    "\u12E0\u12F0\u1300\u1310\u1320\u1330\u12E0\u12E0\u12E0\u12E0\u1340\u1350\u1340" +
                    "\u1360\u1370\u1340\u1380\u1390\u1340\u13A0\u13B0\u13C0\u13D0\u13E0\u1340\u13F0" +
                    "\u1340\u1400\u1410\u1420\u1430\u1440\u1450\u1460\u1340\u1340\u1470\u0D00\u1340" +
                    "\u1480\u1490\u14A0\u0D00\u0D00\u0D00\u14B0\u0D00\u0D00\u14C0\u14D0\u14E0\u0D00" +
                    "\u14F0\u0D00\u1500\u1510\u1520\u12E0\u1530\u1540\u1550\u1560\u1340\u1570\u1580" +
                    "\u1340\u1230\u1230\u1240\u1590\u15A0\u1340\u15B0\u15C0\u0D00\u0D00\u0D00\u0D00" +
                    "\u15D0\u0D00\u0D00\u15E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0" +
                    "\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0" +
                    "\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u12E0\u15F0").toCharArray()

        // The Y table has 5632 entries for a total of 11264 bytes.
        val Y: CharArray =
            ("\u0000\u0000\u0000\u0000\u0000\u0000\u0002\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0004\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0000\u0002\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0006" +
                    "\u0006\u0008\u000a\u0006\u000c\u000e\u000e\u000e\u000e\u0010\u0012\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0014" +
                    "\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0006\u0016\u0018\u0018\u0018\u0018\u001a\u001c\u001a\u001a\u001e" +
                    "\u001a\u001a\u0020\u0022\u001a\u001a\u0024\u0026\u0028\u002a\u002c\u002e\u0030\u0032\u001a\u001a\u001a\u001a\u001a" +
                    "\u001a\u0034\u0036\u0038\u003a\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003e\u0040\u0042\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u0044\u0006\u0044\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0046\u0006\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0004\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0048\u004a\u004a\u004a\u004a\u004c\u004e\u0050\u0050\u0050" +
                    "\u0050\u0050\u0050\u0050\u0006\u0006\u0052\u0054\u0006\u0006\u0006\u0006\u0002\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0056\u0000\u0000\u0000\u0000\u0058\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u005a\u005a\u005c\u0006\u0006\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u005e\u0000\u0000\u0006\u0006\u0000\u0000" +
                    "\u0000\u0000\u0060\u0062\u0064\u0006\u0006\u0006\u0006\u0006\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066" +
                    "\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068" +
                    "\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u006a\u006a" +
                    "\u006a\u006a\u006a\u0006\u0006\u0006\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066\u0066" +
                    "\u0066\u0066\u0066\u0066\u0066\u0006\u0006\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068\u0068" +
                    "\u0068\u0068\u0068\u0068\u0068\u0068\u0006\u0006\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u005e\u006c\u006c\u006c\u006c\u006c\u006e" +
                    "\u006c\u006c\u006c\u006c\u006c\u006c\u006c\u006e\u006c\u006c\u006c\u006e\u006c\u0070\u0072\u0072\u0072\u0072\u0072" +
                    "\u0070\u0072\u0072\u0072\u0072\u0072\u0072\u0072\u0070\u0072\u0072\u0072\u0070\u0074\u0006\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0004\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0076\u0078\u007a\u007c\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a" +
                    "\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007e\u007a\u007a\u007a\u007a\u007e\u0006\u0006\u0080\u0080\u0080" +
                    "\u0006\u0082\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0084\u0082\u0006\u0082\u0084\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0086\u0088\u008a\u008c\u008e\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0090" +
                    "\u0092\u0094\u0094\u0096\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0082\u0006\u0006\u0006\u0098\u009a\u009c\u009e\u00a0\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0082\u0080\u0006\u0006\u00a2\u00a4\u00a6\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0080\u00a8\u00aa\u0094\u0006\u00ac\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0006\u0006\u0086\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0006\u0006" +
                    "\u008e\u0080\u00ae\u00ae\u00ae\u00ae\u00ae\u00b0\u008e\u008e\u0006\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e" +
                    "\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u008e\u00b2\u005a\u00b4\u005c" +
                    "\u0006\u0006\u005a\u005a\u0080\u0080\u0084\u0080\u0084\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0006\u00b6\u00b8\u0006\u00ba\u00ae\u00ae\u00bc\u008e\u00be\u0006\u0006\u0006\u00c0\u00c0" +
                    "\u00c0\u00c0\u00c2\u0006\u0006\u0006\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u00c4\u00c6\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u00c4\u0096" +
                    "\u0080\u0080\u0080\u0080\u00c8\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u00ca" +
                    "\u00b8\u0006\u00cc\u009e\u00a0\u00c0\u00c0\u00c0\u00c2\u0006\u0006\u0006\u0006\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0006\u00ac\u00ce\u00ce\u00ce\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0006\u0088\u0088\u00d0\u008e\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0082\u0006\u0006" +
                    "\u0088\u0088\u00d0\u008e\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0006\u0006\u0006\u0086\u00c0\u00c2" +
                    "\u0006\u0006\u0006\u0006\u0006\u00d2\u00d4\u00d6\u00a0\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0080\u0080" +
                    "\u0080\u0080\u0082\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u00d8\u00d8\u00d8\u00d8\u00d8" +
                    "\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8\u00d8" +
                    "\u00d8\u00da\u0006\u0006\u0006\u0006\u0006\u0006\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc" +
                    "\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00dc\u00de\u0006\u0006\u0006\u00e0" +
                    "\u00e2\u008e\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4" +
                    "\u00e4\u005a\u005a\u0006\u0006\u0006\u0006\u00e6\u00e6\u00e6\u00e6\u00e6\u0006\u0006\u0006\u00e8\u00e8\u00e8\u00e8" +
                    "\u00e8\u0080\u0080\u00ea\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u00ec\u0006\u00b4\u00b6\u00b6" +
                    "\u00ee\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u00f0\u0006\u0006\u0006\u0006\u00f2\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u00f4\u00f4\u00f4\u00f4\u00f4\u00f6\u00f8\u00f8\u00f8\u00f8\u00f8\u00f8\u00f8" +
                    "\u00f8\u00f8\u00fa\u0080\u0080\u0080\u0080\u0080\u00b4\u00fc\u0006\u0080\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u00e4\u00fe\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0100\u00b6\u0080\u0080\u0080\u0080\u0080" +
                    "\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u00c4\u0102\u0102\u0104\u0106\u0108\u0006" +
                    "\u0006\u0006\u0006\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00b6\u00b6\u00b6\u00b6\u00b6" +
                    "\u010A\u010C\u010E\u0110\u0110\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u00b6\u00b6\u00c0\u00c0\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0080\u0080\u0112\u0114\u0116\u0118\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0080\u0082\u0006\u0006\u0006" +
                    "\u0006\u011A\u011C\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u011E" +
                    "\u0120\u0120\u0120\u0006\u0006\u0122\u0122\u0122\u0122\u0122\u0124\u0038\u0038\u0038" +
                    "\u0038\u0126\u0126\u0126\u0126\u0126\u0128\u012A\u012C\u0006\u0006\u0006\u0006\u00ba\u005a" +
                    "\u011C\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u012E\u011A\u005a\u0130\u0132\u011E\u0134\u0120\u0120\u005c" +
                    "\u0006\u0006\u0006\u0006\u0136\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0004\u0006\u0006\u0006\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u005a\u012C\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012A\u005a\u005a" +
                    "\u011A\u005a\u005a\u0100\u00b8\u013A\u013A\u013A\u013A\u013A\u0120\u0120\u013C\u011C" +
                    "\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u013E\u0120\u0004\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u013C\u012E\u005a\u005a\u005a\u005a\u0130\u0140\u0000\u0142\u0120\u0144\u00b6\u011E\u011A" +
                    "\u0138\u0138\u0138\u0138\u0138\u0142\u0142\u0120\u0146\u0148\u0148\u0148\u0148" +
                    "\u014A\u014C\u0014\u0014\u0014\u014E\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0002\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012E\u011A" +
                    "\u005a\u012E\u0150\u0152\u0120\u0120\u0120\u012C\u012A\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0004\u0004\u0000\u0000\u0002\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0002\u0000\u0000\u0000\u0000\u0142\u0006\u0006\u0006\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u012A\u012E\u011A\u005a\u005a\u0100\u00b8\u0006\u0006\u0138\u0138\u0138\u0138\u0138" +
                    "\u0006\u0006\u0006\u005a\u012E\u0002\u0000\u0000\u0000\u0004\u0002\u0004\u0002\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0004\u0000\u0000\u0000\u0004\u0000\u0002\u0000\u0000\u00ba\u0128\u012E\u0130" +
                    "\u012E\u0154\u0156\u0154\u0156\u0158\u0006\u0004\u0006\u0006\u0156\u0006\u0006\u0002\u0000" +
                    "\u0000\u012E\u0006\u00b6\u00b6\u00b6\u00b8\u0006\u00b6\u00b6\u00b8\u0006\u0006\u0006\u0006\u0006\u0000\u0000" +
                    "\u0000\u0000\u0000\u0002\u0006\u0004\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0002\u012E\u011A\u005a\u005a\u005c\u0154\u0156\u0156\u012E" +
                    "\u0154\u012E\u015A\u0128\u0128\u0120\u005e\u000a\u0006\u0006\u0006\u00ba\u00b8\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u013C\u012E\u005a\u005a\u005a\u005a\u012E\u0152\u0130\u0128\u0000\u0142\u0120" +
                    "\u0120\u0138\u0138\u0138\u0138\u0138\u0120\u005e\u0128\u0000\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012E" +
                    "\u011A\u005a\u005a\u0130\u0130\u012E\u011A\u0130\u00b6\u0000\u015C\u0006\u0006\u0006\u0006" +
                    "\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u013C" +
                    "\u012E\u005a\u005a\u0006\u012E\u012E\u005a\u0132\u011E\u0120\u0120\u0120\u0120\u0120" +
                    "\u0120\u0120\u0120\u0120\u0120\u0120\u0000\u0000\u005a\u0006\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u012E\u011A\u005a\u005a\u005a\u0130\u011A\u0132\u015E\u0120\u0004\u0006\u0006" +
                    "\u0006\u0006\u0006\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u00ce\u00ce\u00ce\u00ce\u00ce" +
                    "\u00ce\u0160\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u012A\u011A" +
                    "\u012E\u005a\u005a\u005a\u0162\u0142\u0006\u0006\u0006\u006a\u006a\u006a\u006a\u006a\u0006\u0006\u0006" +
                    "\u0138\u0138\u0138\u0138\u0138\u0164\u0164\u0164\u0164\u0164\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0004\u00b4\u011A\u012E\u005a\u005a\u011A\u005a\u0100\u0006\u0006\u0138" +
                    "\u0138\u0138\u0138\u0138\u0166\u0120\u0168\u0000\u0000\u0000\u0004\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u012E\u011A\u005a\u005a" +
                    "\u005a\u005a\u0132\u011E\u0006\u0006\u016A\u016A\u016A\u016A\u016A\u016A\u016A\u016A" +
                    "\u016A\u016A\u016A\u016A\u016A\u016A\u016A\u016A\u016C\u016C\u016C\u016C\u016C" +
                    "\u016C\u016C\u016C\u016C\u016C\u016C\u016C\u016C\u016C\u016C\u016C\u006a\u006a" +
                    "\u006a\u006a\u006a\u014A\u014C\u0014\u0014\u014E\u0006\u0006\u0006\u0006\u0006\u0002\u0000\u0000\u0000" +
                    "\u0004\u0002\u0006\u0000\u0000\u0000\u0000\u0002\u0004\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u012E\u012E\u012E\u0156\u0154\u00b4\u0150\u0128\u011C\u0132\u0120\u000a" +
                    "\u0006\u0006\u0006\u0006\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u0000\u0000\u0000\u0000" +
                    "\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u013C\u012E\u005a\u005a\u0006\u005a\u012E\u012E\u0128\u015C\u0154\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u012A\u005a\u005a\u016E\u0170\u012C\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u013E" +
                    "\u0152\u005a\u0130\u012A\u005a\u015E\u0120\u0120\u0120\u0144\u0006\u0006\u0006\u0006\u012A" +
                    "\u005a\u005a\u0130\u011A\u005a\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u005a\u005a\u005a\u005a\u005a\u005a" +
                    "\u0130\u00b6\u0120\u015C\u0120\u0120\u000a\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0006" +
                    "\u0006\u0006\u0120\u0120\u0120\u0120\u0120\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0142\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0138\u0138\u0138\u0138\u0138\u0006" +
                    "\u0006\u0006\u0000\u0000\u0000\u0000\u0004\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u013C\u005a\u005a\u005a\u005c\u005a\u005a\u005a\u0172\u0142\u0120" +
                    "\u0120\u0006\u0006\u0006\u0006\u0006\u0138\u0138\u0138\u0138\u0138\u0174\u0174\u0174" +
                    "\u0174\u0174\u0166\u0014\u0014\u0014\u014E\u0006\u0120\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u005a" +
                    "\u005a\u0156\u005a\u005a\u005a\u0130\u005a\u011A\u005c\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0004" +
                    "\u0000\u0002\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u012A\u005a\u005a\u005c\u0006\u005c\u005a\u00b4\u005a\u0152\u00b6\u012A\u0006\u0006\u0006\u0006" +
                    "\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u0000\u0000\u0000\u0002\u0004\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012E\u012E\u0154\u005a" +
                    "\u0156\u011A\u0132\u0004\u0006\u0006\u0006\u006a\u006a\u006a\u006a\u006a\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012A\u0130\u0176" +
                    "\u000a\u0006\u0006\u0006\u005a\u013C\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u012E\u005a\u005a\u005c\u0006\u012E" +
                    "\u0150\u011E\u0120\u0120\u0120\u0120\u0120\u0120\u0138\u0138\u0138\u0138\u0138" +
                    "\u00b8\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0004\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0014\u0178\u003c\u003c\u003c\u017A\u017C\u017E" +
                    "\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u0006\u0006\u0006\u0006\u0006\u0006\u005e\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0180\u0180\u0180\u0180\u0182" +
                    "\u0182\u0182\u0184\u0186\u0186\u0188\u018A\u018A\u018A\u018A\u018C\u018C\u018E" +
                    "\u0190\u0192\u0192\u0192\u0186\u0194\u0196\u0198\u019A\u019C\u018A\u019E\u01A0" +
                    "\u01A2\u01A4\u01A6\u01A8\u01AA\u01AC\u01AE\u01AE\u01B0\u01B2\u01B4\u01B6\u018A" +
                    "\u01B8\u0198\u0198\u0198\u0198\u0198\u0198\u0198\u01BA\u0182\u0182\u01BC\u0120" +
                    "\u0120\u000a\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0142\u000a\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u01BE\u01BE\u01BE\u01BE\u01BE\u01BE\u01BE" +
                    "\u01BE\u0128\u0000\u0000\u013E\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u0006\u0006\u0006\u0006\u0006" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u005a\u005a\u005a\u005a" +
                    "\u005a\u005a\u012E\u011A\u0100\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u006a\u006a\u006a\u006a\u006a" +
                    "\u0006\u0006\u0120\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u006a\u006a\u006a\u006a\u006a\u0006\u0006" +
                    "\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u00b6\u00b6" +
                    "\u011E\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u00b6\u00b6\u00b6\u011E" +
                    "\u0120\u0120\u0018\u0018\u01C0\u01C0\u0168\u0006\u0006\u0006\u0006\u0006\u0138\u0138\u0138" +
                    "\u0138\u0138\u01C2\u0014\u0014\u0014\u0002\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0006\u0006\u0002\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u01C0\u01C4\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u01C6\u01C8\u0120\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006" +
                    "\u0148\u0148\u0148\u0148\u0148\u0148\u0148\u0148\u0148\u0148\u01CA\u01CC\u0120" +
                    "\u000a\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0004\u0006\u00b4\u013C\u012E\u012E\u012E\u012E" +
                    "\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E" +
                    "\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u012E\u0006\u0006\u0006\u00b4" +
                    "\u005a\u01CE\u01C0\u01C0\u01C0\u01C0\u01C0\u01C0\u01C0\u01D0\u01D2\u0006\u0006\u0006" +
                    "\u0006\u0006\u012E\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4" +
                    "\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4" +
                    "\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u0006\u0006\u0006\u0006\u01D4" +
                    "\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u01D6\u01D4" +
                    "\u01D4\u01D4\u01D4\u01D8\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u01C0\u01C0\u01DA\u01C0\u01C0\u01C0\u01DA\u01DC\u0000" +
                    "\u0004\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0004\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0000\u0004\u0002\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0006\u0006\u0006" +
                    "\u0006\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4" +
                    "\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u01D4\u0006\u0006\u0000\u0000" +
                    "\u0000\u0000\u0000\u0004\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0006\u0000\u0000\u0000\u0000\u0004" +
                    "\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0006\u0046\u015E\u01DE\u01DE\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u0018" +
                    "\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u01E0\u01E0\u01E0\u01E0\u01E0" +
                    "\u0006\u0006\u0006\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6" +
                    "\u00b6\u00b6\u00b6\u00b6\u0006\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b8\u0006\u0006" +
                    "\u0006\u0006\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0006\u0006\u0006\u0006" +
                    "\u0006\u0018\u0018\u0018\u0042\u0016\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u01E2\u0162\u00b6\u0018\u01E2\u01E4\u01E4\u01E6\u01DE\u01DE\u01DE\u01E8\u00b6\u00b6" +
                    "\u00b6\u01EA\u0046\u00b6\u00b6\u00b6\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u0018\u0018\u0018\u00b6\u00b6\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u01EC" +
                    "\u0044\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u003c\u00b6\u01EE\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0148\u0148\u0148\u0148\u0148\u0148\u0148" +
                    "\u0148\u0148\u0148\u0006\u0006\u0006\u0006\u0006\u0006\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u0044\u0006\u0006\u0006\u0006\u01F0\u01F0\u01F0\u01F0\u01F0\u01F2\u0014\u0014" +
                    "\u0014\u01F4\u01F4\u01F6\u01F8\u0006\u0006\u0006\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u01FE\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u0200\u01FA\u0006\u0200\u0202\u0200" +
                    "\u0202\u01FA\u0200\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u0204\u0204\u01FC\u01FC" +
                    "\u01FC\u0204\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u0202\u01FA\u0200\u0202\u01FA" +
                    "\u01FA\u01FA\u0200\u01FA\u01FA\u01FA\u0200\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u0202\u01FA\u0200\u01FA\u01FA" +
                    "\u0200\u0200\u0006\u01FA\u01FA\u01FA\u0200\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FC\u01FC\u01FC\u0006\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u0206\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u0208\u01FC\u01FC\u01FC" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u0206" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u0208" +
                    "\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u0206\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u0208\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u0206\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u0208\u01FC\u01FC\u01FC\u01FA\u01FA\u01FA\u01FA" +
                    "\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u01FA\u0206\u01FC\u01FC\u01FC\u01FC" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u0208\u01FC\u01FC\u01FC\u020A" +
                    "\u0006\u020C\u020C\u020C\u020C\u020C\u020E\u020E\u020E\u020E\u020E\u0210\u0210" +
                    "\u0210\u0210\u0210\u0212\u0212\u0212\u0212\u0212\u0214\u0214\u0214\u0214\u0214" +
                    "\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u00b6\u01EA\u0018\u0046\u00b6\u00b6\u00b6\u00b6" +
                    "\u00b6\u00b6\u00b6\u00b6\u01EA\u0018\u0018\u0018\u0046\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u01EA\u0216" +
                    "\u0120\u0120\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u00ba\u00b6\u00b6\u00ba\u00b6\u00b6\u00b6\u00b6\u00b6" +
                    "\u00b6\u00b6\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u01FC\u01FC\u01FC\u01FC\u01FC\u0218" +
                    "\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FC\u01FE\u0006\u0006\u0204" +
                    "\u01FC\u01FC\u01FE\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u005a\u005a\u005a\u005c" +
                    "\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u005a\u005c\u00b4\u005a\u005a\u005a\u00b4\u005c\u005a\u005a\u005c\u0006" +
                    "\u0006\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a" +
                    "\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u007a\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u00b4\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0006\u00b6\u00b6\u00b6\u021A\u01C0\u01C0\u01C0\u0006\u006a" +
                    "\u006a\u006a\u006a\u006a\u0006\u0006\u021C\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u00b8\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u00b6\u00b6\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u021E\u0000\u0000\u0000\u0000\u0000\u01C6" +
                    "\u00b6\u00b6\u0138\u0138\u0138\u0138\u0138\u0006\u0006\u0006\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u00b6\u0220\u0222\u0222\u0222\u0222\u0224\u0006\u005e\u0000\u0000\u0000\u0004\u0000\u0000" +
                    "\u0002\u0004\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0004\u0080\u0080\u0082\u0098\u009a\u009a\u009a\u009a\u00b6" +
                    "\u00b6\u00b6\u00b8\u0006\u0006\u0006\u0006\u0226\u0226\u0226\u0226\u0226\u0226\u0226\u0226" +
                    "\u0226\u0226\u0226\u0226\u0226\u0226\u0226\u0226\u0226\u0228\u0228\u0228\u0228" +
                    "\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228\u0228" +
                    "\u00b6\u0152\u00b6\u022A\u0006\u0006\u022C\u022C\u022C\u022C\u022C\u0006\u0006\u00c0\u0006" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u022E\u0230\u0230\u0230\u0230\u0232\u0234\u0236" +
                    "\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236" +
                    "\u0236\u0236\u0236\u0236\u0238\u023A\u023A\u023A\u023A\u023C\u0236\u023E\u0240" +
                    "\u0242\u0006\u0006\u0006\u0006\u0006\u0244\u0246\u0246\u0246\u0246\u0248\u024A\u0236" +
                    "\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236\u0236" +
                    "\u0236\u0236\u024C\u024E\u024E\u024E\u024E\u0236\u0236\u0236\u0006\u00e4\u00e4\u0250" +
                    "\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u0250\u00fe\u00fe\u0250\u0250" +
                    "\u00e4\u00e4\u00e4\u00e4\u00fe\u00e4\u00e4\u0250\u0250\u0006\u0006\u0006\u00fe\u0006\u0250\u0250\u0250" +
                    "\u0250\u00e4\u0250\u00fe\u00fe\u0250\u0250\u0250\u0250\u0250\u0250\u00fe\u00fe\u0250" +
                    "\u00e4\u00fe\u00e4\u00e4\u00e4\u00fe\u00e4\u00e4\u0250\u00e4\u00fe\u00fe\u00e4\u00e4\u00e4\u00e4\u00e4\u0250" +
                    "\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u0006\u0006\u0250\u00e4\u0250\u00e4\u00e4\u0250\u00e4" +
                    "\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u00e4\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0252" +
                    "\u0006\u0006\u0006\u0006\u0006\u0006\u0006\u0254\u0254\u0256\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0258\u0258\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u025A\u025C" +
                    "\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u025C\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u025E\u025C\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0254\u0258\u0258\u0258\u0258\u0258" +
                    "\u0260\u0262\u0262\u0262\u0262\u0264\u0266\u0254\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0268\u026A\u026A\u026A\u026A\u026A\u026A" +
                    "\u026A\u026A\u026A\u026A\u026A\u026A\u026A\u0018\u0018\u0018\u026A\u026A\u026A\u026A" +
                    "\u026A\u026A\u026A\u026A\u026A\u026A\u026A\u026A\u026A\u003c\u0254\u0254\u026C" +
                    "\u026A\u026A\u026A\u026A\u026A\u026A\u026C\u026A\u026A\u026A\u026A\u026A\u0018" +
                    "\u0018\u026E\u0270\u0272\u0272\u0272\u0272\u026E\u0018\u0018\u0018\u0018\u0018\u0018\u0018" +
                    "\u0018\u0268\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0274\u0274\u0274\u0274\u0274\u0274\u0274\u0274\u0274" +
                    "\u0274\u0274\u0274\u0274\u0270\u0276\u0258\u0258\u0258\u0258\u0258\u0258\u0018" +
                    "\u0018\u0018\u0018\u0018\u026E\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0018\u0270\u0018\u0272" +
                    "\u0272\u0278\u0272\u026E\u0258\u0258\u0018\u0018\u0018\u0018\u027A\u0258\u0258\u0258" +
                    "\u0272\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0254\u0254\u0254\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027E\u0254\u0280\u0280\u0280\u0280\u0282\u027C\u027C\u027C\u027C\u0282" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027E\u027C\u027C\u027C\u0284\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u0254\u0280\u0286\u0280\u0254\u0280\u027C\u0288\u028A" +
                    "\u0284\u027C\u028C\u028E\u0282\u027C\u027C\u0280\u0280\u0280\u0280\u0280\u0280" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u0256\u0286\u027E\u0286\u027C" +
                    "\u0290\u0292\u0292\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027E\u027E\u0288\u027C\u0288\u0288\u0288\u0288" +
                    "\u0288\u028A\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u0288" +
                    "\u0288\u0288\u0288\u0288\u0288\u0288\u0288\u0288\u028A\u027C\u028A\u027C\u0284" +
                    "\u0288\u0284\u0288\u027C\u027C\u027C\u0284\u0284\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u028A\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027E\u025E\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u003c\u003c\u003c\u003c" +
                    "\u0254\u0286\u0282\u027C\u0256\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u0254\u0254\u0254\u0286\u0294\u0286\u0296\u0280\u0280" +
                    "\u0298\u0254\u0254\u0254\u0254\u0254\u0286\u0254\u0280\u0280\u0254\u029A\u0254" +
                    "\u029C\u0298\u0254\u0254\u0254\u0254\u0254\u0254\u027E\u0254\u0294\u0254\u0254" +
                    "\u0254\u0286\u0294\u0254\u0254\u0254\u0254\u0294\u0254\u0254\u0280\u0294\u0254" +
                    "\u0254\u0254\u0254\u0254\u0286\u0280\u0254\u0254\u0254\u0254\u0280\u0294\u0286" +
                    "\u0286\u0254\u0254\u0294\u0254\u0254\u0286\u0254\u0286\u0254\u0254\u0254\u0282" +
                    "\u027C\u027C\u027C\u027C\u0284\u0288\u027C\u0284\u0288\u0288\u003c\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u003c\u027C\u0284\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u0288\u028A\u027C\u027C\u027C\u027C\u028A\u027C\u027C\u0254\u0254\u0286\u029E" +
                    "\u0280\u027C\u0256\u025E\u027C\u0258\u0258\u027C\u027C\u0280\u0280\u0280\u0254" +
                    "\u0286\u025E\u02A0\u0258\u0294\u0286\u027C\u027C\u027C\u027C\u02A0\u0258\u003c" +
                    "\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u0254\u025A\u0258\u025C\u0254\u0254\u003c" +
                    "\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u02A2\u0254\u0254\u0258\u0258\u0258\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u0258\u0258\u02A0\u0258\u0258\u0258\u0258\u0258" +
                    "\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u003c\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u003c\u003c\u0258\u0258\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u0258\u0258" +
                    "\u0258\u003c\u003c\u003c\u003c\u0258\u0258\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u003c\u003c" +
                    "\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u0258\u0254\u0254\u0254\u0254\u0254\u0254" +
                    "\u0258\u0258\u0254\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u003c\u028A\u0284\u027C\u027C" +
                    "\u027C\u027C\u0288\u0288\u0288\u0288\u027C\u027C\u027C\u028A\u027C\u027C\u027C" +
                    "\u027C\u0288\u0288\u0288\u0288\u0288\u02A4\u0288\u028A\u027C\u027C\u027C\u02A6" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u0284\u027C\u027C" +
                    "\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u02A8\u02A8\u0284" +
                    "\u028A\u0288\u0284\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u027C\u0284\u0288" +
                    "\u0284\u0288\u0288\u0288\u0288\u0288\u0288\u027C\u0254\u0254\u0254\u0254\u0254" +
                    "\u0254\u0254\u0258\u027C\u027C\u027C\u027C\u027C\u027C\u02A0\u0258\u027C\u027C" +
                    "\u027C\u027C\u027C\u0258\u0258\u02AA\u027C\u027C\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u0284\u0288\u02A0\u0258\u0258\u0258\u027C\u027C\u027C\u027C\u027C" +
                    "\u027C\u027C\u02A0\u02AA\u027C\u027C\u027C\u027C\u027C\u0258\u0258\u0258\u0288" +
                    "\u0288\u0288\u0288\u02AC\u0258\u0258\u0258\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c" +
                    "\u003c\u0044\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u003c\u01E0\u01E0" +
                    "\u01E0\u01E0\u01E0\u0006\u0006\u0006\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0258" +
                    "\u0258\u0258\u0258\u0258\u0258\u0258\u0258\u0006").toCharArray()

        // The A table has 686 entries for a total of 2744 bytes.
        val A: IntArray = IntArray(686)
        const val A_DATA: String =
            "\u0000\u7005\u0000\u7005\u7800\u0000\u0000\u7005\u0000\u7005\u7800\u0000\u7800\u0000\u7800" +
                    "\u0000\u0000\u0018\u6800\u0018\u0000\u0018\u7800\u0000\u7800\u0000\u0000\u074B\u0000\u074B\u0000" +
                    "\u074B\u0000\u074B\u0000\u046B\u0000\u058B\u0000\u080B\u0000\u080B\u0000\u080B\u7800\u0000" +
                    "\u0000\u001c\u0000\u001c\u0000\u001c\u6800\u780A\u6800\u780A\u6800\u77EA\u6800\u744A\u6800" +
                    "\u77AA\u6800\u742A\u6800\u780A\u6800\u76CA\u6800\u774A\u6800\u780A\u6800\u780A" +
                    "\u6800\u766A\u6800\u752A\u6800\u750A\u6800\u74EA\u6800\u74EA\u6800\u74CA\u6800" +
                    "\u74AA\u6800\u748A\u6800\u74CA\u6800\u754A\u6800\u752A\u6800\u750A\u6800\u74EA" +
                    "\u6800\u74CA\u6800\u772A\u6800\u780A\u6800\u764A\u6800\u780A\u6800\u080B\u6800" +
                    "\u080B\u6800\u080B\u6800\u080B\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u06CB\u6800" +
                    "\u080B\u6800\u001c\u0000\u001c\u0000\u001c\u7800\u0000\u6800\u001c\u7800\u0000\u0000\u001c\u4000" +
                    "\u3006\u4000\u3006\u1800\u040B\u1800\u040B\u1800\u040B\u1800\u040B\u1800\u052B" +
                    "\u1800\u064B\u1800\u080B\u1800\u080B\u1800\u080B\u0000\u042B\u0000\u048B\u0000\u050B" +
                    "\u0000\u080B\u0000\u7005\u0000\u780A\u0000\u780A\u7800\u0000\u4000\u3006\u4000\u3006" +
                    "\u4000\u3006\u7800\u0000\u7800\u0000\u0000\u0018\u0000\u0018\u0000\u760A\u0000\u760A\u0000\u76EA" +
                    "\u0000\u740A\u0000\u780A\u00a2\u7001\u00a2\u7001\u00a1\u7002\u00a1\u7002\u0000\u3409\u0000" +
                    "\u3409\u009e\u7001\u009e\u7001\u009e\u7001\u7800\u0000\u7800\u0000\u009d\u7002\u009d\u7002" +
                    "\u009d\u7002\u009d\u7002\u7800\u0000\u0000\u7004\u0000\u7004\u0000\u7004\u0000\u7004\u0000" +
                    "\u7004\u0000\u7004\u7800\u0000\u0000\u7004\u0000\u7004\u7800\u0000\u0800\u7005\u0800" +
                    "\u7005\u0800\u7005\u7800\u0000\u7800\u0000\u0800\u7005\u7800\u0000\u0800\u0018\u0800" +
                    "\u052B\u0800\u052B\u0800\u052B\u0800\u05EB\u0800\u070B\u0800\u080B\u0800\u080B" +
                    "\u0800\u080B\u0800\u7005\u0800\u001c\u0800\u001c\u0800\u050B\u0800\u050B\u0800" +
                    "\u050B\u0800\u058B\u0800\u06AB\u7800\u0000\u0800\u074B\u0800\u074B\u0800\u074B" +
                    "\u0800\u074B\u0800\u072B\u0800\u072B\u0800\u07AB\u0800\u04CB\u0800\u080B\u7800" +
                    "\u0000\u0800\u04CB\u0800\u052B\u0800\u05AB\u0800\u06CB\u0800\u080B\u0800\u056B" +
                    "\u0800\u066B\u0800\u078B\u0800\u080B\u7800\u0000\u6800\u0018\u0800\u042B\u0800" +
                    "\u042B\u0800\u054B\u0800\u066B\u0800\u7005\u4000\u3006\u7800\u0000\u4000\u3006" +
                    "\u4000\u3006\u4000\u3006\u4000\u3006\u7800\u0000\u7800\u0000\u4000\u3006\u0800" +
                    "\u04CB\u0800\u05EB\u0800\u080B\u7800\u0000\u0800\u0018\u0800\u0018\u0800\u0018\u7800" +
                    "\u0000\u0800\u7005\u0800\u048B\u0800\u080B\u0800\u0018\u0800\u001c\u0800\u7005\u0800" +
                    "\u7005\u4000\u3006\u7800\u0000\u0800\u06CB\u6800\u0018\u6800\u0018\u0800\u05CB\u0800" +
                    "\u06EB\u7800\u0000\u0800\u070B\u0800\u070B\u0800\u070B\u0800\u070B\u0800\u07AB" +
                    "\u0902\u7001\u0902\u7001\u0902\u7001\u7800\u0000\u0901\u7002\u0901\u7002\u0901" +
                    "\u7002\u7800\u0000\u0800\u04EB\u0800\u054B\u0800\u05CB\u0800\u080B\u1000\u7005" +
                    "\u1000\u7005\u3000\u3609\u3000\u3609\u3000\u3409\u3000\u3409\u0800\u7004\u0800" +
                    "\u7005\u0882\u7001\u0882\u7001\u6800\u0014\u0800\u7004\u0881\u7002\u0881\u7002" +
                    "\u0800\u0019\u0800\u0019\u3000\u042B\u3000\u042B\u3000\u054B\u3000\u066B\u3000" +
                    "\u080B\u3000\u080B\u3000\u080B\u7800\u0000\u4000\u3006\u0800\u0014\u1000\u7005" +
                    "\u7800\u0000\u4000\u3006\u4000\u3006\u0800\u048B\u0800\u048B\u0800\u050B\u0800" +
                    "\u062B\u0800\u074B\u0800\u080B\u0800\u080B\u0800\u7005\u4000\u3006\u1000\u060B" +
                    "\u1000\u070B\u1000\u042B\u1000\u080B\u1000\u0018\u1000\u0018\u1000\u0018\u0800\u7005" +
                    "\u0800\u078B\u0800\u078B\u0800\u078B\u0800\u078B\u0800\u042B\u0800\u054B\u0800" +
                    "\u080B\u0000\u3008\u4000\u3006\u0000\u3008\u0000\u7005\u4000\u3006\u0000\u0018\u0000\u0018" +
                    "\u0000\u0018\u6800\u05EB\u6800\u05EB\u6800\u070B\u6800\u042B\u0000\u3749\u0000\u3749" +
                    "\u4000\u3006\u0000\u7005\u0000\u7005\u4000\u3006\u4000\u3006\u0000\u7005\u0000\u3008" +
                    "\u0000\u3008\u4000\u3006\u0000\u3008\u0000\u3008\u4000\u3006\u0000\u0018\u0000\u1010\u7800" +
                    "\u0000\u0000\u1010\u0000\u3609\u0000\u3609\u0000\u3549\u0000\u3549\u0000\u7005\u0000\u3008" +
                    "\u0000\u7005\u4000\u3006\u0000\u3008\u0000\u7005\u0000\u7005\u0000\u0018\u0000\u0018\u4000" +
                    "\u3006\u7800\u0000\u0000\u040B\u0000\u040B\u0000\u040B\u0000\u040B\u0000\u052B\u0000\u064B" +
                    "\u0000\u080B\u0000\u080B\u7800\u0000\u4000\u3006\u0000\u3008\u4000\u3006\u4000\u3006" +
                    "\u0000\u3008\u7800\u0000\u7800\u0000\u0000\u3008\u0000\u3008\u0000\u3008\u4000\u3006\u0000" +
                    "\u3008\u0000\u0018\u0000\u7005\u4000\u3006\u0000\u0018\u6800\u0018\u7800\u0000\u0000\u3008" +
                    "\u4000\u3006\u0000\u34C9\u0000\u34C9\u0000\u060B\u0000\u072B\u0000\u0018\u0000\u001c\u0082\u7001" +
                    "\u0082\u7001\u0081\u7002\u0081\u7002\u4000\u3006\u0000\u3006\u0000\u3006\u4000\u3006" +
                    "\u0000\u3008\u0000\u3006\u0000\u04EB\u0000\u04EB\u0000\u3008\u0000\u0018\u0000\u080B\u6800" +
                    "\u001c\u6800\u001c\u2800\u601A\u2800\u601A\u2800\u601A\u2800\u601A\u6800\u001c\u0000" +
                    "\u744A\u0000\u744A\u0000\u776A\u0000\u776A\u0000\u776A\u0000\u76AA\u0000\u76AA\u0000\u76AA" +
                    "\u0000\u76AA\u0000\u758A\u0000\u758A\u0000\u758A\u0000\u746A\u0000\u746A\u0000\u746A\u0000" +
                    "\u77EA\u0000\u77EA\u0000\u77CA\u0000\u77CA\u0000\u77CA\u0000\u76AA\u0000\u768A\u0000\u768A" +
                    "\u0000\u768A\u0000\u780A\u0000\u780A\u0000\u75AA\u0000\u75AA\u0000\u75AA\u0000\u758A\u0000" +
                    "\u752A\u0000\u750A\u0000\u750A\u0000\u74EA\u0000\u74CA\u0000\u74AA\u0000\u74CA\u0000\u74CA" +
                    "\u0000\u74AA\u0000\u748A\u0000\u748A\u0000\u746A\u0000\u746A\u0000\u744A\u0000\u742A\u0000" +
                    "\u740A\u0000\u770A\u0000\u770A\u0000\u770A\u0000\u764A\u0000\u764A\u0000\u764A\u0000\u764A" +
                    "\u0000\u762A\u0000\u762A\u0000\u760A\u0000\u752A\u0000\u752A\u0000\u780A\u0000\u776A\u0000" +
                    "\u776A\u7800\u0000\u0000\u1010\u0000\u1010\u0000\u7004\u0000\u7004\u7800\u0000\u0000\u05EB" +
                    "\u0000\u7004\u0000\u7005\u0000\u7005\u0000\u7004\u0000\u7004\u0000\u0018\u0000\u05AB\u0000\u05AB" +
                    "\u0000\u05AB\u0000\u0018\u4000\u3006\u0000\u7004\u6800\u0018\u0000\u7004\u4000\u3006\u7800" +
                    "\u0000\u0000\u7005\u0000\u7005\u7800\u0000\u0000\u7005\u0000\u7005\u7800\u0000\u7800\u0000" +
                    "\u0000\u7004\u0000\u7004\u7800\u0000\u4800\u1010\u4800\u1010\u1800\u3609\u1800\u3609" +
                    "\u0000\u001c\u0000\u3008\u0000\u3008\u0000\u3008\u0000\u3008\u4800\u1010\u4800\u1010\u4000" +
                    "\u3006\u4000\u3006\u0000\u001c\u0000\u001c\u6800\u001c\u4000\u3006\u6800\u001c\u0000\u042B" +
                    "\u0000\u042B\u0000\u054B\u0000\u066B\u0000\u05EB\u0000\u05EB\u0000\u05EB\u0000\u054B\u0000" +
                    "\u05AB\u7800\u0000\u0000\u7001\u0000\u7001\u0000\u7002\u0000\u7002\u0000\u7002\u7800\u0000" +
                    "\u0000\u7001\u7800\u0000\u7800\u0000\u0000\u7001\u7800\u0000\u0000\u7002\u0000\u7001\u6800" +
                    "\u0019\u0000\u7002\uE800\u0019\u0000\u7001\u0000\u7002\u1800\u3649\u1800\u3649\u1800" +
                    "\u3509\u1800\u3509\u1800\u37C9\u1800\u37C9\u1800\u3689\u1800\u3689\u1800\u3549" +
                    "\u1800\u3549\u0000\u001c\u0000\u0018\u0000\u7005\u0000\u7002\u4000\u3006\u0000\u7004\u0000" +
                    "\u7005\u0000\u001c\u7800\u0000\u2800\u601A\u0000\u7005\u0000\u35E9\u0000\u35E9\u0000\u35E9" +
                    "\u0000\u35E9\u7800\u0000\u088A\u7001\u088A\u7001\u0889\u7002\u0889\u7002\u4000" +
                    "\u3006\u0800\u7004\u0800\u3609\u0800\u3609\u7800\u0000\u1000\u060B\u1000\u060B" +
                    "\u1000\u060B\u1000\u060B\u1000\u072B\u1000\u044B\u1000\u080B\u1000\u080B\u1000" +
                    "\u080B\u1000\u080B\u1000\u07CB\u1000\u07CB\u1000\u07CB\u1000\u001c\u1000\u080B" +
                    "\u1000\u601A\u1000\u060B\u1000\u060B\u1000\u080B\u1000\u080B\u7800\u0000\u7800" +
                    "\u0000\u1000\u040B\u1000\u040B\u1000\u040B\u1000\u040B\u1000\u052B\u1000\u064B" +
                    "\u1000\u080B\u1000\u001c\u1000\u066B\u1000\u066B\u1000\u066B\u7800\u0000\u1000" +
                    "\u7005\u6800\u0019\u6800\u0019\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u7800\u0000" +
                    "\u7800\u0000\u6800\u001c\u7800\u0000\u7800\u0000\u6800\u001c\u6800\u001c\u6800\u001c\u1800" +
                    "\u040B\u1800\u07EB\u1800\u07EB\u1800\u07EB\u1800\u07EB\u6800\u06AB\u6800\u068B" +
                    "\u6800\u001c\u0000\u001c\u6800\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c" +
                    "\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u0000\u001c\u7800\u0000\u0000\u001c" +
                    "\u0000\u001c\u0000\u001c\u7800\u0000\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c" +
                    "\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800" +
                    "\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c" +
                    "\u6800\u001c\u6800\u001b\u6800\u001b\u6800\u001b\u6800\u001c\u6800\u001c\u6800\u001c\u6800" +
                    "\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c" +
                    "\u6800\u001c\u6800\u001c\u7800\u0000\u6800\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u6800" +
                    "\u001c\u6800\u001c\u6800\u001c\u6800\u001c\u7800\u0000\u6800\u001c\u6800\u001c\u7800\u0000"

        // The B table has 686 entries for a total of 1372 bytes.
        val B: CharArray =
            ("\u0030\u0030\u0000\u0030\u0030\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0020\u0020\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0030\u0030\u0000\u0024\u0024\u0024\u0000\u0000" +
                    "\u0000\u0000\u0030\u0030\u0030\u0030\u0030\u0032\u0032\u0031\u0031\u0020\u0020\u0032\u0032\u0032\u0000\u0000\u0031" +
                    "\u0031\u0031\u0031\u0000\u0031\u0030\u0030\u0031\u0031\u0031\u0000\u0031\u0031\u0000\u0030\u0030\u0030\u0000\u0000" +
                    "\u0030\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0024\u0000\u0024\u0020\u0020\u0020\u0000\u0000\u0020\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0000\u0000\u0000\u0000\u0030\u0030\u0020\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0032\u0032\u0032\u0000\u0031\u0031\u0031\u0000\u0000\u0000\u0000\u0000" +
                    "\u0030\u0030\u0020\u0020\u0020\u0020\u0030\u0030\u0032\u0032\u0000\u0030\u0031\u0031\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0024\u0000\u0030\u0000\u0024\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030" +
                    "\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0030\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0024\u0024\u0024" +
                    "\u0030\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0020\u0020\u0020\u0030\u0030\u0024\u0024\u0030\u0024\u0024" +
                    "\u0024\u0024\u0024\u0020\u0000\u0000\u0000\u0000\u0020\u0020\u0020\u0020\u0030\u0024\u0030\u0020\u0020\u0030\u0030" +
                    "\u0000\u0000\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0024\u0020\u0020\u0024\u0024\u0000" +
                    "\u0000\u0024\u0024\u0020\u0020\u0020\u0000\u0030\u0024\u0000\u0000\u0000\u0020\u0020\u0020\u0020\u0000\u0000\u0000" +
                    "\u0000\u0032\u0032\u0031\u0031\u0024\u0024\u0024\u0024\u0024\u0020\u0000\u0000\u0024\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030" +
                    "\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0030\u0000\u0000\u0000\u0030\u0030\u0000\u0000\u0030\u0030\u0030\u0030" +
                    "\u0030\u0000\u0000\u0000\u0000\u0000\u0024\u0030\u0000\u0030\u0028\u0000\u0038\u0038\u0000\u0038\u0038\u0000\u0000" +
                    "\u0030\u0030\u0000\u0000\u0000\u0020\u0020\u0000\u0020\u0020\u0020\u0020\u0000\u0000\u0020\u0020\u0000\u0000\u0000" +
                    "\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0032\u0032\u0031\u0031\u0031\u0000\u0032" +
                    "\u0000\u0000\u0032\u0000\u0031\u0032\u0000\u0031\u0000\u0032\u0031\u0020\u0020\u0020\u0020\u0020\u0020\u0020\u0020" +
                    "\u0020\u0020\u0000\u0000\u0030\u0031\u0020\u0030\u0030\u0000\u0000\u0000\u0030\u0020\u0020\u0020\u0020\u0000\u0032" +
                    "\u0032\u0031\u0031\u0020\u0030\u0020\u0020\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0000" +
                    "\u0000\u0000\u0000\u0000\u0030\u0000\u0000\u0800\u0800\u08C0\u0800\u0800\u0800\u0800\u0800" +
                    "\u0800\u0800\u0800\u08C0\u0000\u0000\u0000\u0000\u0000\u0000\u0000\u0800\u0000\u0800\u0006\u0006" +
                    "\u0846\u0846\u08C0\u0000\u0000\u08C0\u08C0\u08C0\u04C0\u04C0\u0840\u0800\u08C0" +
                    "\u0840\u0000\u0800\u08C0\u08C0\u08C0\u0840\u0840\u0840\u0840\u08C0\u08C0\u0AC0" +
                    "\u0800\u0840\u0AC0\u0AC0\u0AC0\u08C0\u0AC0\u0A40\u0A40\u0840\u08C0\u05C0\u05C0" +
                    "\u05C0\u0840\u0800\u0A40\u0A40\u0AC0\u0800\u0A40\u0800\u0800\u0AC0\u0AC0\u0840" +
                    "\u08C0\u0800\u0000\u0800\u08C0\u0000\u0000\u08C0\u0CC0\u0CC0\u0800\u08C0\u0AC0\u0800").toCharArray()

        // In all, the character property tables require 18104 bytes.
        init {
            run {
                // THIS CODE WAS AUTOMATICALLY CREATED BY GenerateCharacter:
                val data = Companion.A_DATA.toCharArray()
                assert(data.size == (686 * 2))
                var i = 0
                var j = 0
                while (i < (686 * 2)) {
                    val entry = data[i++].code shl 16
                    Companion.A[j++] = entry or data[i++].code
                }
            }
        }
    }
}
