package org.gnit.lucenekmp.search

import okio.IOException
import org.gnit.lucenekmp.util.MathUtil
import kotlin.jvm.JvmName


/**
 * The Scorer for DisjunctionMaxQuery. The union of all documents generated by the subquery scorers
 * is generated in document number order. The score for each document is the maximum of the scores
 * computed by the subquery scorers that generate that document, plus tieBreakerMultiplier times the
 * sum of the scores for the other subqueries that generate the document.
 */
internal class DisjunctionMaxScorer(/* Multiplier applied to non-maximum-scoring subqueries for a document as they are summed into the result. */
                                    private val tieBreakerMultiplier: Float,
                                    private val subScorers: MutableList<Scorer>,
                                    scoreMode: ScoreMode?,
                                    leadCost: Long
) : DisjunctionScorer(
    subScorers, scoreMode, leadCost
) {
    private var disjunctionBlockPropagator: DisjunctionScoreBlockBoundaryPropagator? = null

    /**
     * Creates a new instance of DisjunctionMaxScorer
     *
     * @param tieBreakerMultiplier Multiplier applied to non-maximum-scoring subqueries for a document
     * as they are summed into the result.
     * @param subScorers The sub scorers this Scorer should iterate on
     */
    init {
        require(!(tieBreakerMultiplier < 0 || tieBreakerMultiplier > 1)) { "tieBreakerMultiplier must be in [0, 1]" }
        if (scoreMode === ScoreMode.TOP_SCORES) {
            this.disjunctionBlockPropagator = DisjunctionScoreBlockBoundaryPropagator(subScorers)
        } else {
            this.disjunctionBlockPropagator = null
        }
    }

    @Throws(IOException::class)
    override fun score(topList: DisiWrapper?): Float {
        var scoreMax = 0f
        var otherScoreSum = 0.0
        var w: DisiWrapper? = topList
        while (w != null) {
            val subScore: Float = w.scorable!!.score()
            if (subScore >= scoreMax) {
                otherScoreSum += scoreMax.toDouble()
                scoreMax = subScore
            } else {
                otherScoreSum += subScore.toDouble()
            }
            w = w.next
        }
        return (scoreMax + otherScoreSum * tieBreakerMultiplier).toFloat()
    }

    @Throws(IOException::class)
    override fun advanceShallow(target: Int): Int {
        if (disjunctionBlockPropagator != null) {
            return disjunctionBlockPropagator!!.advanceShallow(target)
        }
        return super.advanceShallow(target)
    }

    @Throws(IOException::class)
    override fun getMaxScore(upTo: Int): Float {
        var scoreMax = 0f
        var otherScoreSum = 0.0
        for (scorer in subScorers) {
            if (scorer.docID() <= upTo) {
                val subScore = scorer.getMaxScore(upTo)
                if (subScore >= scoreMax) {
                    otherScoreSum += scoreMax.toDouble()
                    scoreMax = subScore
                } else {
                    otherScoreSum += subScore.toDouble()
                }
            }
        }

        if (tieBreakerMultiplier == 0f) {
            return scoreMax
        } else {
            // The error of sums depends on the order in which values are summed up. In
            // order to avoid this issue, we compute an upper bound of the value that
            // the sum may take. If the max relative error is b, then it means that two
            // sums are always within 2*b of each other.
            otherScoreSum *= (1 + 2 * MathUtil.sumRelativeErrorBound(subScorers.size - 1))
            return (scoreMax + otherScoreSum * tieBreakerMultiplier).toFloat()
        }
    }

    @JvmName("setMinCompetitiveScoreKt")
    @Throws(IOException::class)
    fun setMinCompetitiveScore(minScore: Float) {
        if (disjunctionBlockPropagator != null) {
            disjunctionBlockPropagator!!.setMinCompetitiveScore(minScore)
        }
        if (tieBreakerMultiplier == 0f) {
            // TODO: we could even remove some scorers from the priority queue?
            for (scorer in subScorers) {
                scorer.minCompetitiveScore = minScore
            }
        }
    }
}
