/*
 * Decompiled with CFR 0.152.
 */
package org.gorpipe.gor.manager;

import java.io.File;
import java.io.IOException;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.RandomStringUtils;
import org.gorpipe.exceptions.GorSystemException;
import org.gorpipe.gor.manager.BucketCreator;
import org.gorpipe.gor.manager.BucketCreatorGorPipe;
import org.gorpipe.gor.table.BaseTable;
import org.gorpipe.gor.table.BucketableTableEntry;
import org.gorpipe.gor.table.PathUtils;
import org.gorpipe.gor.table.dictionary.DictionaryTable;
import org.gorpipe.gor.table.lock.ExclusiveFileTableLock;
import org.gorpipe.gor.table.lock.TableLock;
import org.gorpipe.gor.table.lock.TableTransaction;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BucketManager<T extends BucketableTableEntry> {
    private static final Logger log = LoggerFactory.getLogger(BucketManager.class);
    public static final int DEFAULT_MIN_BUCKET_SIZE = 20;
    public static final int DEFAULT_BUCKET_SIZE = 100;
    public static final int DEFAULT_MAX_BUCKET_COUNT = 3;
    public static final BucketPackLevel DEFAULT_BUCKET_PACK_LEVEL = BucketPackLevel.CONSOLIDATE;
    static final String BUCKET_FILE_PREFIX = "bucket";
    public static final Duration DEFAULT_LOCK_TIMEOUT = Duration.ofMinutes(30L);
    public static final Class<? extends TableLock> DEFAULT_LOCK_TYPE = ExclusiveFileTableLock.class;
    public static final String HEADER_MIN_BUCKET_SIZE_KEY = "GOR_TABLE_MIN_BUCKET_SIZE";
    public static final String HEADER_BUCKET_SIZE_KEY = "GOR_TABLE_BUCKET_SIZE";
    public static final String HEADER_BUCKET_DIRS_KEY = "GOR_TABLE_BUCKET_DIRS";
    public static final String HEADER_BUCKET_MAX_BUCKETS = "GOR_TABLE_BUCKET_MAX_BUCKETS";
    protected Duration gracePeriodForDeletingBuckets = Duration.ofHours(24L);
    private final List<Path> bucketDirs = new ArrayList<Path>();
    private Map<Path, Long> bucketDirCount;
    private Class<? extends TableLock> lockType = DEFAULT_LOCK_TYPE;
    private Duration lockTimeout = DEFAULT_LOCK_TIMEOUT;
    private final BaseTable<T> table;
    private int bucketSize;
    private int minBucketSize;
    private BucketCreator<T> bucketCreator;

    public BucketManager(BaseTable<T> table) {
        this.table = table;
        this.bucketCreator = new BucketCreatorGorPipe();
        this.setBucketSize(Integer.parseInt(table.getConfigTableProperty(HEADER_BUCKET_SIZE_KEY, Integer.toString(100))));
        this.setMinBucketSize(Integer.parseInt(table.getConfigTableProperty(HEADER_MIN_BUCKET_SIZE_KEY, Integer.toString(20))));
        this.setBucketDirs(this.parseBucketDirString(table.getConfigTableProperty(HEADER_BUCKET_DIRS_KEY, null)));
    }

    private BucketManager(Builder builder) {
        this(builder.table);
        this.lockType = builder.lockType != null ? builder.lockType : DEFAULT_LOCK_TYPE;
        Duration duration = this.lockTimeout = builder.lockTimeout != null ? builder.lockTimeout : DEFAULT_LOCK_TIMEOUT;
        if (builder.bucketSize > 0) {
            this.setBucketSize(builder.bucketSize);
        }
        if (builder.minBucketSize > 0) {
            this.setMinBucketSize(builder.minBucketSize);
        }
        if (builder.bucketCreator != null) {
            this.bucketCreator = builder.bucketCreator;
        }
    }

    public static Builder newBuilder(BaseTable table) {
        return new Builder(table);
    }

    public void bucketize() {
        this.bucketize(DEFAULT_BUCKET_PACK_LEVEL, -1, null, false);
    }

    public int bucketize(BucketPackLevel packLevel, int maxBucketCount) {
        return this.bucketize(packLevel, maxBucketCount, null, false);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int bucketize(BucketPackLevel packLevel, int maxBucketCount, List<Path> bucketDirs, boolean forceClean) {
        if (!this.table.isBucketize()) {
            log.info("Bucketize - Bucketize called on {} but as the table is marked not bucketize so nothing was done.", (Object)this.table.getPath());
            return 0;
        }
        if (bucketDirs != null && !bucketDirs.isEmpty()) {
            this.setBucketDirs(bucketDirs);
        }
        try (TableLock bucketizeLock = TableLock.acquireWrite(this.lockType, this.table, (String)"bucketize", (Duration)Duration.ofMillis(1000L));){
            if (!bucketizeLock.isValid()) {
                long millisRunning = System.currentTimeMillis() - bucketizeLock.lastModified();
                log.debug("Bucketize - Bucketization already in progress on {} (has been running for {} seconds)", (Object)this.table.getName(), (Object)(millisRunning / 1000L));
                int n2 = 0;
                return n2;
            }
            if (maxBucketCount <= 0) {
                maxBucketCount = Integer.parseInt(this.table.getConfigTableProperty(HEADER_BUCKET_MAX_BUCKETS, Integer.toString(3)));
            }
            this.cleanTempBucketFolders(bucketizeLock);
            this.cleanOldBucketFiles(bucketizeLock, forceClean);
            int n = this.doBucketize(bucketizeLock, packLevel, maxBucketCount);
            return n;
        }
        catch (IOException e) {
            throw new GorSystemException((Throwable)e);
        }
    }

    public void deleteBuckets(Path ... buckets) {
        this.deleteBuckets(false, buckets);
    }

    public void deleteBuckets(boolean force, Path ... buckets) {
        try (TableTransaction trans = TableTransaction.openWriteTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);){
            this.deleteBuckets(trans.getLock(), force, buckets);
            trans.commit();
        }
        catch (IOException e) {
            throw new GorSystemException((Throwable)e);
        }
    }

    protected int getEffectiveMinBucketSize() {
        return Math.min(this.getMinBucketSize(), this.getBucketSize());
    }

    public int getBucketSize() {
        return this.bucketSize;
    }

    public void setBucketSize(int bucketSize) {
        this.bucketSize = bucketSize;
    }

    public int getMinBucketSize() {
        return this.minBucketSize;
    }

    public void setMinBucketSize(int minBucketSize) {
        this.minBucketSize = minBucketSize;
    }

    protected Path getDefaultBucketDir() {
        return Paths.get("." + this.table.getName(), "buckets");
    }

    public Duration getLockTimeout() {
        return this.lockTimeout;
    }

    private List<Path> getBucketDirs() {
        return this.bucketDirs;
    }

    public void setBucketDirs(List<Path> newBucketDirs) {
        this.bucketDirs.clear();
        if (newBucketDirs != null && newBucketDirs.size() > 0) {
            for (Path p : newBucketDirs) {
                this.bucketDirs.add(PathUtils.relativize((Path)this.table.getRootPath(), (Path)p));
            }
        } else {
            this.bucketDirs.add(this.getDefaultBucketDir());
        }
    }

    private List<Path> parseBucketDirString(String bucketDirs) {
        if (bucketDirs == null) {
            return new ArrayList<Path>();
        }
        return Arrays.stream(bucketDirs.split(",")).filter(s -> !s.trim().isEmpty()).map(s -> Paths.get(s, new String[0])).collect(Collectors.toList());
    }

    private void checkBucketDirExistance(Path bucketDir) {
        if (!Files.exists(bucketDir, new LinkOption[0])) {
            Path fullPathDefaultBucketDir = PathUtils.resolve((Path)this.table.getRootPath(), (Path)this.getDefaultBucketDir());
            if (bucketDir.equals(fullPathDefaultBucketDir)) {
                try {
                    Files.createDirectories(fullPathDefaultBucketDir, new FileAttribute[0]);
                }
                catch (IOException e) {
                    throw new GorSystemException("Could not create default bucket dir: " + fullPathDefaultBucketDir, (Throwable)e);
                }
            } else {
                throw new GorSystemException(String.format("Bucket dirs must exists, directory %s is not found!", bucketDir), null);
            }
        }
    }

    protected final Path pickBucketDir() {
        if (this.bucketDirs.size() == 0) {
            throw new GorSystemException("Can not pick bucket, the list of bucket dirs is empty!", null);
        }
        String strategy = this.table.getConfigTableProperty("gor.table.buckets.directory.strategy", "least_used");
        if ("random".equals(strategy)) {
            return this.bucketDirs.get(new Random().nextInt(this.bucketDirs.size()));
        }
        if (this.bucketDirCount == null) {
            this.bucketDirCount = this.table.getEntries().stream().filter(l -> l.getBucket() != null).map(l -> Paths.get(l.getBucket(), new String[0]).getParent()).filter(p -> this.bucketDirs.contains(p)).collect(Collectors.groupingByConcurrent(Function.identity(), Collectors.counting()));
            for (Path bucketDir : this.bucketDirs) {
                if (this.bucketDirCount.containsKey(bucketDir)) continue;
                this.bucketDirCount.put(bucketDir, 0L);
            }
        }
        Map.Entry<Path, Long> minEntry = null;
        for (Map.Entry<Path, Long> entry : this.bucketDirCount.entrySet()) {
            if (minEntry != null && entry.getValue().compareTo(minEntry.getValue()) >= 0) continue;
            minEntry = entry;
        }
        minEntry.setValue((Long)minEntry.getValue() + 1L);
        return minEntry.getKey();
    }

    private int doBucketize(TableLock bucketizeLock, BucketPackLevel packLevel, int maxBucketCount) throws IOException {
        BaseTable tempTable;
        Map<Path, List<T>> newBucketsMap;
        Collection<Path> bucketsToDelete;
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketize - Bucketization already in progress");
            return 0;
        }
        try (Object trans = TableTransaction.openReadTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);){
            int unbucketizedCount = this.table.needsBucketizing().size();
            if (packLevel == BucketPackLevel.NO_PACKING && unbucketizedCount < this.getEffectiveMinBucketSize()) {
                log.debug("Bucketize - Nothing to bucketize, aborting {} unbucketized but {} is minimum.", (Object)unbucketizedCount, (Object)this.getEffectiveMinBucketSize());
                int n = 0;
                return n;
            }
            bucketsToDelete = this.findBucketsToDelete(trans.getLock(), packLevel, unbucketizedCount);
            newBucketsMap = this.findBucketsToCreate(trans.getLock(), bucketsToDelete, maxBucketCount);
            if (log.isDebugEnabled()) {
                log.debug("Bucketize - Bucketizing {} files into {} buckets", (Object)newBucketsMap.values().stream().map(List::size).mapToInt(Integer::intValue).sum(), (Object)newBucketsMap.keySet().size());
            }
            tempTable = this.createTempTable(trans.getLock());
        }
        for (Path bucketDir : this.getBucketDirs()) {
            this.doBucketizeForBucketDir(tempTable, bucketDir, newBucketsMap);
        }
        log.trace("Deleting temp table {}", (Object)tempTable.getPath());
        FileUtils.deleteDirectory((File)tempTable.getFolderPath().toFile());
        Files.deleteIfExists(tempTable.getPath());
        if (bucketsToDelete.size() > 0) {
            trans = TableTransaction.openWriteTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);
            try {
                this.deleteBuckets(trans.getLock(), false, bucketsToDelete.toArray(new Path[bucketsToDelete.size()]));
                trans.commit();
            }
            finally {
                if (trans != null) {
                    trans.close();
                }
            }
        }
        return newBucketsMap.size();
    }

    private void doBucketizeForBucketDir(BaseTable tempTable, Path bucketDir, Map<Path, List<T>> newBucketsMap) throws IOException {
        Map<Path, List<T>> newBucketsMapForBucketDir = newBucketsMap.keySet().stream().filter(p -> p.getParent().equals(bucketDir)).collect(Collectors.toMap(Function.identity(), newBucketsMap::get));
        this.createBucketFiles(tempTable, newBucketsMapForBucketDir, PathUtils.resolve((Path)this.table.getRootPath(), (Path)bucketDir));
        for (Path bucket : newBucketsMapForBucketDir.keySet()) {
            List<T> bucketEntries = newBucketsMapForBucketDir.get(bucket);
            this.updateTableWithNewBucket(this.table, bucket, bucketEntries);
        }
    }

    private void updateTableWithNewBucket(BaseTable table, Path bucket, List<T> bucketEntries) {
        try (TableTransaction trans = TableTransaction.openWriteTransaction(this.lockType, (BaseTable)table, (String)table.getName(), (Duration)this.lockTimeout);){
            table.removeFromBucket(bucketEntries);
            table.addToBucket(bucket, bucketEntries);
            table.setProperty(HEADER_BUCKET_SIZE_KEY, Integer.toString(this.getBucketSize()));
            table.setProperty(HEADER_MIN_BUCKET_SIZE_KEY, Integer.toString(this.getMinBucketSize()));
            table.setProperty(HEADER_BUCKET_DIRS_KEY, this.bucketDirs.stream().map(p -> p.toString()).collect(Collectors.joining(",")));
            trans.commit();
        }
    }

    private BaseTable createTempTable(TableLock lock) throws IOException {
        lock.assertValid();
        String ext = FilenameUtils.getExtension((String)this.table.getPath().toString());
        Path tempTablePath = this.table.getRootPath().resolve("." + this.table.getName() + "." + RandomStringUtils.random((int)8, (boolean)true, (boolean)true) + (String)(ext.length() > 0 ? "." + ext : ""));
        log.trace("Creating temp table {}", (Object)tempTablePath);
        Files.copy(this.table.getPath(), tempTablePath, new CopyOption[0]);
        return this.initTempTable(tempTablePath);
    }

    private BaseTable initTempTable(Path path) {
        if (path.toString().toLowerCase().endsWith(".gord")) {
            return ((DictionaryTable.Builder)((DictionaryTable.Builder)((DictionaryTable.Builder)((DictionaryTable.Builder)new DictionaryTable.Builder(path).useHistory(this.table.isUseHistory())).sourceColumn(this.table.getSourceColumn())).securityContext(this.table.getSecurityContext())).validateFiles(this.table.isValidateFiles())).build();
        }
        throw new GorSystemException("BaseTable of type " + path.toString() + " are not supported!", null);
    }

    private void deleteBuckets(TableLock lock, boolean force, Path ... buckets) throws IOException {
        lock.assertValid();
        this.deleteBucketFiles(force, buckets);
        this.table.removeFromBucket((Collection)this.table.filter().buckets(buckets).includeDeleted().get());
    }

    private void deleteBucketFiles(boolean force, Path ... buckets) throws IOException {
        for (Path bucket : buckets) {
            Path bucketFile = PathUtils.resolve((Path)this.table.getRootPath(), (Path)bucket);
            if (!Files.exists(bucketFile, new LinkOption[0])) continue;
            long lastAccessTime = Files.readAttributes(bucketFile, BasicFileAttributes.class, new LinkOption[0]).lastAccessTime().toMillis();
            log.trace("Checking bucket file CTM {} LAT {} GPFDB {}", new Object[]{System.currentTimeMillis(), lastAccessTime, this.gracePeriodForDeletingBuckets.toMillis()});
            if (!force && System.currentTimeMillis() - lastAccessTime <= this.gracePeriodForDeletingBuckets.toMillis()) continue;
            log.debug("Deleting bucket file {}", (Object)bucketFile);
            Files.delete(bucketFile);
        }
    }

    protected void cleanOldBucketFiles(TableLock bucketizeLock, boolean force) throws IOException {
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketization in progress, will skip cleaning bucket files.");
            return;
        }
        HashSet<Path> allBucketDirs = new HashSet<Path>(this.getBucketDirs());
        allBucketDirs.addAll(this.table.getBuckets().stream().map(b -> b.getParent()).collect(Collectors.toSet()));
        for (Path bucketDir : allBucketDirs) {
            Path fullPathBucketDir = PathUtils.resolve((Path)this.table.getRootPath(), (Path)bucketDir);
            if (!Files.exists(fullPathBucketDir, new LinkOption[0])) {
                log.debug("Bucket folder {} never been used, nothing to clean.", (Object)fullPathBucketDir);
                continue;
            }
            List<Path> bucketsToClean = this.collectBucketsToClean(fullPathBucketDir, force);
            if (bucketsToClean.size() <= 0) continue;
            this.deleteBuckets(force, bucketsToClean.toArray(new Path[bucketsToClean.size()]));
            for (Path bucket : bucketsToClean) {
                log.warn("Bucket '{}' removed as it is not used", (Object)bucket);
            }
        }
    }

    private List<Path> collectBucketsToClean(Path fullPathBucketDir, boolean force) throws IOException {
        ArrayList<Path> bucketsToDelete = new ArrayList<Path>();
        try (Stream<Path> pathList = Files.list(fullPathBucketDir);){
            for (Path f : pathList.collect(Collectors.toList())) {
                Path bucketFile = fullPathBucketDir.resolve(f);
                long lastAccessTime = Files.readAttributes(bucketFile, BasicFileAttributes.class, new LinkOption[0]).lastAccessTime().toMillis();
                log.trace("Checking bucket file CTM {} LAT {} GPFDB {}", new Object[]{System.currentTimeMillis(), lastAccessTime, this.gracePeriodForDeletingBuckets.toMillis()});
                if (!bucketFile.getFileName().toString().startsWith(this.getBucketFilePrefix(this.table)) || !bucketFile.getFileName().toString().endsWith(".gorz") || System.currentTimeMillis() - lastAccessTime <= this.gracePeriodForDeletingBuckets.toMillis() && !force || this.table.filter().buckets(new Path[]{bucketFile}).get().size() != 0) continue;
                bucketsToDelete.add(bucketFile);
            }
        }
        return bucketsToDelete;
    }

    void cleanTempBucketFolders(TableLock bucketizeLock) {
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketization in progress, will skip cleaning bucket files.");
            return;
        }
        for (Path bucketDir : this.getBucketDirs()) {
            Path fullPathBucketDir = PathUtils.resolve((Path)this.table.getRootPath(), (Path)bucketDir);
            if (!Files.exists(fullPathBucketDir, new LinkOption[0])) {
                log.debug("Bucket folder {} never been used, nothing to clean.", (Object)fullPathBucketDir);
                continue;
            }
            try {
                Stream<Path> pathList = Files.list(fullPathBucketDir);
                try {
                    for (Path candTempDir : pathList.collect(Collectors.toList())) {
                        BucketCreatorGorPipe.deleteIfTempBucketizingFolder(candTempDir, this.table);
                    }
                }
                finally {
                    if (pathList == null) continue;
                    pathList.close();
                }
            }
            catch (IOException ioe) {
                log.warn("Got exception when trying to clean up temp folders.  Just logging out the exception", (Throwable)ioe);
            }
        }
    }

    private String getBucketFilePrefix(BaseTable table) {
        return table.getName() + "_bucket";
    }

    private Collection<Path> findBucketsToDelete(TableLock lock, BucketPackLevel packLevel, int unbucketizedCount) {
        lock.assertValid();
        HashMap bucketCounts = new HashMap();
        this.table.selectAll().stream().filter(l -> l.hasBucket()).forEach(l -> {
            Path b = Paths.get(l.getBucket(), new String[0]);
            bucketCounts.put(b, bucketCounts.getOrDefault(b, 0) + (!l.isDeleted() ? 1 : 0));
        });
        HashSet<Path> bucketsToDelete = new HashSet<Path>(bucketCounts.keySet().stream().filter(k -> (Integer)bucketCounts.get(k) == 0).collect(Collectors.toSet()));
        if (packLevel == BucketPackLevel.NO_PACKING) {
            return bucketsToDelete;
        }
        if (packLevel == BucketPackLevel.FULL_PACKING) {
            bucketsToDelete.addAll(bucketCounts.keySet().stream().filter(k -> (Integer)bucketCounts.get(k) < this.getBucketSize()).collect(Collectors.toSet()));
        } else if (packLevel == BucketPackLevel.CONSOLIDATE) {
            int totalFilesNeeding = unbucketizedCount + bucketCounts.values().stream().filter(i -> i < this.getBucketSize()).mapToInt(Integer::intValue).sum();
            int totalNewBuckets = totalFilesNeeding / this.getBucketSize();
            int totalSpaceLeftInNewBuckets = totalNewBuckets * this.getBucketSize() - unbucketizedCount;
            for (Map.Entry entry : bucketCounts.entrySet().stream().filter(e -> (Integer)e.getValue() < this.getBucketSize()).sorted(Map.Entry.comparingByValue()).collect(Collectors.toList())) {
                if (totalSpaceLeftInNewBuckets <= 0) break;
                bucketsToDelete.add((Path)entry.getKey());
                totalSpaceLeftInNewBuckets -= ((Integer)entry.getValue()).intValue();
            }
        }
        return bucketsToDelete;
    }

    private Map<Path, List<T>> findBucketsToCreate(TableLock lock, Collection<Path> bucketsToDelete, int maxBucketCount) {
        lock.assertValid();
        List relBucketsToDelete = bucketsToDelete != null ? bucketsToDelete.stream().map(b -> PathUtils.resolve((Path)this.table.getRootPath(), (Path)b)).collect(Collectors.toList()) : null;
        List lines2bucketize = this.table.selectAll().stream().filter(l -> !l.hasBucket() || !l.isDeleted() && relBucketsToDelete != null && relBucketsToDelete.contains(Paths.get(l.getBucketReal(), new String[0]))).collect(Collectors.toList());
        int bucketCreateCount = (int)Math.ceil((double)lines2bucketize.size() / (double)this.getBucketSize());
        if (lines2bucketize.size() - (bucketCreateCount - 1) * this.getBucketSize() < this.getEffectiveMinBucketSize()) {
            --bucketCreateCount;
        }
        String bucketNamePrefix = String.format("%s_%s_%s_", this.getBucketFilePrefix(this.table), new SimpleDateFormat("yyyy_MMdd_HHmmss").format(new Date()), RandomStringUtils.random((int)8, (boolean)true, (boolean)true));
        HashMap<Path, List<T>> bucketsToCreate = new HashMap<Path, List<T>>();
        this.bucketDirCount = null;
        for (int i = 1; i <= Math.min(bucketCreateCount, maxBucketCount > 0 ? maxBucketCount : Integer.MAX_VALUE); ++i) {
            Path bucketDir = this.pickBucketDir();
            int nextToBeAddedIndex = (i - 1) * this.getBucketSize();
            int nextBucketSize = Math.min(this.getBucketSize(), lines2bucketize.size() - nextToBeAddedIndex);
            bucketsToCreate.put(Paths.get(bucketDir.resolve(bucketNamePrefix).toString() + i + ".gorz", new String[0]), lines2bucketize.subList(nextToBeAddedIndex, nextToBeAddedIndex + nextBucketSize));
        }
        return bucketsToCreate;
    }

    private void createBucketFiles(BaseTable table, Map<Path, List<T>> bucketsToCreate, Path absBucketDir) throws IOException {
        this.checkBucketDirExistance(absBucketDir);
        this.bucketCreator.createBuckets(table, bucketsToCreate, absBucketDir);
    }

    public static final class Builder<T extends BucketableTableEntry> {
        private final BaseTable<T> table;
        private Duration lockTimeout = null;
        private Class<? extends TableLock> lockType = null;
        private int minBucketSize = -1;
        private int bucketSize = -1;
        private BucketCreator<T> bucketCreator = null;

        private Builder(BaseTable<T> table) {
            this.table = table;
        }

        public Builder lockType(Class val) {
            this.lockType = val;
            return this;
        }

        public Builder lockTimeout(Duration val) {
            this.lockTimeout = val;
            return this;
        }

        public Builder minBucketSize(int val) {
            this.minBucketSize = val;
            return this;
        }

        public Builder bucketSize(int val) {
            this.bucketSize = val;
            return this;
        }

        public Builder bucketCreator(BucketCreator<T> val) {
            this.bucketCreator = val;
            return this;
        }

        public BucketManager build() {
            return new BucketManager(this);
        }
    }

    public static enum BucketPackLevel {
        NO_PACKING,
        CONSOLIDATE,
        FULL_PACKING;

    }
}

