/*
 * Decompiled with CFR 0.152.
 */
package org.gorpipe.gor.manager;

import java.io.IOException;
import java.net.URI;
import java.nio.file.attribute.FileAttribute;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.RandomStringUtils;
import org.apache.parquet.Strings;
import org.gorpipe.exceptions.GorSystemException;
import org.gorpipe.gor.driver.DataSource;
import org.gorpipe.gor.manager.BucketCreator;
import org.gorpipe.gor.manager.BucketCreatorGorPipe;
import org.gorpipe.gor.table.Table;
import org.gorpipe.gor.table.dictionary.BaseDictionaryTable;
import org.gorpipe.gor.table.dictionary.DictionaryEntry;
import org.gorpipe.gor.table.dictionary.DictionaryTable;
import org.gorpipe.gor.table.lock.ExclusiveFileTableLock;
import org.gorpipe.gor.table.lock.TableLock;
import org.gorpipe.gor.table.lock.TableTransaction;
import org.gorpipe.gor.table.util.PathUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class BucketManager<T extends DictionaryEntry> {
    private static final Logger log = LoggerFactory.getLogger(BucketManager.class);
    public static final int DEFAULT_MIN_BUCKET_SIZE = 20;
    public static final int DEFAULT_BUCKET_SIZE = 100;
    public static final int DEFAULT_MAX_BUCKET_COUNT = 3;
    public static final BucketPackLevel DEFAULT_BUCKET_PACK_LEVEL = BucketPackLevel.CONSOLIDATE;
    static final String BUCKET_FILE_PREFIX = "bucket";
    public static final Duration DEFAULT_LOCK_TIMEOUT = Duration.ofMinutes(30L);
    public static final Duration BUCKET_CLEANUP_INTERVAL = Duration.ofMinutes(30L);
    public static final Class<? extends TableLock> DEFAULT_LOCK_TYPE = ExclusiveFileTableLock.class;
    public static final String HEADER_MIN_BUCKET_SIZE_KEY = "MIN_BUCKET_SIZE";
    public static final String HEADER_BUCKET_SIZE_KEY = "BUCKET_SIZE";
    public static final String HEADER_BUCKET_DIRS_KEY = "BUCKET_DIRS";
    public static final String HEADER_BUCKET_DIRS_LOCATION_KEY = "BUCKET_DIRS_LOCATION";
    public static final String HEADER_BUCKET_MAX_BUCKETS = "BUCKET_MAX_BUCKETS";
    protected Duration gracePeriodForDeletingBuckets = Duration.ofHours(24L);
    private final List<String> bucketDirs = new ArrayList<String>();
    private Map<String, Long> bucketDirCount;
    private Class<? extends TableLock> lockType = DEFAULT_LOCK_TYPE;
    private Duration lockTimeout = DEFAULT_LOCK_TIMEOUT;
    private final BaseDictionaryTable<T> table;
    private int bucketSize;
    private int minBucketSize;
    private BucketCreator<T> bucketCreator;
    private long lastCleanupTimeMillis = 0L;

    public BucketManager(BaseDictionaryTable<T> table) {
        this.table = table;
        this.bucketCreator = new BucketCreatorGorPipe();
        this.setBucketSize(Integer.parseInt(table.getConfigTableProperty(HEADER_BUCKET_SIZE_KEY, Integer.toString(100))));
        this.setMinBucketSize(Integer.parseInt(table.getConfigTableProperty(HEADER_MIN_BUCKET_SIZE_KEY, Integer.toString(20))));
        this.setBucketDirs(this.parseBucketDirString(table.getConfigTableProperty(HEADER_BUCKET_DIRS_KEY, null)));
    }

    private BucketManager(Builder builder) {
        this(builder.table);
        this.lockType = builder.lockType != null ? builder.lockType : DEFAULT_LOCK_TYPE;
        Duration duration = this.lockTimeout = builder.lockTimeout != null ? builder.lockTimeout : DEFAULT_LOCK_TIMEOUT;
        if (builder.bucketSize > 0) {
            this.setBucketSize(builder.bucketSize);
        }
        if (builder.minBucketSize > 0) {
            this.setMinBucketSize(builder.minBucketSize);
        }
        if (builder.bucketCreator != null) {
            this.bucketCreator = builder.bucketCreator;
        }
    }

    public static Builder newBuilder(BaseDictionaryTable table) {
        return new Builder(table);
    }

    public void bucketize() {
        this.bucketize(DEFAULT_BUCKET_PACK_LEVEL, -1, null, false);
    }

    public int bucketize(BucketPackLevel packLevel, int maxBucketCount) {
        return this.bucketize(packLevel, maxBucketCount, null, false);
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public int bucketize(BucketPackLevel packLevel, int maxBucketCount, List<String> bucketDirs, boolean forceClean) {
        if (!this.table.isBucketize()) {
            log.info("Bucketize - Bucketize called on {} but as the table is marked not bucketize so nothing was done.", (Object)this.table.getPath());
            return 0;
        }
        if (bucketDirs != null && !bucketDirs.isEmpty()) {
            this.setBucketDirs(bucketDirs);
        }
        try (TableLock bucketizeLock = TableLock.acquireWrite(this.lockType, this.table, (String)"bucketize", (Duration)Duration.ofMillis(1000L));){
            if (!bucketizeLock.isValid()) {
                long millisRunning = System.currentTimeMillis() - bucketizeLock.lastModified();
                log.debug("Bucketize - Bucketization already in progress on {} (has been running for {} seconds)", (Object)this.table.getName(), (Object)(millisRunning / 1000L));
                int n2 = 0;
                return n2;
            }
            if (maxBucketCount <= 0) {
                maxBucketCount = Integer.parseInt(this.table.getConfigTableProperty(HEADER_BUCKET_MAX_BUCKETS, Integer.toString(3)));
                int n = maxBucketCount = maxBucketCount > 0 ? maxBucketCount : 10000;
            }
            if (forceClean || System.currentTimeMillis() - this.lastCleanupTimeMillis > BUCKET_CLEANUP_INTERVAL.toMillis()) {
                this.lastCleanupTimeMillis = System.currentTimeMillis();
                this.cleanTempBucketData(bucketizeLock);
                this.cleanOldBucketFiles(bucketizeLock, forceClean);
            }
            int n = this.doBucketize(bucketizeLock, packLevel, maxBucketCount);
            return n;
        }
        catch (IOException e) {
            throw new GorSystemException((Throwable)e);
        }
    }

    public void deleteBuckets(String ... buckets) {
        this.deleteBuckets(false, buckets);
    }

    public void deleteBuckets(boolean force, String ... buckets) {
        try (TableTransaction trans = TableTransaction.openWriteTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);){
            this.deleteBuckets(trans.getLock(), force, buckets);
            trans.commit();
        }
        catch (IOException e) {
            throw new GorSystemException((Throwable)e);
        }
    }

    protected int getEffectiveMinBucketSize() {
        return Math.min(this.getMinBucketSize(), this.getBucketSize());
    }

    public int getBucketSize() {
        return this.bucketSize;
    }

    public void setBucketSize(int bucketSize) {
        this.bucketSize = bucketSize;
    }

    public int getMinBucketSize() {
        return this.minBucketSize;
    }

    public void setMinBucketSize(int minBucketSize) {
        this.minBucketSize = minBucketSize;
    }

    protected String getDefaultBucketDir() {
        return "." + this.table.getName() + "/buckets";
    }

    public Duration getLockTimeout() {
        return this.lockTimeout;
    }

    private List<String> getBucketDirs() {
        return this.bucketDirs;
    }

    public void setBucketDirs(List<String> newBucketDirs) {
        this.bucketDirs.clear();
        if (newBucketDirs != null && newBucketDirs.size() > 0) {
            this.table.setProperty(HEADER_BUCKET_DIRS_KEY, newBucketDirs.stream().map(p -> PathUtils.formatUri((String)p)).collect(Collectors.joining(",")));
            for (String p2 : newBucketDirs) {
                this.bucketDirs.add(PathUtils.relativize((URI)this.table.getRootUri(), (String)p2));
            }
        } else {
            this.table.setProperty(HEADER_BUCKET_DIRS_KEY, "");
            this.bucketDirs.add(this.getDefaultBucketDir());
        }
    }

    private List<String> parseBucketDirString(String bucketDirs) {
        if (bucketDirs == null) {
            return new ArrayList<String>();
        }
        return Arrays.stream(bucketDirs.split(",")).filter(s -> !s.trim().isEmpty()).collect(Collectors.toList());
    }

    private void checkBucketDirExistance(URI bucketDir) {
        if (!this.table.getFileReader().exists(bucketDir.toString())) {
            URI fullPathDefaultBucketDir = this.table.getRootUri().resolve(this.getDefaultBucketDir().toString());
            if (bucketDir.equals(fullPathDefaultBucketDir)) {
                try {
                    this.table.getFileReader().createDirectoryIfNotExists(fullPathDefaultBucketDir.toString(), new FileAttribute[0]);
                }
                catch (IOException e) {
                    throw new GorSystemException(String.format("Could not create bucket directory %s", fullPathDefaultBucketDir), (Throwable)e);
                }
            } else {
                throw new GorSystemException(String.format("Bucket dirs must exists, directory %s is not found!", bucketDir), null);
            }
        }
    }

    protected final String pickBucketDir() {
        if (this.bucketDirs.size() == 0) {
            throw new GorSystemException("Can not pick bucket, the list of bucket dirs is empty!", null);
        }
        String strategy = this.table.getConfigTableProperty("gor.table.buckets.directory.strategy", "least_used");
        if ("random".equals(strategy)) {
            return this.bucketDirs.get(new Random().nextInt(this.bucketDirs.size()));
        }
        if (this.bucketDirCount == null) {
            this.bucketDirCount = this.table.getEntries().stream().filter(l -> l.getBucket() != null).map(l -> PathUtils.getParent((String)l.getBucket())).filter(p -> this.bucketDirs.contains(p)).collect(Collectors.groupingByConcurrent(Function.identity(), Collectors.counting()));
            for (String bucketDir : this.bucketDirs) {
                if (this.bucketDirCount.containsKey(bucketDir)) continue;
                this.bucketDirCount.put(bucketDir, 0L);
            }
        }
        Map.Entry<String, Long> minEntry = null;
        for (Map.Entry<String, Long> entry : this.bucketDirCount.entrySet()) {
            if (minEntry != null && entry.getValue().compareTo(minEntry.getValue()) >= 0) continue;
            minEntry = entry;
        }
        minEntry.setValue((Long)minEntry.getValue() + 1L);
        return minEntry.getKey();
    }

    private int doBucketize(TableLock bucketizeLock, BucketPackLevel packLevel, int maxBucketCount) throws IOException {
        BaseDictionaryTable tempTable;
        Map<String, List<T>> newBucketsMap;
        Collection<String> bucketsToDelete;
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketize - Bucketization already in progress");
            return 0;
        }
        try (Object trans = TableTransaction.openReadTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);){
            log.trace("Bucketize - Get unbucketized count");
            int unbucketizedCount = this.table.needsBucketizing().size();
            if (packLevel == BucketPackLevel.NO_PACKING && unbucketizedCount < this.getEffectiveMinBucketSize()) {
                log.debug("Bucketize - Nothing to bucketize, aborting {} unbucketized but {} is minimum.", (Object)unbucketizedCount, (Object)this.getEffectiveMinBucketSize());
                int n = 0;
                return n;
            }
            log.trace("Bucketize - Finding files to bucketize");
            bucketsToDelete = this.findBucketsToDelete(trans.getLock(), packLevel, unbucketizedCount, maxBucketCount);
            newBucketsMap = this.findBucketsToCreate(trans.getLock(), bucketsToDelete, maxBucketCount);
            if (log.isDebugEnabled()) {
                log.debug("Bucketize - Bucketizing {} files into {} buckets", (Object)newBucketsMap.values().stream().map(List::size).mapToInt(Integer::intValue).sum(), (Object)newBucketsMap.keySet().size());
            }
            log.trace("Bucketize - Creating the temptable");
            tempTable = this.createTempTable(trans.getLock());
        }
        if (!newBucketsMap.isEmpty()) {
            for (String bucketDir : this.getBucketDirs()) {
                log.trace("Bucketize - Bucketizing dir {}", (Object)bucketDir);
                this.doBucketizeForBucketDir(tempTable, bucketDir, newBucketsMap);
            }
        }
        log.trace("Deleting temp table {}", (Object)tempTable.getPath());
        tempTable.delete();
        if (!bucketsToDelete.isEmpty()) {
            trans = TableTransaction.openWriteTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);
            try {
                this.deleteBuckets(trans.getLock(), false, bucketsToDelete.toArray(new String[bucketsToDelete.size()]));
                trans.commit();
            }
            finally {
                if (trans != null) {
                    trans.close();
                }
            }
        }
        return newBucketsMap.size();
    }

    private void doBucketizeForBucketDir(BaseDictionaryTable tempTable, String bucketDir, Map<String, List<T>> newBucketsMap) throws IOException {
        Map<String, List<T>> newBucketsMapForBucketDir = newBucketsMap.keySet().stream().filter(p -> PathUtils.getParent((String)p).equals(PathUtils.stripTrailingSlash((String)bucketDir))).collect(Collectors.toMap(Function.identity(), newBucketsMap::get));
        this.createBucketFilesForBucketDir(tempTable, newBucketsMapForBucketDir, this.getAbsoluteBucketDir(bucketDir));
        for (String bucket : newBucketsMapForBucketDir.keySet()) {
            List<T> bucketEntries = newBucketsMapForBucketDir.get(bucket);
            this.updateTableWithNewBucket(this.table, bucket, bucketEntries);
        }
    }

    private URI getAbsoluteBucketDir(String bucketDir) throws IOException {
        URI absBucketDir;
        String bucketDirsLocation = this.table.getConfigTableProperty(HEADER_BUCKET_DIRS_LOCATION_KEY, "");
        if (!Strings.isNullOrEmpty((String)bucketDirsLocation)) {
            URI tableRoot = PathUtils.relativize((URI)URI.create(this.table.getFileReader().getCommonRoot()), (URI)this.table.getRootUri());
            absBucketDir = PathUtils.resolve((URI)URI.create(bucketDirsLocation), (String)PathUtils.resolve((URI)tableRoot, (String)bucketDir).toString());
        } else {
            absBucketDir = PathUtils.resolve((URI)this.table.getRootUri(), (String)bucketDir);
        }
        this.table.getFileReader().createDirectoryIfNotExists(absBucketDir.toString(), new FileAttribute[0]);
        return absBucketDir;
    }

    private void updateTableWithNewBucket(BaseDictionaryTable table, String bucket, List<T> bucketEntries) {
        try (TableTransaction trans = TableTransaction.openWriteTransaction(this.lockType, (Table)table, (String)table.getName(), (Duration)this.lockTimeout);){
            table.removeFromBucket(bucketEntries);
            table.addToBucket(bucket, bucketEntries);
            table.setProperty(HEADER_BUCKET_SIZE_KEY, Integer.toString(this.getBucketSize()));
            table.setProperty(HEADER_MIN_BUCKET_SIZE_KEY, Integer.toString(this.getMinBucketSize()));
            trans.commit();
        }
    }

    private BaseDictionaryTable createTempTable(TableLock lock) throws IOException {
        lock.assertValid();
        String ext = FilenameUtils.getExtension((String)this.table.getPath().toString());
        URI tempTablePath = this.table.getRootUri().resolve(this.getTempTablePrefix() + RandomStringUtils.random((int)8, (boolean)true, (boolean)true) + (String)(ext.length() > 0 ? "." + ext : ""));
        log.trace("Creating temp table {}", (Object)tempTablePath);
        this.table.getFileReader().copy(this.table.getPathUri().toString(), tempTablePath.toString());
        return this.initTempTable(tempTablePath.toString());
    }

    String getTempTablePrefix() {
        return "." + this.table.getName() + ".temp.bucketizing.";
    }

    private BaseDictionaryTable initTempTable(String path) {
        if (path.toString().toLowerCase().endsWith(".gord")) {
            return ((DictionaryTable.Builder)((DictionaryTable.Builder)((DictionaryTable.Builder)((DictionaryTable.Builder)new DictionaryTable.Builder(path).useHistory(this.table.isUseHistory())).sourceColumn(this.table.getSourceColumn())).fileReader(this.table.getFileReader())).validateFiles(this.table.isValidateFiles())).build();
        }
        throw new GorSystemException("BaseTable of type " + path.toString() + " are not supported!", null);
    }

    private void deleteBuckets(TableLock lock, boolean force, String ... buckets) throws IOException {
        lock.assertValid();
        this.deleteBucketFiles(force, buckets);
        this.table.removeFromBucket((Collection)this.table.filter().buckets(buckets).includeDeleted().get());
    }

    private void deleteBucketFiles(boolean force, String ... buckets) throws IOException {
        for (String bucket : buckets) {
            URI bucketFile = PathUtils.resolve((URI)this.table.getRootUri(), (String)bucket);
            DataSource source = this.table.getFileReader().resolveUrl(bucketFile.toString());
            if (!source.exists()) continue;
            long lastAccessTime = source.getSourceMetadata().getLastModified();
            log.trace("Checking bucket file CTM {} LAT {} GPFDB {}", new Object[]{System.currentTimeMillis(), lastAccessTime, this.gracePeriodForDeletingBuckets.toMillis()});
            if (!force && System.currentTimeMillis() - lastAccessTime <= this.gracePeriodForDeletingBuckets.toMillis()) continue;
            log.debug("Deleting bucket file {}", (Object)bucketFile);
            source.delete();
            this.deleteFileIfExists(source.getFullPath() + ".gori");
            this.deleteFileIfExists(source.getFullPath() + ".meta");
            this.deleteLinkFileIfExists(bucketFile.toString());
        }
    }

    private void deleteFileIfExists(String path) {
        try {
            DataSource source = this.table.getFileReader().resolveUrl(path);
            if (source != null && source.exists()) {
                source.delete();
            }
        }
        catch (IOException iOException) {
            // empty catch block
        }
    }

    private void deleteLinkFileIfExists(String path) {
        try {
            String linkFile = path + (path.endsWith(".link") ? "" : ".link");
            DataSource linkSource = this.table.getFileReader().resolveDataSource(this.table.getFileReader().createSourceReference(linkFile, false));
            if (linkSource != null && linkSource.exists()) {
                linkSource.delete();
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    void cleanTempBucketData(TableLock bucketizeLock) {
        log.trace("Bucketize - cleanTempBucketData - begin");
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketization in progress, will skip cleaning bucket files.");
            return;
        }
        try (Stream candStream = this.table.getFileReader().list(this.table.getRootUri().toString());){
            String prefix = this.getTempTablePrefix();
            for (String candTempFile : candStream.collect(Collectors.toList())) {
                if (!candTempFile.contains(prefix)) continue;
                this.table.getFileReader().resolveUrl(this.table.getRootUri().resolve(candTempFile).toString()).delete();
            }
        }
        catch (IOException ioe) {
            log.warn("Got exception when trying to clean up temp folders.  Just logging out the exception", (Throwable)ioe);
        }
        log.trace("Bucketize - cleanTempBucketData - end");
    }

    protected void cleanOldBucketFiles(TableLock bucketizeLock, boolean force) throws IOException {
        log.trace("Bucketize - cleanOldBucketFiles - begin");
        if (!bucketizeLock.isValid()) {
            log.debug("Bucketization in progress, will skip cleaning bucket files.");
            return;
        }
        HashMap<String, List<String>> bucketsToCleanMap = new HashMap<String, List<String>>();
        try (TableTransaction trans = TableTransaction.openReadTransaction(this.lockType, this.table, (String)this.table.getName(), (Duration)this.lockTimeout);){
            HashSet<String> allBucketDirs = new HashSet<String>(this.getBucketDirs());
            allBucketDirs.addAll(this.table.getBuckets().stream().map(b -> PathUtils.getParent((String)b)).collect(Collectors.toSet()));
            for (String bucketDir : allBucketDirs) {
                URI fullPathBucketDir = this.table.getRootUri().resolve(bucketDir);
                if (!this.table.getFileReader().exists(fullPathBucketDir.toString())) {
                    log.debug("Bucket folder {} never been used, nothing to clean.", (Object)fullPathBucketDir);
                    continue;
                }
                List<String> bucketsToClean = this.collectBucketsToClean(fullPathBucketDir, force);
                if (bucketsToClean.size() <= 0) continue;
                bucketsToCleanMap.put(bucketDir, bucketsToClean);
            }
        }
        for (String bucketDir : bucketsToCleanMap.keySet()) {
            log.trace("Bucketize - cleanOldBucketFiles - clean {}", (Object)bucketDir);
            List bucketsToClean = (List)bucketsToCleanMap.get(bucketDir);
            this.deleteBuckets(force, bucketsToClean.toArray(new String[0]));
            for (String bucket : bucketsToClean) {
                log.warn("Bucket '{}' removed as it is not used", (Object)bucket);
            }
        }
        log.trace("Bucketize - cleanOldBucketFiles - End");
    }

    private List<String> collectBucketsToClean(URI fullPathBucketDir, boolean force) throws IOException {
        ArrayList<String> bucketsToDelete = new ArrayList<String>();
        HashSet usedBuckets = new HashSet(this.table.getBuckets());
        try (Stream pathList = this.table.getFileReader().list(fullPathBucketDir.toString());){
            for (String f : pathList.collect(Collectors.toList())) {
                String fileName = PathUtils.getFileName((String)f);
                String bucketFile = PathUtils.relativize((URI)this.table.getRootUri(), (String)fullPathBucketDir.resolve(f).toString());
                long lastAccessTime = 0L;
                log.trace("Checking bucket file CTM {} LAT {} GPFDB {}", new Object[]{System.currentTimeMillis(), lastAccessTime, this.gracePeriodForDeletingBuckets.toMillis()});
                if (!fileName.startsWith(this.getBucketFilePrefix(this.table)) || !fileName.endsWith(".gorz") || System.currentTimeMillis() - lastAccessTime <= this.gracePeriodForDeletingBuckets.toMillis() && !force || usedBuckets.contains(bucketFile)) continue;
                bucketsToDelete.add(bucketFile);
            }
        }
        return bucketsToDelete;
    }

    private String getBucketFilePrefix(BaseDictionaryTable table) {
        return table.getName() + "_bucket";
    }

    private Collection<String> findBucketsToDelete(TableLock lock, BucketPackLevel packLevel, int unbucketizedCount, int maxBucketCount) {
        int totalFilesNeeding;
        int totalNeededNewBuckets;
        int totalNewBuckets;
        lock.assertValid();
        HashMap bucketCounts = new HashMap();
        this.table.selectAll().stream().filter(l -> l.hasBucket()).forEach(l -> bucketCounts.put(l.getBucket(), bucketCounts.getOrDefault(l.getBucket(), 0) + (!l.isDeleted() ? 1 : 0)));
        HashSet<String> bucketsToDelete = new HashSet<String>(bucketCounts.keySet().stream().filter(k -> (Integer)bucketCounts.get(k) == 0).collect(Collectors.toSet()));
        if (packLevel == BucketPackLevel.NO_PACKING) {
            return bucketsToDelete;
        }
        if (packLevel == BucketPackLevel.FULL_PACKING) {
            bucketsToDelete.addAll(bucketCounts.keySet().stream().filter(k -> (Integer)bucketCounts.get(k) < this.getBucketSize()).collect(Collectors.toSet()));
        } else if (packLevel == BucketPackLevel.CONSOLIDATE && (totalNewBuckets = Math.min(totalNeededNewBuckets = (totalFilesNeeding = unbucketizedCount + bucketCounts.values().stream().filter(i -> i < this.getBucketSize()).mapToInt(Integer::intValue).sum()) / this.getBucketSize(), maxBucketCount)) == totalNeededNewBuckets) {
            int totalSpaceLeftInNewBuckets = totalNewBuckets * this.getBucketSize() - unbucketizedCount;
            for (Map.Entry entry : bucketCounts.entrySet().stream().filter(e -> (Integer)e.getValue() < this.getBucketSize()).sorted(Map.Entry.comparingByValue()).collect(Collectors.toList())) {
                if (totalSpaceLeftInNewBuckets <= 0) break;
                bucketsToDelete.add((String)entry.getKey());
                totalSpaceLeftInNewBuckets -= ((Integer)entry.getValue()).intValue();
            }
        }
        return bucketsToDelete;
    }

    private Map<String, List<T>> findBucketsToCreate(TableLock lock, Collection<String> bucketsToDelete, int maxBucketCount) {
        lock.assertValid();
        List relBucketsToDelete = bucketsToDelete != null ? bucketsToDelete.stream().map(b -> PathUtils.resolve((String)this.table.getRootPath(), (String)b)).collect(Collectors.toList()) : null;
        List lines2bucketize = this.table.selectAll().stream().filter(l -> !l.hasBucket() || !l.isDeleted() && relBucketsToDelete != null && relBucketsToDelete.contains(l.getBucketReal(this.table.getRootUri()))).collect(Collectors.toList());
        int bucketCreateCount = (int)Math.ceil((double)lines2bucketize.size() / (double)this.getBucketSize());
        if (lines2bucketize.size() - (bucketCreateCount - 1) * this.getBucketSize() < this.getEffectiveMinBucketSize()) {
            --bucketCreateCount;
        }
        String bucketNamePrefix = String.format("%s_%s_%s_", this.getBucketFilePrefix(this.table), new SimpleDateFormat("yyyy_MMdd_HHmmss").format(new Date()), RandomStringUtils.random((int)8, (boolean)true, (boolean)true));
        HashMap<String, List<T>> bucketsToCreate = new HashMap<String, List<T>>();
        this.bucketDirCount = null;
        for (int i = 1; i <= Math.min(bucketCreateCount, maxBucketCount); ++i) {
            String bucketDir = this.pickBucketDir();
            int nextToBeAddedIndex = (i - 1) * this.getBucketSize();
            int nextBucketSize = Math.min(this.getBucketSize(), lines2bucketize.size() - nextToBeAddedIndex);
            bucketsToCreate.put(PathUtils.resolve((String)bucketDir, (String)bucketNamePrefix) + i + ".gorz", lines2bucketize.subList(nextToBeAddedIndex, nextToBeAddedIndex + nextBucketSize));
        }
        return bucketsToCreate;
    }

    private void createBucketFilesForBucketDir(BaseDictionaryTable table, Map<String, List<T>> bucketsToCreate, URI absBucketDir) throws IOException {
        this.checkBucketDirExistance(absBucketDir);
        this.bucketCreator.createBucketsForBucketDir(table, bucketsToCreate, absBucketDir);
    }

    public static final class Builder<T extends DictionaryEntry> {
        private final BaseDictionaryTable<T> table;
        private Duration lockTimeout = null;
        private Class<? extends TableLock> lockType = null;
        private int minBucketSize = -1;
        private int bucketSize = -1;
        private BucketCreator<T> bucketCreator = null;

        private Builder(BaseDictionaryTable<T> table) {
            this.table = table;
        }

        public Builder lockType(Class val) {
            this.lockType = val;
            return this;
        }

        public Builder lockTimeout(Duration val) {
            this.lockTimeout = val;
            return this;
        }

        public Builder minBucketSize(int val) {
            this.minBucketSize = val;
            return this;
        }

        public Builder bucketSize(int val) {
            this.bucketSize = val;
            return this;
        }

        public Builder bucketCreator(BucketCreator<T> val) {
            this.bucketCreator = val;
            return this;
        }

        public BucketManager build() {
            return new BucketManager(this);
        }
    }

    public static enum BucketPackLevel {
        NO_PACKING,
        CONSOLIDATE,
        FULL_PACKING;

    }
}

