/*
 * Copyright 2014 Guidewire Software, Inc.
 */

package gw.internal.gosu.parser.expressions;

import gw.config.CommonServices;
import gw.internal.gosu.parser.ParserBase;
import gw.lang.IDimension;
import gw.lang.parser.ICoercionManager;
import gw.lang.parser.expressions.IMultiplicativeExpression;
import gw.lang.reflect.IMethodInfo;
import gw.lang.reflect.IPlaceholder;
import gw.lang.reflect.IType;
import gw.lang.reflect.ReflectUtil;
import gw.lang.reflect.TypeSystem;
import gw.lang.reflect.java.JavaTypes;
import gw.util.Rational;

import java.math.BigDecimal;
import java.math.MathContext;


/**
 * Represents a multiplicative expression in the Gosu grammar:
 * <pre>
 * <i>multiplicative-expression</i>
 *   &lt;unary-expression&gt;
 *   &lt;multiplicative-expression&gt; <b>*</b> &lt;unary-expression&gt;
 *   &lt;multiplicative-expression&gt; <b>/</b> &lt;unary-expression&gt;
 *   &lt;multiplicative-expression&gt; <b>%</b> &lt;unary-expression&gt;
 * </pre>
 * <p/>
 *
 * @see gw.lang.parser.IGosuParser
 */
public final class MultiplicativeExpression extends ArithmeticExpression implements IMultiplicativeExpression
{
  /**
   * Evaluate the expression.
   */
  public Object evaluate()
  {
    if( !isCompileTimeConstant() )
    {
      return super.evaluate();
    }

    return evaluate( getType(), getLHS().evaluate(), getRHS().evaluate(), getLHS().getType(), getRHS().getType(), getOperator().charAt( 0 ), false );
  }

  public static Object evaluate( IType type, Object lhsValue, Object rhsValue, IType lhsType, IType rhsType, int iOperator, boolean bNullSafe, Object ctx, int startLhs, int endLhs, int startRhs, int endRhs )
  {
    return evaluate( type, lhsValue, rhsValue, lhsType, rhsType, iOperator, bNullSafe );
  }


  // Potentially called from generated bytecode
  @SuppressWarnings({"ConstantConditions"})
  public static Object evaluate( IType type, Object lhsValue, Object rhsValue, IType lhsType, IType rhsType, int iOperator, boolean bNullSafe )
  {
    boolean bDynamic = false;
    if( lhsType instanceof IPlaceholder && ((IPlaceholder)lhsType).isPlaceholder() )
    {
      lhsType = TypeSystem.getFromObject( lhsValue );
      bDynamic = true;
    }
    if( rhsType instanceof IPlaceholder && ((IPlaceholder)rhsType).isPlaceholder() )
    {
      rhsType = TypeSystem.getFromObject( rhsValue );
      bDynamic = true;
    }
    if( bDynamic )
    {
      ArithmeticExpression overrideMethod = new MultiplicativeExpression();
      type = ParserBase.resolveRuntimeType( overrideMethod, lhsType, iOperator, rhsType );
      IMethodInfo mi = overrideMethod.getOverride();
      if( mi != null )
      {
        return mi.getCallHandler().handleCall( lhsValue, ReflectUtil.coerceArgsIfNecessary( mi.getParameters(), rhsValue ) );
      }
    }

    if( lhsValue == null )
    {
      if( bNullSafe )
      {
        return null;
      }
      throw new NullPointerException("left-hand operand was null");
    }
    if( rhsValue == null )
    {
      if( bNullSafe )
      {
        return null;
      }
      throw new NullPointerException("right-hand operand was null");
    }

    IDimension customNumberBase = null;
    if( JavaTypes.IDIMENSION().isAssignableFrom( type ) )
    {
      DimensionOperandResolver customNumberResolver = DimensionOperandResolver.resolve( type, lhsType, lhsValue, rhsType, rhsValue );
      type = customNumberResolver.getRawNumberType();
      lhsValue = customNumberResolver.getLhsValue();
      rhsValue = customNumberResolver.getRhsValue();
      customNumberBase = customNumberResolver.getBase();
    }

    Object retValue;
    ICoercionManager cm = CommonServices.getCoercionManager();
    switch( iOperator )
    {
      case '*':
      {
        if( type == JavaTypes.RATIONAL() )
        {
          retValue = cm.makeRationalFrom( lhsValue ).multiply( cm.makeRationalFrom( rhsValue ) );
        }
        else if( type == JavaTypes.BIG_DECIMAL() )
        {
          retValue = cm.makeBigDecimalFrom( lhsValue ).multiply( cm.makeBigDecimalFrom( rhsValue ) );
        }
        else if( type == JavaTypes.BIG_INTEGER() )
        {
          retValue = cm.makeBigIntegerFrom( lhsValue ).multiply( cm.makeBigIntegerFrom( rhsValue ) );
        }
        else if( type == JavaTypes.INTEGER() || type == JavaTypes.pINT() )
        {
          retValue = cm.makeIntegerFrom( lhsValue ) * cm.makeIntegerFrom( rhsValue );
        }
        else if( type == JavaTypes.LONG() || type == JavaTypes.pLONG() )
        {
          retValue = makeLong( cm.makeLongFrom( lhsValue ) * cm.makeLongFrom( rhsValue ) );
        }
        else if( type == JavaTypes.DOUBLE() || type == JavaTypes.pDOUBLE() )
        {
          retValue =  makeDoubleValue( makeDoubleValue( lhsValue ) * makeDoubleValue( rhsValue ) );
        }
        else if( type == JavaTypes.FLOAT() || type == JavaTypes.pFLOAT() )
        {
          retValue = makeFloatValue( makeFloatValue( lhsValue ) * makeFloatValue( rhsValue ) );
        }
        else if( type == JavaTypes.SHORT() || type == JavaTypes.pSHORT() )
        {
          retValue = Integer.valueOf( cm.makeIntegerFrom( lhsValue ) * cm.makeIntegerFrom( rhsValue ) ).shortValue();
        }
        else if( type == JavaTypes.BYTE() || type == JavaTypes.pBYTE() )
        {
          retValue = (byte)(cm.makeIntegerFrom( lhsValue ) * cm.makeIntegerFrom( rhsValue ));
        }
        else
        {
          throw new UnsupportedNumberTypeException(type);
        }
        break;
      }

      case '/':
      {
        if( type == JavaTypes.RATIONAL() )
        {
          retValue = cm.makeRationalFrom( lhsValue ).divide( cm.makeRationalFrom( rhsValue ) );
        }
        else if( type == JavaTypes.BIG_DECIMAL() )
        {
          retValue = cm.makeBigDecimalFrom( lhsValue ).divide( cm.makeBigDecimalFrom( rhsValue ), MathContext.DECIMAL128 );
        }
        else if( type == JavaTypes.BIG_INTEGER() )
        {
          retValue = cm.makeBigIntegerFrom( lhsValue ).divide( cm.makeBigIntegerFrom( rhsValue ) );
        }
        else if( type == JavaTypes.INTEGER() || type == JavaTypes.pINT() )
        {
          retValue = cm.makeIntegerFrom( lhsValue ) / cm.makeIntegerFrom( rhsValue );
        }
        else if( type == JavaTypes.LONG() || type == JavaTypes.pLONG() )
        {
          retValue = makeLong( cm.makeLongFrom( lhsValue ) / cm.makeLongFrom( rhsValue ) );
        }
        else if( type == JavaTypes.DOUBLE() || type == JavaTypes.pDOUBLE() )
        {
          retValue = makeDoubleValue( makeDoubleValue( lhsValue ) / makeDoubleValue( rhsValue ) );
        }
        else if( type == JavaTypes.FLOAT() || type == JavaTypes.pFLOAT() )
        {
          retValue = makeFloatValue( makeFloatValue( lhsValue ) / makeFloatValue( rhsValue ) );
        }
        else if( type == JavaTypes.SHORT() || type == JavaTypes.pSHORT() )
        {
          retValue = Integer.valueOf( cm.makeIntegerFrom( lhsValue ) / cm.makeIntegerFrom( rhsValue ) ).shortValue();
        }
        else if( type == JavaTypes.BYTE() || type == JavaTypes.pBYTE() )
        {
          retValue = (byte)(cm.makeIntegerFrom( lhsValue ) / cm.makeIntegerFrom( rhsValue ));
        }
        else
        {
          throw new UnsupportedNumberTypeException(type);
        }
        break;
      }

      case '%':
      {
        if( type == JavaTypes.RATIONAL() )
        {
          retValue = cm.makeRationalFrom( lhsValue ).modulo( cm.makeRationalFrom( rhsValue ) );
        }
        else if( type == JavaTypes.BIG_DECIMAL() )
        {
          retValue = cm.makeBigDecimalFrom( lhsValue ).remainder( cm.makeBigDecimalFrom( rhsValue ), MathContext.DECIMAL128 ).abs();
        }
        else if( type == JavaTypes.BIG_INTEGER() )
        {
          retValue = cm.makeBigIntegerFrom( lhsValue ).mod( cm.makeBigIntegerFrom( rhsValue ) );
        }
        else if( type == JavaTypes.INTEGER() || type == JavaTypes.pINT() )
        {
          retValue = cm.makeIntegerFrom( lhsValue ) % cm.makeIntegerFrom( rhsValue );
        }
        else if( type == JavaTypes.LONG() || type == JavaTypes.pLONG() )
        {
          retValue = makeLong( cm.makeLongFrom( lhsValue ) % cm.makeLongFrom( rhsValue ) );
        }
        else if( type == JavaTypes.DOUBLE() || type == JavaTypes.pDOUBLE() )
        {
          retValue = makeDoubleValue( makeDoubleValue( lhsValue ) % makeDoubleValue( rhsValue ) );
        }
        else if( type == JavaTypes.FLOAT() || type == JavaTypes.pFLOAT() )
        {
          retValue = makeFloatValue( makeFloatValue( lhsValue ) % makeFloatValue( rhsValue ) );
        }
        else if( type == JavaTypes.SHORT() || type == JavaTypes.pSHORT() )
        {
          retValue = Integer.valueOf( cm.makeIntegerFrom( lhsValue ) % cm.makeIntegerFrom( rhsValue ) ).shortValue();
        }
        else if( type == JavaTypes.BYTE() || type == JavaTypes.pBYTE() )
        {
          retValue = (byte)(cm.makeIntegerFrom( lhsValue ) % cm.makeIntegerFrom( rhsValue ));
        }
        else
        {
          throw new UnsupportedNumberTypeException(type);
        }
        break;
      }

      default:
        retValue = null;
    }

    if( retValue != null )
    {
      if( customNumberBase != null )
      {
        //noinspection unchecked
        retValue = customNumberBase.fromNumber( (Number)retValue );
      }
    }
    return retValue;
  }
}
