package editor;

import editor.util.EditorUtilities;
import editor.util.Experiment;

import editor.util.LabButton;
import javax.swing.*;
import javax.swing.border.EmptyBorder;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.io.File;
import java.util.ArrayList;
import java.util.List;
import java.util.StringTokenizer;
import java.util.stream.Collectors;

public class ClasspathDialog extends JDialog implements IHandleCancel
{
  private JTextPane _pathsList;
  private JTextPane _backingSourceList;
  private File _dir;

  public ClasspathDialog( File dir )
  {
    super( (JFrame)KeyboardFocusManager.getCurrentKeyboardFocusManager().getActiveWindow(), "Dependencies", true );
    _dir = dir;
    configUI();
    addWindowListener( new WindowAdapter()
    {
      public void windowClosing( WindowEvent e )
      {
        dispose();
      }
    } );
  }


  protected void configUI()
  {
    JComponent contentPane = (JComponent)getContentPane();
    contentPane.setBackground( Scheme.active().getMenu() );
    contentPane.setBorder( BorderFactory.createEmptyBorder( 8, 8, 8, 8 ) );
    contentPane.setLayout( new BorderLayout() );

    GridBagLayout gridBag = new GridBagLayout();
    JPanel mainPanel = new JPanel( gridBag );
    GridBagConstraints c = new GridBagConstraints();

    mainPanel.setBorder( BorderFactory.createCompoundBorder( BorderFactory.createLineBorder( Scheme.active().getScrollbarBorderColor() ),
                                                             BorderFactory.createEmptyBorder( 5, 5, 5, 5 ) ) );



    JLabel label = new JLabel( "<html>Need to use classes outside your experiment? Configure a '<b>" + File.pathSeparator + "</b>' " +
                               "separated list of directories and/or jar files containing Gosu or Java classes your experiment uses." );
    label.setBorder( new EmptyBorder( 2, 0, 0, 0 ) );
    int iY = 0;
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 0;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = 0;
    c.insets = new Insets( 0, 0, 10, 0 );
    mainPanel.add( label, c );

    label = new JLabel( "Dependencies", EditorUtilities.loadIcon( "images/folder.png" ), SwingConstants.LEFT );
    label.setBorder( new EmptyBorder( 0, 0, 2, 0 ) );
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = 0;
    c.insets = new Insets( 0, 0, 2, 0 );
    mainPanel.add( label, c );

    _pathsList = new JTextPane();
    JScrollPane scroller = new JScrollPane( _pathsList );
    setPathsList();
    _pathsList.setBorder( BorderFactory.createEmptyBorder() );
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.BOTH;
    c.gridx = 0;
    c.gridy = iY;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = .5;
    c.insets = new Insets( 0, 0, 10, 0 );
    mainPanel.add( scroller, c );

    JButton btnPaths = new LabButton( "..." );
    btnPaths.setToolTipText( "Find a directory or Jar file" );
    btnPaths.addActionListener( e -> updatePaths( _pathsList ) );
    c.anchor = GridBagConstraints.NORTH;
    c.fill = GridBagConstraints.NONE;
    c.gridx = 1;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets( 0, 4, 2, 0 );
    mainPanel.add( btnPaths, c );

    label = new JLabel( "<html>If you have source files corresponding with any of the above dependencies, you can add those here.  They are useful for debugging, code navigation, etc." );
    label.setBorder( new EmptyBorder( 2, 0, 0, 0 ) );
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.HORIZONTAL;
    c.gridx = 0;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = 0;
    c.insets = new Insets( 0, 0, 10, 0 );
    mainPanel.add( label, c );

    label = new JLabel( "Dependency Sources (optional)", EditorUtilities.loadIcon( "images/folder.png" ), SwingConstants.LEFT );
    label.setBorder( new EmptyBorder( 0, 0, 2, 0 ) );
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.NONE;
    c.gridx = 0;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = 0;
    c.insets = new Insets( 0, 0, 2, 0 );
    mainPanel.add( label, c );

    _backingSourceList = new JTextPane();
    scroller = new JScrollPane( _backingSourceList );
    setBackingSourceList();
    _backingSourceList.setBorder( BorderFactory.createEmptyBorder() );
    c.anchor = GridBagConstraints.WEST;
    c.fill = GridBagConstraints.BOTH;
    c.gridx = 0;
    c.gridy = iY;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 1;
    c.weighty = .5;
    c.insets = new Insets( 0, 0, 10, 0 );
    mainPanel.add( scroller, c );

    btnPaths = new LabButton( "..." );
    btnPaths.setToolTipText( "Find a directory or Jar file" );
    btnPaths.addActionListener( e -> updatePaths( _backingSourceList ) );
    c.anchor = GridBagConstraints.NORTH;
    c.fill = GridBagConstraints.NONE;
    c.gridx = 1;
    c.gridy = iY++;
    c.gridwidth = 1;
    c.gridheight = 1;
    c.weightx = 0;
    c.weighty = 0;
    c.insets = new Insets( 0, 4, 2, 0 );
    mainPanel.add( btnPaths, c );

    contentPane.add( mainPanel, BorderLayout.CENTER );

    JPanel south = new JPanel( new BorderLayout() );
    south.setBorder( BorderFactory.createEmptyBorder( 4, 0, 0, 0 ) );
    south.setBackground( Scheme.active().getMenu() );
    JPanel filler = new JPanel();
    filler.setBackground( Scheme.active().getMenu() );
    south.add( filler, BorderLayout.CENTER );

    JPanel buttonPanel = new JPanel();
    buttonPanel.setLayout( new BoxLayout( buttonPanel, BoxLayout.X_AXIS ) );

    JButton btnFind = new LabButton( "OK" );
    btnFind.setMnemonic( 'O' );
    btnFind.addActionListener(
      e -> {
        save();
        close();
      } );
    buttonPanel.add( btnFind );
    getRootPane().setDefaultButton( btnFind );

    JButton btnCancel = new LabButton( "Cancel" );
    btnCancel.addActionListener( e -> close() );
    buttonPanel.add( btnCancel );
    buttonPanel.setBackground( Scheme.active().getMenu() );

    south.add( buttonPanel, BorderLayout.EAST );
    contentPane.add( south, BorderLayout.SOUTH );

    mapCancelKeystroke( "Cancel", this::close );

    setSize( 600, 400 );

    EditorUtilities.centerWindowInFrame( this, getOwner() );
  }

  private void setPathsList()
  {
    String strPaths = "";
    List<String> paths =  LabFrame.instance().getGosuPanel().getExperimentView().getExperiment().getSourcePath();
    for( int i = 0; i < paths.size(); i++ )
    {
      String strPath = paths.get( i );
      strPaths += strPath + (i == paths.size() - 1 ? "" : File.pathSeparator + "\n");
    }
    _pathsList.setText( strPaths );
  }

  private void setBackingSourceList()
  {
    String strPaths = "";
    List<String> paths =  LabFrame.instance().getGosuPanel().getExperimentView().getExperiment().getBackingSourcePath();
    for( int i = 0; i < paths.size(); i++ )
    {
      String strPath = paths.get( i );
      strPaths += strPath + (i == paths.size() - 1 ? "" : File.pathSeparator + "\n");
    }
    _backingSourceList.setText( strPaths );
  }

  private void save()
  {
    List<File> classpath = saveClassPath( _pathsList.getText() );
    List<File> backingSourcePath = saveClassPath( _backingSourceList.getText() );
    savePathsAndReopenExperiment( classpath, backingSourcePath );
  }
  private List<File> saveClassPath( String strPaths )
  {
    StringTokenizer tokenizer = new StringTokenizer( strPaths, File.pathSeparator + "\n\r\t" );
    List<String> paths = new ArrayList<>();
    for(; tokenizer.hasMoreTokens(); )
    {
      String strPath = tokenizer.nextToken();
      paths.add( strPath.trim() );
    }
    List<File> pathFiles = new ArrayList<>();
    //noinspection Convert2streamapi
    for( String strPath : paths )
    {
      pathFiles.add( new File( strPath ).getAbsoluteFile() );
    }
    return pathFiles;
  }

  private void savePathsAndReopenExperiment( List<File> pathFiles, List<File> backingSourcePath )
  {
    GosuPanel gosuPanel = LabFrame.instance().getGosuPanel();
    Experiment experiment = gosuPanel.getExperimentView().getExperiment();
    List<String> sourcePaths = pathFiles.stream().map( File::getAbsolutePath ).collect( Collectors.toList() );
    experiment.setSourcePath( sourcePaths );
    List<String> backingSourcePaths = backingSourcePath.stream().map( File::getAbsolutePath ).collect( Collectors.toList() );
    experiment.setBackingSourcePath( backingSourcePaths );
    gosuPanel.openExperiment( experiment.getExperimentDir() );
  }

  private void updatePaths( JTextPane target )
  {
    JFileChooser fc = new JFileChooser( getCurrentDir() );
    fc.setDialogTitle( "Add Paths" );
    fc.setDialogType( JFileChooser.OPEN_DIALOG );
    fc.setFileSelectionMode( JFileChooser.FILES_AND_DIRECTORIES );
    fc.setMultiSelectionEnabled( true );
    fc.setFileFilter(
      new FileFilter()
      {
        public boolean accept( File f )
        {
          return f.isDirectory() || f.getName().endsWith( ".zip" ) || f.getName().endsWith( ".jar" );
        }

        public String getDescription()
        {
          return "Directories or Archive Files";
        }
      } );
    int returnVal = fc.showOpenDialog( editor.util.EditorUtilities.frameForComponent( this ) );
    if( returnVal == JFileChooser.APPROVE_OPTION )
    {
      String strExisting = target.getText().trim();
      if( strExisting.endsWith( File.pathSeparator ) )
      {
        strExisting = strExisting.substring( 0, strExisting.length()-1 );
      }
      for( File f : fc.getSelectedFiles() )
      {
        strExisting += File.pathSeparatorChar + "\n" + f.getAbsolutePath();
      }
      _dir = fc.getCurrentDirectory();
      target.setText( strExisting );
    }
  }

  private File getCurrentDir()
  {
    return _dir;
  }

  private void close()
  {
    dispose();
  }
}
