/*
  GRANITE DATA SERVICES
  Copyright (C) 2011 GRANITE DATA SERVICES S.A.S.

  This file is part of Granite Data Services.

  Granite Data Services is free software; you can redistribute it and/or modify
  it under the terms of the GNU Library General Public License as published by
  the Free Software Foundation; either version 2 of the License, or (at your
  option) any later version.

  Granite Data Services is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Library General Public License
  for more details.

  You should have received a copy of the GNU Library General Public License
  along with this library; if not, see <http://www.gnu.org/licenses/>.
*/

package org.granite.tide.data;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import static org.granite.tide.data.DataDispatcher.TIDE_DATA_TYPE_KEY;
import static org.granite.tide.data.DataDispatcher.TIDE_DATA_TYPE_VALUE;



public class DataObserveParams implements Serializable {
	
	private static final long serialVersionUID = 1L;
	
	
	private Map<String, Set<String>> params = new HashMap<String, Set<String>>();
	private String selector = null;
	
	
	public boolean isEmpty() {
		return selector == null && params.isEmpty();
	}
	
	public boolean addValue(String paramName, String value) {
		if (paramName == null || value == null || paramName.trim().length() == 0 || value.trim().length() == 0)
			throw new NullPointerException("paramName and value cannot be null or empty");
		if (this.selector != null)
			throw new IllegalArgumentException("Cannot mix manual and automatic selectors");
		
		Set<String> values = params.get(paramName);
		if (values == null) {
			values = new HashSet<String>();
			params.put(paramName, values);
		}
		return values.add(value);
	}
	
	public void setSelector(String selector) {
		if (selector == null || selector.trim().length() == 0)
			throw new NullPointerException("selector cannot be null or empty");
		if (!this.params.isEmpty())
			throw new IllegalArgumentException("Cannot mix manual and automatic selectors");
		
		this.selector = selector;
	}
	
	
	public void append(StringBuilder sb) {

		if (selector != null) {
			sb.append("(").append(selector).append(")");
			return;
		}

		boolean f = true;
		for (Map.Entry<String, Set<String>> me : params.entrySet()) {
			if (f)
				f = false;
			else
				sb.append(" AND ");
			
			Set<String> values = me.getValue();
			if (values.size() == 1)
				sb.append(me.getKey()).append(" = '").append(values.iterator().next()).append("'");
			else {
				sb.append(me.getKey()).append(" IN (");
				boolean ff = true;
				for (String value : values) {
					if (ff)
						ff = false;
					else
						sb.append(", ");
					sb.append("'").append(value).append("'");
				}
				sb.append(")");
			}
		}
	}
    
    
    private static boolean containsParams(List<DataObserveParams> selectors, DataObserveParams params) {
    	for (DataObserveParams selector : selectors) {
    		if (selector.containsParams(params))
    			return true;
    	}
    	return false;
    }	
    
    private boolean containsParams(DataObserveParams params) {
    	if (this.selector != null && !this.selector.equals(params.selector))
    		return false;
    	
    	for (Map.Entry<String, Set<String>> me : params.params.entrySet()) {
    		Set<String> values = this.params.get(me.getKey());
    		if (values == null || !values.containsAll(me.getValue()))
    			return false;
    	}
    	
    	return params.params.keySet().containsAll(this.params.keySet());
    }

	public String updateDataSelector(String dataSelector, List<DataObserveParams> selectors) {
		if (!containsParams(selectors, this)) {
			if (!isEmpty()) {
				List<DataObserveParams> sels = new ArrayList<DataObserveParams>(selectors);
				selectors.clear();
				for (DataObserveParams s : sels) {
					if (!this.containsParams(s))
						selectors.add(s);
				}
				selectors.add(this);
			}
			
			return buildSelectorString(selectors);
		}
		else if (dataSelector == null) {
			return TIDE_DATA_TYPE_KEY + " = 'UNINITIALIZED'";
		}
		return dataSelector;
	}

	private String buildSelectorString(List<DataObserveParams> selectors) {
		StringBuilder sb = new StringBuilder(TIDE_DATA_TYPE_KEY + " = '" + TIDE_DATA_TYPE_VALUE + "'");
		
		if (!selectors.isEmpty()) {
			sb.append(" AND (");
			boolean first = true;
			for (DataObserveParams selector : selectors) {
				if (first)
					first = false;
				else
					sb.append(" OR ");
				sb.append("(");
				selector.append(sb);
				sb.append(")");
			}
			sb.append(")");
		}
		
		return sb.toString();
	}
	
	@Override
	public String toString() {
		StringBuilder sb = new StringBuilder();
		append(sb);
		return sb.toString();
	}
}
