/*
  GRANITE DATA SERVICES
  Copyright (C) 2007-2008 ADEQUATE SYSTEMS SARL

  This file is part of Granite Data Services.

  Granite Data Services is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 3 of the License, or (at your
  option) any later version.

  Granite Data Services is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, see <http://www.gnu.org/licenses/>.
*/

package org.granite.generator.as3.reflect;

import java.lang.reflect.Type;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.persistence.IdClass;
import javax.persistence.ManyToMany;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.OneToOne;

import org.granite.generator.as3.As3Type;

/**
 * @author Franck WOLFF
 */
public class JavaEntityBean extends JavaBean {

    protected final List<JavaFieldProperty> identifiers;
    protected final JavaType idClass;
    protected JavaFieldProperty version;
	private Map<String, As3Type> hasMany = new HashMap<String, As3Type>();

    
    public JavaEntityBean(JavaTypeFactory provider, Class<?> type, URL url) {
        super(provider, type, url);

        // Find identifiers.
        List<JavaFieldProperty> tmpIdentifiers = new ArrayList<JavaFieldProperty>();
        for (JavaProperty property : properties.values()) {
            if (property instanceof JavaFieldProperty) {
                JavaFieldProperty fieldProperty = (JavaFieldProperty)property;
                if (provider.isId(fieldProperty))
                    tmpIdentifiers.add(fieldProperty);
                else if (provider.isVersion(fieldProperty))
                	version = fieldProperty;
            }
        }
        this.identifiers = (tmpIdentifiers.isEmpty() ? null : Collections.unmodifiableList(tmpIdentifiers));

        // Find IdClass (if any).
        this.idClass = (
            type.isAnnotationPresent(IdClass.class) ?
            provider.getJavaType(type.getAnnotation(IdClass.class).value()) :
            null
        );

        // Collect additional imports.
        if (idClass != null)
            addToImports(provider.getJavaImport(idClass.getType()));
        
        // Collect xToMany association properties
        for (JavaProperty property : properties.values()) {
    		Map<String, String> d = new HashMap<String, String>();
    		
        	if (property.isAnnotationPresent(OneToMany.class)) {
        		OneToMany anno = property.getAnnotation(OneToMany.class);
        		Class<?> targetEntity = anno.targetEntity();
        		Type[] types = property.getGenericTypes();
        		if (targetEntity == void.class && types != null && types.length > 0)
        			targetEntity = (Class<?>)property.getGenericTypes()[0];
        		if (targetEntity != void.class) {
        			hasMany.put(property.getName(), getProvider().getAs3Type(targetEntity));
        			
        			d.put("association", "oneToMany");
        			if (!("".equals(anno.mappedBy())))
        				d.put("bidirectional", "true");
        		}
        	}
        	if (property.isAnnotationPresent(ManyToMany.class)) {
        		ManyToMany anno = property.getAnnotation(ManyToMany.class);
        		Class<?> targetEntity = anno.targetEntity();
        		Type[] types = property.getGenericTypes();
        		if (targetEntity == void.class && types != null && types.length > 0)
        			targetEntity = (Class<?>)property.getGenericTypes()[0];
        		if (targetEntity != void.class) {
        			hasMany.put(property.getName(), getProvider().getAs3Type(targetEntity));
        			
        			d.put("association", "manyToMany");
        			if (!("".equals(anno.mappedBy())))
        				d.put("bidirectional", "true");
        		}
        	}
        	if (property.isAnnotationPresent(OneToOne.class)) {
        		OneToOne anno = property.getAnnotation(OneToOne.class);
        		Class<?> targetEntity = anno.targetEntity();
        		if (targetEntity == void.class)
        			targetEntity = property.getType();
        		if (targetEntity != void.class)
        			d.put("association", "manyToOne");
        	}
        	if (property.isAnnotationPresent(ManyToOne.class)) {
        		ManyToOne anno = property.getAnnotation(ManyToOne.class);
        		Class<?> targetEntity = anno.targetEntity();
        		if (targetEntity == void.class)
        			targetEntity = property.getType();
        		if (targetEntity != void.class)
        			d.put("association", "manyToOne");
        	}
        }
    }

    public boolean hasIdentifiers() {
        return identifiers != null && !identifiers.isEmpty();
    }
    public List<JavaFieldProperty> getIdentifiers() {
        return identifiers;
    }
    public JavaFieldProperty getFirstIdentifier() {
        return (identifiers != null ? identifiers.get(0) : null);
    }

    public boolean hasIdClass() {
        return idClass != null;
    }
    public JavaType getIdClass() {
        return idClass;
    }
    
    public boolean hasVersion() {
    	return version != null;
    }
    public JavaFieldProperty getVersion() {
    	return version;
    }
    
    public Map<String, As3Type> getHasMany() {
    	return hasMany;
    }
}
