/*
  GRANITE DATA SERVICES
  Copyright (C) 2007-2008 ADEQUATE SYSTEMS SARL

  This file is part of Granite Data Services.

  Granite Data Services is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 3 of the License, or (at your
  option) any later version.

  Granite Data Services is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, see <http://www.gnu.org/licenses/>.
*/

package org.granite.generator.as3.reflect;

import java.lang.annotation.Annotation;
import java.lang.reflect.Method;
import java.util.Map;

import org.granite.generator.as3.As3Type;
import org.granite.messaging.service.annotations.Param;

/**
 * @author Franck WOLFF
 */
public class JavaMethod extends JavaMember<Method> {

    public enum MethodType {
        GETTER,
        SETTER,
        OTHER
    }

    private final String name;
    private final boolean override;
    private final MethodType type;
    private final Class<?>[] parameterTypes;
    private final As3Type[] as3ParameterTypes;
    private final String[] as3ParameterNames;
    
    public JavaMethod(Method method, MethodType type) {
    	this(method, type, null);
    }
    
    public JavaMethod(Method method, MethodType type, JavaTypeFactory provider) {
        super(method);

        this.name = method.getName();
        
        boolean override = false;
        for (Class<?> superclass = method.getDeclaringClass().getSuperclass();
            superclass != null;
            superclass = superclass.getSuperclass()) {

            try {
                Method superMethod = superclass.getMethod(method.getName(), method.getParameterTypes());
                // Strict return type checking (isAssignableFrom?)...
                if (superMethod.getReturnType() == method.getReturnType()) {
                    override = true;
                    break;
                }
            } catch (NoSuchMethodException e) {
                // continue...
            }
        }
        this.override = override;

        this.type = type;
        
		if (type == MethodType.OTHER && provider != null) {
			this.parameterTypes = method.getParameterTypes();
			this.as3ParameterTypes = new As3Type[this.parameterTypes.length];
			this.as3ParameterNames = new String[this.parameterTypes.length];
			for (int i = 0; i < this.parameterTypes.length; i++) {
				as3ParameterNames[i] = getParamName(method, i);
				if (Map.class.isAssignableFrom(parameterTypes[i]))
					as3ParameterTypes[i] = As3Type.OBJECT;
				else
					as3ParameterTypes[i] = provider.getAs3Type(this.parameterTypes[i]);
			}
		} 
		else {
			this.parameterTypes = null;
			this.as3ParameterTypes = null;
			this.as3ParameterNames = null;
		}
    }
    
    private String getParamName(Method method, int paramIndex) {
    	Annotation[][] annotations = method.getParameterAnnotations();
    	if (annotations != null && annotations.length > paramIndex && annotations[paramIndex] != null) {
    		for (Annotation annotation : annotations[paramIndex]) {
    			if (annotation.annotationType().equals(Param.class))
    				return ((Param)annotation).value();
    		}
    	}
    	return "arg" + paramIndex;
    }

    public boolean isOverride() {
        return override;
    }

    public MethodType getType() {
        return type;
    }

    public String getTypeName() {
        return type.name();
    }
    
    @Override
	public String getName() {
        return name;
    }
   
    public Class<?>[] getParameterTypes() {
    	return parameterTypes;
    }

	public As3Type[] getAs3ParameterTypes() {
		return as3ParameterTypes;
	}

	public String[] getAs3ParameterNames() {
		return as3ParameterNames;
	}
}
