/*
  GRANITE DATA SERVICES
  Copyright (C) 2007 ADEQUATE SYSTEMS SARL

  This file is part of Granite Data Services.

  Granite Data Services is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published by
  the Free Software Foundation; either version 3 of the License, or (at your
  option) any later version.

  Granite Data Services is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
  FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
  for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.tide.spring;

import java.util.Map;

import javax.servlet.ServletContext;

import org.granite.config.flex.Destination;
import org.granite.context.GraniteContext;
import org.granite.logging.Logger;
import org.granite.messaging.service.ExtendedServiceExceptionHandler;
import org.granite.messaging.service.ServiceException;
import org.granite.messaging.service.ServiceFactory;
import org.granite.messaging.service.ServiceInvoker;
import org.granite.messaging.webapp.HttpGraniteContext;
import org.granite.tide.TideServiceInvoker;
import org.granite.tide.data.PersistenceExceptionConverter;
import org.granite.util.ClassUtil;
import org.granite.util.XMap;
import org.springframework.context.ApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import flex.messaging.messages.RemotingMessage;


/**
 * @author Sebastien Deleuze
 */
public class SpringServiceFactory extends ServiceFactory {

    private static final long serialVersionUID = 1L;
    
	private static final Logger log = Logger.getLogger(SpringServiceFactory.class);
    
    public static final String PERSISTENCE_MANAGER_BEAN_NAME = "persistence-manager-bean-name";

    private String lookup = null;

    public String getLookup() {
        return lookup;
    }


    @Override
    public void configure(XMap properties) throws ServiceException {
        String sServiceExceptionHandler = properties.get("service-exception-handler");
        if (sServiceExceptionHandler == null) {
            XMap props = new XMap(properties);
            props.put("service-exception-handler", ExtendedServiceExceptionHandler.class.getName());
            super.configure(props);
        }
        else
            super.configure(properties);
        
        GraniteContext graniteContext = GraniteContext.getCurrentInstance();
        try {
        	graniteContext.getGraniteConfig().registerExceptionConverter(PersistenceExceptionConverter.class);
        }
        catch (Throwable t) {
        	log.error("Could not register exception converter", t);
        }
        
        this.lookup = properties.get("lookup");
    }


    @Override
    public ServiceInvoker<?> getServiceInstance(RemotingMessage request) throws ServiceException {
   	
    	String messageType = request.getClass().getName();
        String destinationId = request.getDestination();

        GraniteContext context = GraniteContext.getCurrentInstance();
        Map<String, Object> cache = context.getSessionMap();
        Destination destination = context.getServicesConfig().findDestinationById(messageType, destinationId);
        if (destination == null)
            throw new ServiceException("No matching destination: " + destinationId);
        
        String key = TideServiceInvoker.class.getName() + '.' + destinationId;
        
        return getServiceInvoker(cache, destination, key);
        
    }

    private ServiceInvoker<?> getServiceInvoker(Map<String, Object> cache, Destination destination, String key) {
        GraniteContext context = GraniteContext.getCurrentInstance();
        synchronized (context.getSessionLock()) {
            ServiceInvoker<?> invoker = (ServiceInvoker<?>)cache.get(key);
            if (invoker == null) {
                SpringServiceContext tideContext = null;
                ServletContext sc = ((HttpGraniteContext)context).getServletContext();
                ApplicationContext springContext = WebApplicationContextUtils.getRequiredWebApplicationContext(sc);
                @SuppressWarnings("unchecked")
                Map<String, Object> beans = springContext.getBeansOfType(SpringServiceContext.class);
                if (beans.size() > 1)
                    throw new RuntimeException("More than one SpringServiceContext bean found");
                else if (beans.size() == 1)
                    tideContext = (SpringServiceContext)beans.values().iterator().next();
                else {
                	// Try to create Spring MVC context when Spring MVC available
                	String className = "org.granite.tide.spring.SpringMVCServiceContext";
                	try {
                		Class<SpringServiceContext> clazz = ClassUtil.forName(className, SpringServiceContext.class);
                		tideContext = clazz.newInstance();
                	}
                	catch (Exception e) {
                		tideContext = new SpringServiceContext();
                	}
                }
                
                String beanName = destination.getProperties().get(PERSISTENCE_MANAGER_BEAN_NAME);
            	tideContext.setPersistenceManagerBeanName(beanName);                
                
                invoker = new TideServiceInvoker<SpringServiceFactory>(destination, this, tideContext);
                cache.put(key, invoker);
            }
            return invoker;
        }

    }
}
