/*
 GRANITE DATA SERVICES
 Copyright (C) 2007-2010 ADEQUATE SYSTEMS SARL

 This file is part of Granite Data Services.

 Granite Data Services is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 3 of the License, or (at your
 option) any later version.

 Granite Data Services is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 for more details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.reflect
{

/**
 * Represents an ActionScript 3 annotation (or metadata) with all its arguments.
 *
 * @see AnnotedElement
 *
 * @author Franck WOLFF
 */
public class Annotation extends DescribedElement
{

    /**
     * @private
     */
    private var _annotatedElement:IAnnotatedElement;

    /**
     * Constructs a new <code>Annotation</code> instance.
     *
     * @param annotatedElement the element (class, variable, method, etc.) to which
     *         this annotation is attached.
     * @param desc the XML description of this annotation.
     */
    function Annotation( annotatedElement:IAnnotatedElement, desc:XML )
    {
        super( annotatedElement.declaredBy, desc );

        _annotatedElement = annotatedElement;
    }

    /**
     * The name of this annotation.
     */
    public function get name():String
    {
        return desc.@name;
    }

    /**
     * The <code>IAnnotatedElement</code> (class, variable, method, etc.) to which
     * this annotation is attached.
     */
    public function get annotatedElement():IAnnotatedElement
    {
        return _annotatedElement;
    }

    /**
     * An array of <code>Arg</code> instances that contains all arguments of this
     * annotation.
     *
     * @see Arg
     */
    public function get args():Array
    {
        var args:Array = new Array();
        for each ( var arg:XML in desc.arg )
            args.push( new Arg( this, arg ) );
        return args;
    }

    /**
     * Tests if this annotation has an argument whose key is equal to the supplied
     * parameter.
     *
     * @param the key (or name) of the argument to check for ("" for the default
     *         argument).
     * @return <code>true</code> if this annotation holds an argurment with this name,
     *         <code>false</code> otherwise.
     */
    public function isArgPresent( key:String = "" ):Boolean
    {
        return desc.arg.(@key == key).length() > 0;
    }

    /**
     * Returns the value of the first argument of this annotation whose key is equal
     * to the supplied parameter.
     *
     * @param the key (or name) of the argument to get value from ("" for the default
     *         argument).
     * @return the argument value, or <code>null</code> if this annotation has no
     *         such argument.
     */
    public function getArgValue( key:String = "" ):String
    {
        var arg:XMLList = desc.arg.(@key == key);
        return (arg.length() > 0 ? arg[0].@value : null);
    }

    /**
     * Returns an array that contains all values of the arguments of this annotation
     * whose key is equal to the supplied parameter.
     *
     * @param the key (or name) of the argument to get value from.
     * @return the argument value, or <code>null</code> if this annotation has no
     *         such argument.
     */
    public function getArgValues( key:String ):Array
    {
        var values:Array = new Array();
        for each ( var arg:XML in desc.arg.(@key == key) )
            values.push( arg.@value );
        return values;
    }

    /**
     * @inheritDoc
     */
    override public function equals( o:* ):Boolean
    {
        if( o === this )
            return true;
        if( !(o is Annotation) )
            return false;
        var a:Annotation = (o as Annotation);
        return _annotatedElement.equals( a._annotatedElement ) && (toString() == a.toString());
    }
}
}