/*
 GRANITE DATA SERVICES
 Copyright (C) 2007-2010 ADEQUATE SYSTEMS SARL

 This file is part of Granite Data Services.

 Granite Data Services is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 3 of the License, or (at your
 option) any later version.

 Granite Data Services is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 for more details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.reflect
{

/**
 * A <code>Field</code> instance represents either a constant, a variable or an
 * accessor (get/set), static or not, of a given ActionScript 3 class (only accessors
 * for interfaces).
 *
 * @see Type#fields
 *
 * @author Franck WOLFF
 */
public class Field extends Member implements IAnnotatedElement, IVisitableElement
{

    /**
     * Constructs a new <code>Field</code> instance.
     *
     * <p>
     * The declaredBy parameter is adjusted in order to match the actual type that
     * declares this field, by introspecting superclasses (or superinterfaces for
     * interface types). Hence, the returned value of the <code>Field.declaredBy</code>
     * accessor may not return the declaredBy parameter value, but one of its
     * superclasses.
     * </p>
     *
     * @param annotatedElement the <code>Type</code> that declares this field.
     * @param desc the XML description of this annotation.
     */
    function Field( declaredBy:Type, desc:XML )
    {
        super( initDeclaredBy( declaredBy, desc ), desc );
    }

    /**
     * The name of this <code>Field</code> instance, ie. the name of the underlying
     * constant, variable or accessor.
     */
    override public function get name():String
    {
        return desc.@name;
    }

    /**
     * The namespace's uri of this field ("" for public fields).
     */
    override public function get uri():String
    {
        var uri:String = desc.@uri;
        // Descriptions of interfaces accessors have uris set to the qualified
        // name of the interface (with a single ':' instead of the usual double '::').
        // Just ignore them.
        if( uri == "" || (isAccessor() && declaredBy.isInterface()) )
            return "";
        return uri;
    }

    /**
     * The type of this <code>Field</code> instance, ie. the type of the underlying
     * constant, variable or accessor.
     */
    public function get type():Type
    {
        return Type.forName( desc.@type, declaredBy.domain );
    }

    /**
     * Returns the value of the field represented by this Field, on the specified instance.
     *
     * @param instance object from which the represented field's value is to be extracted. This
     *         parameter may be <code>null</code> if this field represents a static member.
     * @return the value of the represented field in the instance parameter.
     * @throws org.granite.reflect.IllegalAccessError if this field is not readable.
     * @throws ArgumentError if this field is an instance field and the supplied instance
     *         parameter is <code>null</code>, or if the instance parameter is not an instance of
     *         this field declaring class.
     */
    public function getValue( instance:* ):*
    {

        if( !isReadable() )
            throw new IllegalAccessError( "Field " + qName + " is not readable" );

        if( !isStatic() && instance == null )
            throw new ArgumentError( "Cannot get the non-static field " + qName + " value with a null instance" );

        if( instance == null )
            return declaredBy.getClass()[qName];

        if( !(instance is declaredBy.getClass()) )
            throw new ArgumentError( "Instance parameter is not an instance of the " + qName + " field declaring class" );

        return instance[qName];
    }

    /**
     * Sets the field represented by this Field object on the specified instance argument to
     * the specified new value.
     *
     * @param instance the object whose field should be modified. This parameter may be
     *         <code>null</code> if this field represents a static member.
     * @param value the new value for the field of instance being modified.
     * @throws org.granite.reflect.IllegalAccessError if this field is not writeable.
     * @throws ArgumentError if this field is an instance field and the supplied instance
     *         parameter is <code>null</code>, if the instance parameter is not an instance of
     *         this field declaring class, or if the supplied value cannot be converted to
     *         this field type.
     */
    public function setValue( instance:*, value:* ):void
    {

        if( !isWriteable() )
            throw new IllegalAccessError( "Field " + qName + " is not writeable" );

        if( !isStatic() && instance == null )
            throw new ArgumentError( "Cannot set the non-static field " + qName + " value with a null instance" );

        if( instance == null )
            declaredBy.getClass()[qName] = value;
        else
        {
            if( !(instance is declaredBy.getClass()) )
                throw new ArgumentError( "Instance parameter is not an instance of the " + qName + " field declaring class" );
            instance[qName] = value;
        }
    }

    /**
     * @inheritDoc
     */
    override public function getAnnotations( recursive:Boolean = false, pattern:String = "^_?[^_]" ):Array
    {
        var re:RegExp = new RegExp( (pattern == null ? '' : pattern) );
        var annotations:Array = new Array();

        for each ( var meta:XML in desc.metadata )
        {
            if( meta.@name.toString().search( re ) != -1 )
                annotations.push( new Annotation( this, meta ) );
        }

        if( recursive && isAccessor() && !isStatic() )
        {
            for each ( var superclass:Type in declaredBy.superclasses )
            {
                var f:Field = getSuperField( superclass );
                if( f != null )
                {
                    for each ( var a:Annotation in f.getAnnotations( false, pattern ) )
                        annotations.push( a );
                }
            }
            for each ( var interfaze:Type in declaredBy.interfaces )
            {
                f = getSuperField( interfaze );
                if( f != null )
                {
                    for each ( a in f.getAnnotations( false, pattern ) )
                        annotations.push( a );
                }
            }
        }

        return annotations;
    }

    /**
     * @inheritDoc
     */
    override public function getAnnotation( type:String, recursive:Boolean = false ):Annotation
    {
        var meta:XMLList = desc.metadata.(@name == type);
        if( meta.length() > 0 )
            return new Annotation( this, meta[0] );

        if( recursive && isAccessor() && !isStatic() )
        {
            for each ( var superclass:Type in declaredBy.superclasses )
            {
                var f:Field = getSuperField( superclass );
                if( f != null )
                {
                    var a:Annotation = f.getAnnotation( type, false );
                    if( a != null )
                        return a;
                }
            }
            for each ( var interfaze:Type in declaredBy.interfaces )
            {
                f = getSuperField( interfaze );
                if( f != null )
                {
                    a = f.getAnnotation( type, false );
                    if( a != null )
                        return a;
                }
            }
        }

        return null;
    }

    /**
     * @inheritDoc
     */
    override public function isAnnotationPresent( type:String, recursive:Boolean = false ):Boolean
    {
        if( type == null )
            return true;

        if( XMLList( desc.metadata.(@name == type) ).length() > 0 )
            return true;

        if( recursive && isAccessor() && !isStatic() )
        {
            for each ( var superclass:Type in declaredBy.superclasses )
            {
                var f:Field = getSuperField( superclass );
                if( f != null && f.isAnnotationPresent( type, false ) )
                    return true;
            }
            for each ( var interfaze:Type in declaredBy.interfaces )
            {
                f = getSuperField( interfaze );
                if( f != null && f.isAnnotationPresent( type, false ) )
                    return true;
            }
        }

        return false;
    }

    /**
     * @inheritDoc
     */
    override public function equals( o:* ):Boolean
    {
        if( o === this )
            return true;
        if( !(o is Field) )
            return false;
        var f:Field = (o as Field);
        return declaredBy.equals( f.declaredBy ) && (isStatic() == f.isStatic()) && (toString() == f.toString());
    }

    /**
     * @private
     */
    override protected function initModifiers():uint
    {
        return getModifiers( desc );
    }

    /**
     * @private
     */
    private function getSuperField( type:Type ):Field
    {
        var f:Field = null;
        try
        {
            if( !isStatic() )
                f = type.getInstanceField( name, ns );
        }
        catch ( e:NoSuchFieldError )
        {
        }
        return f;
    }

    /**
     * @private
     */
    private static function initDeclaredBy( declaredBy:Type, desc:XML ):Type
    {

        // Static or instance accessors: just use the @declaredBy attribute.
        if( desc.@declaredBy != undefined )
            return Type.forName( desc.@declaredBy, declaredBy.domain );

        var fname:String = desc.@name;
        var modifiers:uint = getModifiers( desc );

        // We should only have constants and variables here (accessors must have a
        // @declaredBy attribute).
        if( (modifiers & (CONSTANT | VARIABLE)) == 0 )
            throw new ReflectionError( "Could not find declaring class for static field: " + fname );

        var declaringType:Type = declaredBy;

        // Static constants or variables: returns the first found class (by iterating on
        // declaredBy superclasses) that declares this field.
        if( (modifiers & STATIC) != 0 )
        {
            while( declaringType.getClass()[fname] == undefined )
                declaringType = declaringType.superclass;
            if( declaringType == null )
                throw new ReflectionError( "Could not find declaring class for static field: " + fname );
            return declaringType;
        }

        // Instance constants or variables: returns the first found class (by iterating on
        // declaredBy superclasses) that declares this field. Use the fact that the AS3
        // compiler doesn't allow instance variable or constant overriding.
        for each ( var superclass:Type in declaredBy.superclasses )
        {
            var superDesc:XMLList;

            if( desc.name() == "constant" )
                superDesc = superclass.desc.factory.constant.(@name == fname);
            else
                superDesc = superclass.desc.factory.variable.(@name == fname);

            // Only one result is possible (cannot have multiple non static constant/variable with
            // the same name).
            if( superDesc.length() > 0 )
                declaringType = superclass;
        }

        return declaringType;
    }

    /**
     * @private
     */
    private static function getModifiers( desc:XML ):uint
    {
        var modifiers:uint = 0;

        var parent:* = desc.parent();
        if( parent is XML && (parent as XML).name() == "type" )
            modifiers |= STATIC;

        switch( desc.name().toString() )
        {
        case "constant":
            modifiers |= CONSTANT | READABLE;
            break;
        case "variable":
            modifiers |= VARIABLE | READABLE | WRITABLE;
            break;
        case "accessor":
            modifiers |= ACCESSOR;
            switch( desc.@access.toString() )
            {
            case "readonly":
                modifiers |= READABLE;
                break;
            case "writeonly":
                modifiers |= WRITABLE;
                break;
            case "readwrite":
                modifiers |= READABLE | WRITABLE;
                break;
            default:
                throw new ReflectionError( "Unknown field access: " + desc.@access.toString() );
            }
            break;
        default:
            throw new ReflectionError( "Unknown field name: " + desc.name() );
        }

        return modifiers;
    }
}
}