/*
 GRANITE DATA SERVICES
 Copyright (C) 2007-2010 ADEQUATE SYSTEMS SARL

 This file is part of Granite Data Services.

 Granite Data Services is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 3 of the License, or (at your
 option) any later version.

 Granite Data Services is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 for more details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.reflect.visitor
{

import flash.utils.Dictionary;

import org.granite.reflect.Type;
import org.granite.reflect.visitor.handlers.BeanHandler;
import org.granite.reflect.visitor.handlers.CollectionHandler;
import org.granite.reflect.visitor.handlers.DictionaryHandler;
import org.granite.reflect.visitor.handlers.NullHandler;
import org.granite.reflect.visitor.handlers.ObjectHandler;
import org.granite.reflect.visitor.handlers.PrimitiveHandler;
import org.granite.reflect.visitor.handlers.VectorHandler;

public class Guide
{

    public static const RS_DICTIONARY:String = "dictionary";
    public static const RS_STACK:String = "stack";

    private static const _NULL_HANDLER:NullHandler = new NullHandler();

    // Order is important!
    private static const _DEFAULT_HANDLERS:Array = [
        new PrimitiveHandler(),
        new CollectionHandler(),
        new VectorHandler(),
        new DictionaryHandler(),
        new ObjectHandler(),
        new BeanHandler()
    ];

    public static function get defaultHandlers():Array
    {
        return _DEFAULT_HANDLERS;
    }

    private var _visitor:IVisitor;
    private var _handlers:Array;
    private var _recursionStrategy:String;
    private var _context:Object;

    function Guide( visitor:IVisitor, handlers:Array = null, recursionStrategy:String = RS_DICTIONARY )
    {

        // Check and save visitor
        if( visitor == null )
            throw new ArgumentError( "Parameter visitor cannot be null" );
        _visitor = visitor;

        // Check and configure handlers
        _handlers = new Array();
        _handlers.push( _NULL_HANDLER );
        if( handlers != null )
        {
            for each ( var handler:IHandler in handlers )
                _handlers.push( handler );
        }
        for each ( handler in _DEFAULT_HANDLERS )
            _handlers.push( handler );

        // Check and save recursion startegy
        if( [RS_DICTIONARY, RS_STACK].indexOf( recursionStrategy ) == -1 )
            throw new ArgumentError( "Unknown recursion startegy type: " + recursionStrategy );
        _recursionStrategy = recursionStrategy;
    }

    public function get visitor():IVisitor
    {
        return _visitor;
    }

    public function get handlers():Array
    {
        return _handlers;
    }

    public function get recursionStrategy():String
    {
        return _recursionStrategy;
    }

    public function explore( o:* ):void
    {
        if( o == null || o == undefined )
            return;

        exploreVisitable( new Visitable( null, Type.forInstance( o ), o ) );
    }

    public function exploreVisitable( visitable:Visitable ):void
    {
        if( visitable == null )
            return;

        if( _context != null )
            throw new Error( "Guide instance is currently in use" );

        if( _recursionStrategy == RS_DICTIONARY )
            _context = new Dictionary();
        else
            _context = new Array();

        try
        {
            if( _visitor.accept( visitable ) )
                accept( _visitor, visitable );
        }
        catch ( e:InterruptVisitError )
        {
            // visit interrupted by visitor.
        }
        finally
        {
            _context = null;
        }
    }

    /**
     * Checks if the supplied parameter is already in the recursion context (ie. already
     * visited).
     *
     * @param value the value to check against the current recursion context.
     * @return <code>true</code> if value is already in the recursion context, <code>false</code>
     *         otherwise.
     */
    protected function isInContext( value:* ):Boolean
    {
        if( _context is Dictionary )
            return (value in (_context as Dictionary));
        return ((_context as Array).indexOf( value ) != 1);
    }

    /**
     * Put the supplied parameter in the recursion context (so it is marked as already visited).
     *
     * @param value the value to put in the recursion context.
     */
    protected function pushInContext( value:* ):void
    {
        if( _context is Dictionary )
            (_context as Dictionary)[value] = true;
        else
            (_context as Array).push( value );
    }

    /**
     * Remove the supplied parameter from the recursion context (this method does nothing when
     * the recursion strategy is "dictionary").
     *
     * @param value the value to remove from the recursion context.
     */
    protected function popFromContext( value:* ):void
    {
        if( _context is Array )
            (_context as Array).pop();
    }

    /**
     * Try to find a <code>IHandler</code> instance that will handle the visit of the visitable
     * item by the visitor object.
     *
     * @param visitor the <code>IVisitor</code> that will visit the item.
     * @param visitable the <code>Visitable</code> item to visit.
     */
    protected function accept( visitor:IVisitor, visitable:Visitable ):void
    {

        function filter( visitable:Visitable ):Boolean
        {
            return visitor.accept( visitable );
        }

        if( visitor.visit( visitable ) && !isInContext( visitable.value ) )
        {

            pushInContext( visitable.value );
            try
            {
                var children:Array = null;

                for each ( var handler:IHandler in _handlers )
                {
                    if( handler.canHandle( visitable ) )
                    {
                        children = handler.handle( visitable, filter );
                        break;
                    }
                }

                if( children != null && children.length > 0 )
                {
                    for each ( var child:Visitable in children )
                        accept( visitor, child );
                }
            }
            finally
            {
                popFromContext( visitable.value );
            }
        }
    }
}
}