/*
 GRANITE DATA SERVICES
 Copyright (C) 2007-2008 ADEQUATE SYSTEMS SARL

 This file is part of Granite Data Services.

 Granite Data Services is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 3 of the License, or (at your
 option) any later version.

 Granite Data Services is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 for more details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.tide
{

import flash.system.ApplicationDomain;
import flash.events.Event;
import flash.events.EventDispatcher;
import flash.events.TimerEvent;
import flash.errors.IllegalOperationError;
import flash.net.LocalConnection;
import flash.display.DisplayObject;
import flash.display.DisplayObjectContainer;
import flash.display.LoaderInfo;
import flash.utils.Dictionary;
import flash.utils.Proxy;
import flash.utils.Timer;
import flash.utils.describeType;
import flash.utils.flash_proxy;
import flash.utils.IExternalizable;
import flash.utils.getDefinitionByName;
import flash.utils.getQualifiedClassName;

import mx.managers.SystemManager;
import mx.utils.ObjectUtil;
import mx.utils.StringUtil;
import mx.binding.BindabilityInfo;
import mx.binding.utils.BindingUtils;
import mx.collections.ArrayCollection;
import mx.collections.IList;
import mx.controls.Alert;
import mx.core.mx_internal;
import mx.core.Application;
import mx.core.IUIComponent;
import mx.controls.Alert;
import mx.events.PropertyChangeEvent;
import mx.logging.ILogger;
import mx.logging.Log;
import mx.messaging.config.ServerConfig;
import mx.messaging.events.ChannelFaultEvent;
import mx.messaging.events.MessageEvent;
import mx.messaging.messages.AsyncMessage;
import mx.messaging.messages.ErrorMessage;
import mx.rpc.AbstractOperation;
import mx.rpc.AsyncToken;
import mx.rpc.Fault;
import mx.rpc.events.FaultEvent;
import mx.rpc.events.InvokeEvent;
import mx.rpc.events.ResultEvent;
import mx.rpc.remoting.mxml.Operation;
import mx.rpc.remoting.mxml.RemoteObject;
import mx.utils.DescribeTypeCache;
import mx.utils.DescribeTypeCacheRecord;
import mx.utils.ObjectProxy;
import mx.utils.ObjectUtil;

import ru.etcs.utils.getDefinitionNames;

import org.granite.tide.IComponent;
import org.granite.tide.collections.PersistentCollection;
import org.granite.tide.collections.PersistentMap;
import org.granite.tide.validators.ValidatorResponder;
import org.granite.tide.service.IServiceInitializer;
import org.granite.tide.service.DefaultServiceInitializer;
import org.granite.tide.events.TideEvent;
import org.granite.tide.events.TidePluginEvent;
import org.granite.tide.events.TideFaultEvent;
import org.granite.tide.events.TideResultEvent;
import org.granite.tide.events.TideContextEvent;


[Bindable]
/**
 *     Tide is the base implementation of the Tide application manager singleton
 *
 *     @author William DRAI
 */
public class Tide extends EventDispatcher
{

    private static var log:ILogger = Log.getLogger( "org.granite.tide.Tide" );

    protected static const DEFAULT_CONTEXT:String = "__DEFAULT_CONTEXT__";

    public static const PLUGIN_ADD_COMPONENT:String = "org.granite.tide.plugin.addComponent";
    public static const PLUGIN_SET_CREDENTIALS:String = "org.granite.tide.plugin.setCredentials";
    public static const PLUGIN_LOGIN_SUCCESS:String = "org.granite.tide.plugin.loginSuccess";
    public static const PLUGIN_LOGIN_FAULT:String = "org.granite.tide.plugin.loginFault";
    public static const PLUGIN_LOGOUT:String = "org.granite.tide.plugin.logout";

    public static const SCOPE_UNKNOWN:int = 0;
    public static const SCOPE_SESSION:int = 1;
    public static const SCOPE_CONVERSATION:int = 2;

    public static const RESTRICT_UNKNOWN:int = 0;
    public static const RESTRICT_NO:int = 1;
    public static const RESTRICT_YES:int = 2;

    public static const CONTEXT_CREATE:String = "org.granite.tide.contextCreate";
    public static const CONTEXT_DESTROY:String = "org.granite.tide.contextDestroy";
    public static const CONTEXT_RESULT:String = "org.granite.tide.contextResult";
    public static const CONTEXT_FAULT:String = "org.granite.tide.contextFault";

    public static const STARTUP:String = "org.granite.tide.startup";
    public static const LOGIN:String = "org.granite.tide.login";
    public static const LOGOUT:String = "org.granite.tide.logout";
    public static const LOGGED_OUT:String = "org.granite.tide.loggedOut";

    public static const CONVERSATION_TAG:String = "conversationId";
    public static const CONVERSATION_PROPAGATION_TAG:String = "conversationPropagation";
    public static const IS_LONG_RUNNING_CONVERSATION_TAG:String = "isLongRunningConversation";
    public static const WAS_LONG_RUNNING_CONVERSATION_ENDED_TAG:String = "wasLongRunningConversationEnded";
    public static const WAS_LONG_RUNNING_CONVERSATION_CREATED_TAG:String = "wasLongRunningConversationCreated";
    public static const IS_FIRST_CALL_TAG:String = "org.granite.tide.isFirstCall";
    public static const IS_FIRST_CONVERSATION_CALL_TAG:String = "org.granite.tide.isFirstConversationCall";


    private static var _tide:Tide;

    private var _destination:String = null;

    private var _contextClass:Class = BaseContext;
    private var _componentClass:Class;

    private var _modules:Dictionary = new Dictionary();
    private var _moduleApplicationDomain:Dictionary = new Dictionary( true );
    private var _moduleComponents:Dictionary = new Dictionary( true );
    private var _componentDescriptors:Dictionary = new Dictionary();
    private var _observersByType:Dictionary = new Dictionary();
    private var _implementationsByType:Dictionary = new Dictionary();
    private var _injectionPointsByType:Dictionary = new Dictionary();
    private var _entityDescriptors:Dictionary = new Dictionary( true );
    private var _applicationDomains:Dictionary = new Dictionary( true );

    private var _sessionId:String = null;
    private var _firstCall:Boolean = true;

    protected var _ro:RemoteObject = null;
    protected var _roInitialize:RemoteObject = null;
    protected var _rosByDestination:Dictionary = new Dictionary();

    private var _currentContextId:uint = 1;

    private var _ctx:Dictionary = new Dictionary();
    private var _contextsToDestroy:Array = new Array();
    private var _initializing:Boolean = true;
    private var _logoutInProgress:Boolean = false;
    private var _waitForLogout:int = 0;

    private var _exceptionHandlers:Array = new Array();

    private var _registeredListeners:ArrayCollection = new ArrayCollection();
    private var _newListeners:ArrayCollection = new ArrayCollection();

    private var _currentDomain:ApplicationDomain = ApplicationDomain.currentDomain;
    private var _currentModulePrefix:String = "";

    public static var showBusyCursor:Boolean = true;
    public var busy:Boolean = false;
    public var disconnected:Boolean = false;


    public function Tide( destination:String )
    {
        log.info( "Initializing Tide proxy" );

        _destination = destination;

        DescribeTypeCache.registerCacheHandler( "bindabilityInfo", bindabilityInfoHandler );
        DescribeTypeCache.registerCacheHandler( "componentInfo", componentInfoHandler );
        DescribeTypeCache.registerCacheHandler( "entityInfo", entityInfoHandler );

        init( Context, null );

        initDefaultContext();

        log.info( "Tide proxy initialized" );
    }

    /**
     *    @private
     *     Init context and component default types
     */
    protected function init( contextClass:Class, componentClass:Class ):void
    {
        _contextClass = contextClass;
        _componentClass = componentClass;
        addComponent( "meta_dirty", Boolean );
    }

    protected function get ro():RemoteObject
    {
        if( _ro == null )
            initRemoteObject();
        return _ro;
    }

    protected function get roInitialize():RemoteObject
    {
        if( _roInitialize == null )
            initRemoteObject();
        return _roInitialize;
    }

    /**
     *     @private
     *  Init RemoteObject
     */
    protected function initRemoteObject():void
    {
    }

    /**
     *     @private
     *  Create RemoteObject
     *
     *     @param destination destination
     *  @param concurrency remote object concurrenty
     *  @return internal RemoteObject
     */
    protected function createRemoteObject( destination:String = null, concurrency:String = "multiple" ):RemoteObject
    {
        var ro:RemoteObject = new RemoteObject( destination != null ? destination : _destination );
        var serviceInitializer:IServiceInitializer = IServiceInitializer( getContext().byType( IServiceInitializer ) );
        if( serviceInitializer != null )
            serviceInitializer.initialize( ro );
        else
            ro.destination = destination != null ? destination : _destination;
        ro.concurrency = concurrency;
        ro.showBusyCursor = showBusyCursor;
        ro.makeObjectsBindable = true;
        return ro;
    }


    /**
     *     @private
     *  Create Operation
     *
     *  @param name operation name
     *
     *  @return internal operation
     */
    public function createOperation( name:String, ro:RemoteObject = null ):TideOperation
    {
        var op:TideOperation = new TideOperation( this, ro );
        op.name = name;
        return op;
    }


    /**
     *  RemoteObject destination used
     */
    public function get destination():String
    {
        return _destination;
    }

    /**
     *    Factory for current instance Tide singleton
     *
     *    @param destination default destination (not used here)
     *  @return current singleton instance
     */
    public static function getInstance( destination:String = null, tideClass:Class = null ):Tide
    {
        if( !_tide )
        {
            if( tideClass == null )
                _tide = new Tide( destination );
            else
                _tide = new tideClass( destination );
        }
        return _tide;
    }

    /**
     *    Clear Tide singleton (should be used only for testing)
     */
    public static function resetInstance():void
    {
        if( _tide )
        {
            _tide.destroyContexts( true );
            _tide.resetApplication();
        }
        _tide = null;
    }

    /**
     *  @private
     *     Init default context
     */
    protected function initDefaultContext():BaseContext
    {
        _ctx = new Dictionary();
        var ctx:BaseContext = new _contextClass( this, null );

        _ctx[DEFAULT_CONTEXT] = ctx;
        return ctx;
    }

    /**
     *    Return a context from its id
     *
     *     @param contextId context id
     *  @param create should create when not existing
     *  @return context
     */
    public function getContext( contextId:String = null, parentContextId:String = null, create:Boolean = true ):BaseContext
    {
        var ctx:BaseContext = _ctx[contextId != null ? contextId : DEFAULT_CONTEXT];
        if( ctx == null && create )
        {
            var parentCtx:BaseContext = parentContextId == null ? getContext() : BaseContext( _ctx[parentContextId] );
            ctx = new _contextClass( this, parentCtx );
            ctx.meta_init( contextId );
            _ctx[contextId != null ? contextId : DEFAULT_CONTEXT] = ctx;
            ctx.raiseEvent( CONTEXT_CREATE );
        }
        return ctx;
    }

    /**
     *  @private
     *  Create a new context if it does not exist
     *
     *  @param contextId the requested context id
     *  @return the context
     */
    public function newContext( contextId:String = null, parentContextId:String = null ):BaseContext
    {
        var ctx:BaseContext = contextId != null ? _ctx[contextId] as BaseContext : null;
        if( ctx != null && ctx.meta_finished )
        {
            ctx.meta_clear();
            delete _ctx[contextId];
            removeFromContextsToDestroy( contextId );
            ctx = null;
        }
        if( ctx == null )
        {
            var parentCtx:BaseContext = parentContextId != null ? _ctx[parentContextId] as BaseContext : getContext();
            ctx = new _contextClass( this, parentCtx );
            ctx.meta_init( contextId );
            if( contextId != null )
                _ctx[contextId] = ctx;
            ctx.raiseEvent( CONTEXT_CREATE );
        }
        return ctx;
    }


    /**
     *     @private
     *     Destroy a context
     *
     *  @param contextId context id
     *  @param force force complete destruction of context
     */
    public function destroyContext( contextId:String, force:Boolean = false ):void
    {
        var ctx:BaseContext = contextId != null ? _ctx[contextId] : null;
        if( ctx != null )
        {
            // Destroy child contexts
            for each ( var c:BaseContext in _ctx )
            {
                if( c.meta_parentContext === ctx )
                    destroyContext( c.contextId, force );
            }

            removeFromContextsToDestroy( contextId );
            ctx.raiseEvent( CONTEXT_DESTROY );
            _ctx[contextId].meta_clear( force );
            delete _ctx[contextId];
        }
    }


    /**
     *     Returns the list of conversation contexts
     *
     *  @return conversation contexts
     */
    public function getAllContexts():Array
    {
        var contexts:Array = new Array();
        for each ( var ctx:BaseContext in _ctx )
        {
            if( !ctx.meta_isGlobal() )
                contexts.push( ctx );
        }
        return contexts;
    }


    /**
     *     Execute a function for each conversation context
     *
     *  @param parentContext parent context
     *  @param callback callback function
     *  @param token token passed to the function
     */
    public function forEachChildContext( parentContext:BaseContext, callback:Function, token:Object = null ):void
    {
        for each ( var ctx:BaseContext in _ctx )
        {
            if( ctx.meta_parentContext == parentContext )
            {
                if( token )
                    callback( ctx, token );
                else
                    callback( ctx );
            }
        }
    }


    /**
     *     @private
     *     Current sessionId
     *
     *     @return sessionId
     */
    public function get sessionId():String
    {
        return _sessionId;
    }

    /**
     *     @private
     *     Is it the first remote call ?
     *
     *     @return is first call
     */
    public function get firstCall():Boolean
    {
        return _firstCall;
    }

    /**
     *  @private
     *  Name of the current module in which observers and injections are handled
     *
     *  @return module prefix
     */
    public function get currentModulePrefix():String
    {
        return _currentModulePrefix;
    }

    public function set currentModulePrefix( prefix:String ):void
    {
        _currentModulePrefix = prefix;
    }

    /**
     *  @private
     *  Name of the current namespace in which observers and injections are handled
     *
     *  @return namespace
     */
    public function get currentNamespace():String
    {
        return _currentModulePrefix ? _currentModulePrefix.substring( 0, _currentModulePrefix.length - 1 ) : _currentModulePrefix;
    }


    /**
     *     Register a plugin instance
     *
     *  @param plugin the plugin instance
     */
    public function addPlugin( plugin:ITidePlugin ):void
    {
        plugin.tide = this;
    }


    private function currentApplication():Object
    {
        var app:Object = Application.application;
        if( app == null )
        {
            // Flex 4 spark application
            var flexGlobals:Class = getDefinitionByName( "mx.core.FlexGlobals" ) as Class;
            app = flexGlobals.topLevelApplication;
        }
        return app;
    }


    /**
     *     Init the Tide application
     *
     *     @param autoRegisterUIComponents enable automatic registration/unregistration of annotated UI components
     */
    public function initApplication( autoRegisterUIComponents:Boolean = true ):void
    {
        var ctx:BaseContext = getContext();
        var app:Object = currentApplication();
        ctx.application = app;
        if( autoRegisterUIComponents )
        {
            app.addEventListener( Event.ADDED, addedHandler, 0, false, true );
            app.systemManager.addEventListener( Event.ADDED, addedHandler, 0, false, true );
            app.addEventListener( Event.REMOVED, removedHandler, 0, false, true );
            app.systemManager.addEventListener( Event.REMOVED, removedHandler, 0, false, true );
        }

        ctx.raiseEvent( STARTUP );
    }

    protected function resetApplication():void
    {
        var app:Object = currentApplication();
        app.removeEventListener( Event.ADDED, addedHandler );
        app.systemManager.removeEventListener( Event.ADDED, addedHandler );
        app.removeEventListener( Event.REMOVED, removedHandler );
        app.systemManager.removeEventListener( Event.REMOVED, removedHandler );
    }


    /**
     *     @private
     *     Check is the UI component is registered in any managed context
     *
     *     @param uiComponent an UI component
     *     @return the context which manages the UI component or null if not found
     */
    public function findContextForUIComponent( uiComponent:IUIComponent ):BaseContext
    {
        var defaultContext:BaseContext = _ctx[DEFAULT_CONTEXT];
        if( defaultContext.meta_listensTo( uiComponent ) )
            return defaultContext;

        for each ( var ctx:BaseContext in _ctx )
        {
            if( ctx === defaultContext )
                continue;
            if( ctx.meta_listensTo( uiComponent ) )
                return ctx;
        }

        return null;
    }

    /**
     *     @private
     *
     *     Unregister listeners for component in all contexts
     *
     *  @param name component name
     *  @param component component
     */
    public function unregisterComponent( name:String, component:Object ):void
    {
        for each ( var ctx:BaseContext in _ctx )
            ctx.meta_unregisterComponent( name, component );
    }


    /**
     *     @private
     *
     *     Get implementation for type
     *
     *  @param type class required at injection point
     *  @return name of implementation component
     */
    public function getImplementation( type:Class ):String
    {
        var typeName:String = getQualifiedClassName( type );
        var implName:String = _implementationsByType[typeName];
        return implName ? implName : null;
    }

    /**
     *     @private
     *
     *     Register a component implementation for a type
     *
     *  @param typeName type class name
     *  @param name of implementation component
     */
    private function registerImplementation( typeName:String, name:String ):void
    {
        if( isFrameworkType( typeName ) )
            return;

        var implName:String = _implementationsByType[typeName] as String;
        if( implName == null )
            _implementationsByType[typeName] = name;
        else if( name != implName )
            log.warn( "Many implementations for type " + typeName + ", ignored new implementation " + name );
    }

    private function unregisterImplementation( name:String ):void
    {
        var typesToRemove:Array = new Array();
        for( var typeName:String in _implementationsByType )
        {
            if( _implementationsByType[typeName] == name )
                typesToRemove.push( typeName )
        }
        for each ( typeName in typesToRemove )
            delete _implementationsByType[typeName];
    }

    /**
     *     @private
     *
     *     Get injection points for type
     *
     *  @param type type required at injection point
     *  @return array of injection points [ componentName, propertyName ]
     */
    public function getInjectionPoints( typeName:String ):Array
    {
        var injectionPoints:Array = _injectionPointsByType[typeName];
        return injectionPoints ? injectionPoints : null;
    }

    /**
     *     @private
     *
     *     Register an injection point for a type
     *
     *  @param typeName type class name
     *  @param componentName target component name
     *  @param propertyName target property name
     */
    private function registerInjectionPoint( typeName:String, componentName:String, propertyName:String ):void
    {
        if( isFrameworkType( typeName ) )
            return;

        var ips:Array = _injectionPointsByType[typeName] as Array;
        if( ips == null )
        {
            ips = new Array();
            _injectionPointsByType[typeName] = ips;
        }
        var found:Boolean = false;
        for each ( var ip:Array in ips )
        {
            if( ip[0] == componentName && ip[1] == propertyName )
            {
                found = true;
                break;
            }
        }
        if( !found )
            ips.push( [ componentName, propertyName ] );
    }

    private function unregisterInjectionPoints( componentName:String ):void
    {
        var typeNamesToDelete:Array = new Array();
        var typeName:String;
        for( typeName in _injectionPointsByType )
        {
            var ips:Array = _injectionPointsByType[typeName];
            for( var j:int = 0; j < ips.length; j++ )
            {
                if( ips[j][0] == componentName )
                {
                    ips.splice( j, 1 );
                    j--;
                }
                if( ips.length == 0 )
                    typeNamesToDelete.push( typeName );
            }
        }
        for each ( typeName in typeNamesToDelete )
            delete _injectionPointsByType[typeName];
    }

    private function isFrameworkType( typeName:String ):Boolean
    {
        return isFxClass( typeName ) || typeName == "Object" || typeName == "org.granite.tide::IComponent" || typeName == "org.granite.tide::IPropertyHolder";
    }


    /**
     *     @private
     *
     *     Get array of types for type (class, superclass, interfaces)
     *
     *  @param description describeType for the type
     *  @return types
     */
    public function getComponentTypes( description:XML ):Array
    {
        var types:Array = [ description.@name.toXMLString() ];
        for each ( var ec:XML in description.factory.extendsClass )
            types.push( ec.@type.toXMLString() );
        for each ( var ii:XML in description.factory.implementsInterface )
            types.push( ii.@type.toXMLString() );

        return types.filter( function( item:*, index:int, array:Array ):Boolean
        {
            return !isFrameworkType( String( item ) );
        } );
    }


    /**
     *     @private
     *
     *  Check if a class name is a Flex framework class
     *
     *  @param className class name
     *  @return true if class package name is a Flex one
     */
    private function isFxClass( className:String ):Boolean
    {
        return className.substring( 0, 3 ) == 'mx.'
                || className.substring( 0, 6 ) == 'flash.'
                || className.substring( 0, 7 ) == 'flashx.'
                || className.substring( 0, 6 ) == 'spark.'
                || className.substring( 0, 4 ) == 'air.'
                || className.substring( 0, 3 ) == 'fl.';
    }

    /**
     *     @private
     *     Internal handler for ADDED events that scans UI components for [Name] annotations
     *
     *     @param event the ADDED event
     */
    private function addedHandler( event:Event ):void
    {
        internalAdd( event.target );
    }

    private function internalAdd( component:Object ):void
    {
        var className:String = getQualifiedClassName( component );
        if( !isFxClass( className ) )
        {
            var info:ComponentInfo = DescribeTypeCache.describeType( component )['componentInfo'] as ComponentInfo;
            if( info.name != null )
            {
                var name:String = null;
                if( info.name.length > 0 )
                    name = info.name;
                else if( info.module.length > 0 )
                    name = info.module + "." + (component.id ? component.id : component.name);
                else
                    name = component.id ? component.id : component.name;

                var saveModulePrefix:String = _currentModulePrefix;
                _currentModulePrefix = "";

                var ctx:BaseContext = findContext( component as IUIComponent );
                if( (info.scope != "conversation" && ctx.meta_isGlobal())
                        || (info.scope == "conversation" && !ctx.meta_isGlobal()) )
                {

                    var instance:Object = ctx.meta_getInstance( name, false, true );
                    if( instance !== component )
                    {
                        ctx[name] = component;
                        log.info( "added component " + name );
                    }
                }

                _currentModulePrefix = saveModulePrefix;
            }
        }

        if( component is DisplayObjectContainer )
        {
            try
            {
                for( var i:uint = 0; i < component.numChildren; i++ )
                    internalAdd( component.getChildAt( i ) );
            }
            catch ( e:SecurityError )
            {
                // Stop here: component does not allow access to its children
            }
        }
    }

    /**
     *     @private
     *     Internal handler for REMOVED events that unregister UI components from the context
     *
     *     @param event the REMOVED event
     */
    private function removedHandler( event:Event ):void
    {
        internalRemove( event.target );
    }

    private function internalRemove( component:Object ):void
    {
        if( component is DisplayObjectContainer )
        {
            try
            {
                for( var i:uint = 0; i < component.numChildren; i++ )
                    internalRemove( component.getChildAt( i ) );
            }
            catch ( e:SecurityError )
            {
                // Stop here: component does not allow access to its children
            }
        }

        var className:String = getQualifiedClassName( component );
        if( isFxClass( className ) )
            return;	// Ignore framework components

        var info:ComponentInfo = DescribeTypeCache.describeType( component )['componentInfo'] as ComponentInfo;
        if( info.name != null )
        {
            var name:String = null;
            var autoName:Boolean = false;
            if( info.name.length > 0 )
                name = info.name;
            else if( info.module.length > 0 )
            {
                autoName = true;
                name = info.module + "." + (component.id ? component.id : component.name);
            }
            else
            {
                autoName = true;
                name = component.id ? component.id : component.name;
            }

            var saveModulePrefix:String = _currentModulePrefix;
            _currentModulePrefix = "";

            if( autoName )
            {
                removeComponent( name );
                log.info( "removed component " + name );
            }
            else
            {
                var ctx:BaseContext = findContext( component as IUIComponent, false );
                if( ctx != null )
                {
                    var instance:Object = ctx.meta_getInstance( name, false, true );
                    if( instance !== null )
                    {
                        ctx[name] = null;
                        log.info( "removed component instance " + name );
                    }
                }
            }

            _currentModulePrefix = saveModulePrefix;
        }
    }


    public function scanComponents( packagePrefix:String, loaderInfo:LoaderInfo = null, appDomain:ApplicationDomain = null ):void
    {
        if( loaderInfo == null )
            loaderInfo = SystemManager.getSWFRoot( this ).loaderInfo;
        if( appDomain == null )
            appDomain = _currentDomain;

        var allTypes:Array = getDefinitionNames( loaderInfo );
        var types:Array = new Array();
        for( var i:int = 0; i < allTypes.length; i++ )
        {
            var className:String = allTypes[i];
            if( isFxClass( className )
                    || className.indexOf( packagePrefix ) != 0 )
                continue;

            var clazz:Class = appDomain.getDefinition( className ) as Class;
            var names:XMLList = describeType( clazz )..factory..metadata.(@name == 'Name');
            if( names.length() > 0 )
                if( names[0]..arg.(@key == '').@value.toXMLString() )
                    types.push( clazz );
        }
        if( types.length > 0 )
            addComponents( types, appDomain );
    }


    /**
     *     @private
     *     Find the context where the uiComponent is managed
     *
     *     @param uiComponent an UI component
     *     @param returnDefault also search in default context
     *
     *     @return the context managing the uiComponent
     */
    private function findContext( uiComponent:IUIComponent, returnDefault:Boolean = true ):BaseContext
    {
        var ctx:BaseContext = findContextForUIComponent( uiComponent );
        if( ctx != null )
            return ctx;

        if( uiComponent.parent != null && uiComponent.parent is IUIComponent )
            return findContext( uiComponent.parent as IUIComponent );

        return returnDefault ? _ctx[DEFAULT_CONTEXT] : null;
    }


    private var _moduleInitializing:ITideModule = null;

    /**
     *     Register a Tide module
     *
     *     @param moduleClass the module class (that must implement ITideModule)
     *     @param appDomain the Flex application domain for modules loaded dynamically
     */
    public function addModule( moduleClass:Class, appDomain:ApplicationDomain = null ):void
    {
        var saveDomain:ApplicationDomain = _currentDomain;
        if( appDomain != null )
            _currentDomain = appDomain;

        try
        {
            var module:ITideModule = _modules[moduleClass];
            if( module == null )
            {
                module = new moduleClass() as ITideModule;
                _modules[moduleClass] = module;
                _moduleInitializing = module;
                _moduleComponents[module] = new Array();
                if( appDomain != null )
                    _moduleApplicationDomain[module] = appDomain;

                try
                {
                    if( Object( module ).hasOwnProperty( "moduleName" ) )
                        addSubcontext( Object( module ).moduleName );

                    module.init( this );
                }
                catch ( e:Error )
                {
                    log.error( "Module " + moduleClass + " initialization failed", e );
                    throw e;
                }
                _moduleInitializing = null;
            }
            else
                throw new Error( "Module " + moduleClass + " already added" );
        }
        finally
        {
            _currentDomain = saveDomain;
        }
    }

    /**
     *     Unregister a Tide module
     *
     *     @param moduleClass the module class (that must implement ITideModule)
     */
    public function removeModule( moduleClass:Class ):void
    {
        var module:ITideModule = _modules[moduleClass];
        if( module == null )
            throw new Error( "Module " + moduleClass + " not found" );

        var componentNames:Array = _moduleComponents[module] as Array;
        for each ( var name:String in componentNames )
            removeComponent( name );

        var appDomain:ApplicationDomain = _moduleApplicationDomain[module];

        delete _moduleApplicationDomain[module];
        delete _moduleComponents[module];
        delete _modules[moduleClass];

        if( appDomain != null )
            removeApplicationDomain( appDomain );
    }

    /**
     *  @private
     *
     *     Unregister all Tide components from a Flex application domain
     *
     *  @param applicationDomain Flex application domain to clear
     *  @param withModules also removes all components from modules which are defined in the app domain
     */
    private function removeApplicationDomain( applicationDomain:ApplicationDomain, withModules:Boolean = true ):void
    {
        if( _applicationDomains[applicationDomain] == null )
            throw new Error( "ApplicationDomain " + applicationDomain + " is not managed by Tide" );

        for each ( var desc:ComponentDescriptor in _componentDescriptors )
        {
            if( desc.applicationDomain === applicationDomain )
                removeComponent( desc.name );
        }

        for( var moduleClass:Object in _modules )
        {
            if( _moduleApplicationDomain[_modules[moduleClass]] === applicationDomain )
                removeModule( Class( moduleClass ) );
        }

        delete _applicationDomains[applicationDomain];
    }

    /**
     *     Register a Tide component with static dependency injections
     *  Dynamic dependency injections are also scanned
     *
     *     @param name component name
     *     @param type component class
     *  @param properties a factory object containing values to inject in the component instance
     *  @param inConversation true if the component is conversation-scoped
     *  @param autoCreate true if the component needs to be automatically instantiated
     *  @param restrict true if the component needs to be cleared when the user is not logged in
     *  @param appDomain the Flex application domain for classes loaded dynamically from Flex modules
     *  @param overrideIfPresent allows to override an existing component definition (should normally only be used internally)
     */
    public function addComponentWithFactory( name:String, type:Class, properties:Object, inConversation:Boolean = false, autoCreate:Boolean = true, restrict:int = RESTRICT_UNKNOWN, appDomain:ApplicationDomain = null, overrideIfPresent:Boolean = true ):void
    {
        internalAddComponent( name, type, properties, inConversation, autoCreate, restrict, appDomain, overrideIfPresent );
    }

    /**
     *     Register a Tide component with a specified definition and scan dynamic dependency injections
     *
     *     @param name component name
     *     @param type component class
     *  @param inConversation true if the component is conversation-scoped
     *  @param autoCreate true if the component needs to be automatically instantiated
     *  @param restrict true if the component needs to be cleared when the user is not logged in
     *  @param appDomain the Flex application domain for classes loaded dynamically from Flex modules
     *  @param overrideIfPresent allows to override an existing component definition (should normally only be used internally)
     */
    public function addComponent( name:String, type:Class, inConversation:Boolean = false, autoCreate:Boolean = true, restrict:int = RESTRICT_UNKNOWN, appDomain:ApplicationDomain = null, overrideIfPresent:Boolean = true ):void
    {
        internalAddComponent( name, type, null, inConversation, autoCreate, restrict, appDomain, overrideIfPresent );
    }

    /**
     *     Register a Tide namespace
     *
     *     @param name component name
     */
    public function addSubcontext( name:String ):void
    {
        if( !isComponent( name ) || getDescriptor( name ).factory == null || getDescriptor( name ).factory.type !== Subcontext )
            internalAddComponent( name, Subcontext, null );
    }


    private static const VOID_TYPES:Array = ["Object", "void", "*"];

    private static function isInterface( desc:XML ):Boolean
    {
        return VOID_TYPES.indexOf( desc.@name.toXMLString() ) == -1 && desc.factory.extendsClass.length() == 0;
    }


    /**
     *  @private
     *     Internal implementation of component registration
     *
     *     @param name component name
     *     @param type component class
     *  @param properties a factory object containing values to inject in the component instance
     *  @param inConversation true if the component is conversation-scoped
     *  @param autoCreate true if the component needs to be automatically instantiated
     *  @param restrict true if the component needs to be cleared when the user is not logged in
     *  @param appDomain the Flex application domain for classes loaded dynamically from Flex modules
     *  @param overrideIfPresent allows to override an existing component definition (should normally only be used internally)
     */
    private function internalAddComponent( name:String, type:Class, properties:Object, inConversation:Boolean = false, autoCreate:Boolean = true, restrict:int = RESTRICT_UNKNOWN, appDomain:ApplicationDomain = null, overrideIfPresent:Boolean = true ):void
    {
        var factory:ComponentFactory = new ComponentFactory( type, properties );
        var description:XML = factory.describe();
        if( isInterface( description ) )
            throw new Error( "Cannot register interface " + type + " as component class" );

        var isGlobal:Boolean = (type == Subcontext
                || description..factory..implementsInterface.(@type == 'org.granite.tide::IComponent').length() > 0
                || description..factory..metadata.(@name == 'Name')..arg.(@key == 'global').@value.toXMLString() == 'true');

        if( _moduleInitializing != null && name.indexOf( "." ) < 0
                && Object( _moduleInitializing ).hasOwnProperty( "moduleName" ) && !isGlobal )
            name = Object( _moduleInitializing ).moduleName + "." + name;

        if( !isGlobal )
        {
            var idx:int = name.lastIndexOf( "." );
            if( idx > 0 )
                addSubcontext( name.substring( 0, idx ) );
        }

        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        if( descriptor != null && !overrideIfPresent )
            return;

        if( descriptor == null )
            descriptor = getDescriptor( name );

        if( _moduleInitializing != null )
        {
            if( _moduleComponents[_moduleInitializing].indexOf( name ) < 0 )
                _moduleComponents[_moduleInitializing].push( name );
        }

        descriptor.factory = factory;
        descriptor.global = isGlobal;
        descriptor.scope = inConversation ? SCOPE_CONVERSATION : SCOPE_SESSION;
        descriptor.autoCreate = autoCreate;
        descriptor.restrict = restrict;
        descriptor.types = getComponentTypes( description );

        if( descriptor.types.indexOf( "org.granite.tide::IEntity" ) < 0 )
        {
            for each ( var typeName:String in descriptor.types )
            {
                if( typeName == "org.granite.tide::IComponent" || typeName == "org.granite.tide::Component" )
                    continue;

                registerImplementation( typeName, name );
            }
        }

        var currentDomain:ApplicationDomain = ApplicationDomain.currentDomain;    // Used only to debug

        var componentDomain:ApplicationDomain = appDomain;
        if( componentDomain == null )
        {
            var typeClassName:String = getQualifiedClassName( type );
            // Lookup application domain of component in registered domains
            if( _currentDomain.hasDefinition( typeClassName ) && _currentDomain.getDefinition( typeClassName ) === type )
            {
                componentDomain = _currentDomain;
            }
        }
        if( componentDomain == null )
        {
            // Lookup in known application domains
            for( var ad:Object in _applicationDomains )
            {
                if( ApplicationDomain( ad ).hasDefinition( typeClassName ) && ApplicationDomain( ad ).getDefinition( typeClassName ) === type )
                {
                    componentDomain = ApplicationDomain( ad );
                    break;
                }
            }
        }
        if( componentDomain == null )
            throw new Error( "Could not guess ApplicationDomain for component " + name + " of class " + type );

        _applicationDomains[componentDomain] = true;
        descriptor.applicationDomain = componentDomain;

        // Initialize component

        if( description..factory..implementsInterface.(@type == "org.granite.tide::IComponent").length() > 0 )
            descriptor.remoteSync = true;

        var methods:XMLList = description..factory..method;
        for each ( var m:XML in methods )
        {
            var observers:XMLList = m..metadata.(@name == 'Observer');
            for each ( var observer:XML in observers )
            {
                var remote:Boolean = (observer..arg.(@key == 'remote').@value.toXMLString() == "true");
                var create:Boolean = !(observer..arg.(@key == 'create').@value.toXMLString() == "false");
                var localOnly:Boolean = (observer..arg.(@key == 'localOnly').@value.toXMLString() == "true");
                var eventTypes:String = observer..arg.(@key == '').@value.toXMLString();
                if( eventTypes == null || StringUtil.trim( eventTypes ).length == 0 )
                    eventTypes = "$TideEvent$" + m..parameter.(@index == '1').@type.toXMLString();
                var eventTypeArray:Array = eventTypes.split( "," );
                for each ( var eventType:String in eventTypeArray )
                    internalAddEventObserver( StringUtil.trim( eventType ), name, m, remote, create, localOnly );
            }

            var destroyers:XMLList = m..metadata.(@name == 'Destroy');
            for each ( var destroy:XML in destroyers )
                getDescriptor( m.@name.toXMLString() ).destroyMethodName = m.@name.toXMLString();
        }

        var modulePrefix:String = getModulePrefix( name );

        if( properties != null )
        {
            scanPropertyInjections( properties, description..factory..accessor, modulePrefix, inConversation, restrict, componentDomain );
            scanPropertyInjections( properties, description..factory..variable, modulePrefix, inConversation, restrict, componentDomain );
        }

        setupDescriptor( name, descriptor );

        scanInjections( name, description..factory..accessor, modulePrefix, inConversation, restrict, componentDomain );
        scanInjections( name, description..factory..variable, modulePrefix, inConversation, restrict, componentDomain );

        scanOutjections( description..factory..accessor, modulePrefix, inConversation, restrict, componentDomain );
        scanOutjections( description..factory..variable, modulePrefix, inConversation, restrict, componentDomain );

        dispatchEvent( new TidePluginEvent( PLUGIN_ADD_COMPONENT, { descriptor: descriptor, description: description } ) );
    }

    /**
     *     Register many Tide components at once
     *  The component definitions are scanned from annotations
     *
     *     @param types array of component classes
     *  @param appDomain the Flex application domain for classes loaded dynamically from Flex modules
     */
    public function addComponents( types:Array, appDomain:ApplicationDomain = null ):void
    {
        for each ( var type:Class in types )
        {
            // Cannot use DescribeTypeCache here, only describeType on objects can work
            var desc:XML = describeType( type );
            var info:ComponentInfo = new ComponentInfo( desc );
            if( info.name != null )
            {
                var name:String = info.module.length > 0
                        ? info.module + "." + info.name
                        : info.name;

                internalAddComponent( name, type, null,
                        info.scope == "conversation",
                        info.create != "false",
                        info.restrict == "true" ? RESTRICT_YES : (info.restrict == "false" ? RESTRICT_NO : RESTRICT_UNKNOWN)
                        );
            }
        }
    }


    /**
     *     @private
     *     Internal implementation of component definition annotation scanning
     *
     *  @param modulePrefix current module prefix
     *     @param name component name
     *  @param a XML fragment of describeType
     *  @param global should be in global module
     *  @param inConversation the component is conversation scoped
     *  @param remote should be defined as remote
     *  @param autoCreate the component is auto instantiated
     *  @param restrict the component is secured
     *  @param componentDomain the Flex application domain for the component class
     */
    private function addComponentFromXML( modulePrefix:String, name:String, a:XML, global:Boolean, inConversation:Boolean, remote:String, autoCreate:String, restrict:int, componentDomain:ApplicationDomain ):void
    {
        var pt:String = a.@type.toXMLString();
        var type:Class = componentDomain.getDefinition( pt ) as Class;
        var d2:XML = describeType( type );
        if( d2..factory..extendsClass.(@type == "org.granite.tide::BaseContext").length() > 0 || isInterface( d2 ) )
            return;

        x = d2..factory..implementsInterface.(@type == "org.granite.tide::IComponent");
        if( x.length() > 0 )
        {
            addComponent( name, type, inConversation, autoCreate != "false", restrict, componentDomain, false );
            if( global )
                setComponentGlobal( name, true );
            return;
        }

        if( !global )
            name = modulePrefix + name;

        var x:XMLList = d2..factory..implementsInterface.(@type == "org.granite.tide::IEntity");
        if( d2..factory.@type.toXMLString() == "org.granite.tide::IEntity" || x.length() > 0 || d2.@alias.length() > 0 || d2..factory..constructor.length() > 0 )
        {
            addComponent( name, type, inConversation, autoCreate == "true", restrict, componentDomain, false );
            if( global )
                setComponentGlobal( name, true );
            if( remote )
                setComponentRemoteSync( name, remote == 'true' );
            return;
        }
        x = d2..factory..implementsInterface.(@type == "mx.collections::IList");
        if( d2..factory.@type.toXMLString() == "mx.collections::IList" || x.length() > 0 )
        {
            addComponent( name, type, inConversation, autoCreate == "true", restrict, componentDomain, false );
            if( global )
                setComponentGlobal( name, true );
            if( remote )
                setComponentRemoteSync( name, remote == 'true' );
            return;
        }
        x = d2..factory..implementsInterface.(@type == "mx.core::IUIComponent");
        if( d2..factory.@type.toXMLString() == "mx.core::IUIComponent" || x.length() > 0 )
        {
            addComponent( name, type, inConversation, autoCreate == "true", restrict, componentDomain, false );
            if( global )
                setComponentGlobal( name, true );
            if( remote )
                setComponentRemoteSync( name, remote == 'true' );
            return;
        }
    }


    /**
     *     @private
     *     Internal implementation of observer registration
     *
     *     @param eventType event type
     *  @param name name of observer component
     *  @param m XML fragment of describeType for observer method
     *  @param remote observer for remote events
     *  @param create observer should be instantiated if not existent
     *  @param localOnly observer does not observe events localOnly from inner contexts/subcontexts
     */
    private function internalAddEventObserver( eventType:String, name:String, m:XML, remote:Boolean, create:Boolean, localOnly:Boolean ):void
    {
        var methodName:Object = m.@name.toXMLString();
        var uri:String = m.@uri.toXMLString();
        if( uri )
            methodName = new QName( uri, methodName );
        var names:ArrayCollection = _observersByType[eventType] as ArrayCollection;
        if( names == null )
        {
            names = new ArrayCollection();
            _observersByType[eventType] = names;
        }
        var found:Boolean = false;
        for each ( var o:Object in names )
        {
            if( o.name == name && o.methodName == methodName )
            {
                found = true;
                break;
            }
        }
        if( !found )
        {
            var p:XMLList = m..parameter;
            var pt:String = p.(@index == '1').@type.toXMLString();
            names.addItem( {
                name: name,
                methodName: methodName,
                event: (p.length() == 1 && pt == "org.granite.tide.events::TideContextEvent"),
                argumentsCount: p.length(),
                create: create,
                localOnly: localOnly
            } );
        }

        if( remote )
            addRemoteEventListener( eventType, null );
    }


    /**
     *     @private
     *     Internal implementation of injections annotation scanning
     *
     *  @param componentName scanned component name
     *     @param accs accessors/variables fragment of describeType
     *  @param modulePrefix current module prefix
     *  @param inConversation the component is conversation scoped
     *  @param autoCreate the component is auto instantiated
     *  @param restrict the component is secured
     *  @param componentDomain the Flex application domain for the component class
     */
    private function scanInjections( componentName:String, accs:XMLList, modulePrefix:String, inConversation:Boolean, restrict:int, componentDomain:ApplicationDomain ):void
    {
        for each ( var a:XML in accs )
        {
            var name:String;
            var autoCreate:String;
            var global:String;

            var injectors:XMLList = a..metadata.(@name == 'In');
            for each ( var injector:XML in injectors )
            {
                name = a.@name.toXMLString();
                var sourcePropName:String = injector..arg.(@key == '').@value.toXMLString();
                if( sourcePropName.match( /#{.*}/ ) )
                    continue;

                if( sourcePropName != null && sourcePropName.length > 0 )
                    name = sourcePropName;
                autoCreate = injector..arg.(@key == 'create').@value.toXMLString();
                global = injector..arg.(@key == 'global').@value.toXMLString();

                if( (!modulePrefix && !isComponent( name ))
                        || (modulePrefix && !isComponent( modulePrefix + name )) )
                {
                    addComponentFromXML( modulePrefix, name, a,
                            global == 'true', inConversation, null,
                            autoCreate, Tide.RESTRICT_UNKNOWN, componentDomain
                            );
                }
            }

            var typedInjectors:XMLList = a..metadata.(@name == 'Inject');
            for each ( var typedInjector:XML in typedInjectors )
            {
                name = internalNameForTypedComponent( a.@type.toXMLString() );
                autoCreate = typedInjector..arg.(@key == 'create').@value.toXMLString();
                global = typedInjector..arg.(@key == 'global').@value.toXMLString();

                if( (!modulePrefix && !isComponent( name ))
                        || (modulePrefix && !isComponent( modulePrefix + name )) )
                {
                    addComponentFromXML( modulePrefix, name, a,
                            global == 'true', inConversation, null,
                            autoCreate, Tide.RESTRICT_UNKNOWN, componentDomain
                            );
                }

                registerInjectionPoint( a.@type.toXMLString(), componentName, a.@name.toXMLString() );
            }
        }
    }

    public static const TYPED_IMPL_PREFIX:String = "__typedImpl__";

    public function internalNameForTypedComponent( name:String ):String
    {
        return TYPED_IMPL_PREFIX + name;
    }


    /**
     *     @private
     *     Internal implementation of outjections annotation scanning
     *
     *     @param accs accessors/variables fragment of describeType
     *  @param modulePrefix current module prefix
     *  @param inConversation the component is conversation scoped
     *  @param autoCreate the component is auto instantiated
     *  @param restrict the component is secured
     *  @param componentDomain the Flex application domain for the component class
     */
    private function scanOutjections( accs:XMLList, modulePrefix:String, inConversation:Boolean, restrict:int, componentDomain:ApplicationDomain ):void
    {
        for each ( var a:XML in accs )
        {
            var outjectors:XMLList = a..metadata.(@name == 'Out');
            for each ( var outjector:XML in outjectors )
            {
                var name:String = a.@name.toXMLString();
                var destPropName:String = outjector..arg.(@key == '').@value.toXMLString();
                if( destPropName != null && destPropName.length > 0 )
                    name = destPropName;
                var global:String = outjector..arg.(@key == 'global').@value.toXMLString();
                var remote:String = outjector..arg.(@key == 'remote').@value.toXMLString();

                if( (!modulePrefix && !isComponent( name ))
                        || (modulePrefix && !isComponent( modulePrefix + name )) )
                {
                    addComponentFromXML( modulePrefix, name, a,
                            global == 'true', inConversation, remote,
                            null, Tide.RESTRICT_UNKNOWN, componentDomain );
                }
                else if( remote )
                    setComponentRemoteSync( name, remote == "true" );
            }
        }
    }

    /**
     *     @private
     *     Internal implementation of static injections
     *
     *  @param properties factory object
     *     @param accs accessors/variables fragment of describeType
     *  @param modulePrefix current module prefix
     *  @param inConversation the component is conversation scoped
     *  @param autoCreate the component is auto instantiated
     *  @param restrict the component is secured
     *  @param componentDomain the Flex application domain for the component class
     */
    private function scanPropertyInjections( properties:Object, accs:XMLList, modulePrefix:String, inConversation:Boolean, restrict:int, componentDomain:ApplicationDomain ):void
    {
        for( var p:String in properties )
        {
            if( !(properties[p] is String) || !properties[p].match( /#{[^\.]*}/ ) )
                continue;

            var props:XMLList = accs.(@name.toXMLString() == p);
            if( props.length() == 0 )
                continue;

            for each ( var a:XML in props )
            {
                if( (!modulePrefix && !isComponent( p ))
                        || (modulePrefix && !isComponent( modulePrefix + p )) )
                {
                    addComponentFromXML( modulePrefix, p, a,
                            false, inConversation, null,
                            null, Tide.RESTRICT_UNKNOWN, componentDomain
                            );
                }
            }
        }
    }


    /**
     *     Unregister a Tide component and destroys all instances
     *
     *     @param name component name
     */
    public function removeComponent( name:String ):void
    {
        if( isComponent( name ) && getDescriptor( name ).factory != null && getDescriptor( name ).factory.type === Subcontext )
            return;

        for each ( var ctx:BaseContext in _ctx )
            ctx.meta_destroy( name, true );

        removeComponentDescriptor( name );

        for( var eventType:String in _observersByType )
        {
            var names:ArrayCollection = _observersByType[eventType] as ArrayCollection;
            if( names != null )
            {
                for( var i:int = 0; i < names.length; i++ )
                {
                    if( names[i].name == name )
                    {
                        names.removeItemAt( i );
                        i--;
                    }
                }
                if( names.length == 0 )
                    delete _observersByType[eventType];
            }
        }

        unregisterInjectionPoints( name );

        unregisterImplementation( name );
    }

    public function removeComponentDescriptor( name:String ):void
    {
        delete _componentDescriptors[name];
    }

    public function isProxy( instance:Object ):Boolean
    {
        if( _componentClass == null )
            return false;
        return instance is _componentClass;
    }


    /**
     *     Checks a name is a registered Tide component
     *
     *     @param name component name
     *
     *  @return true if there is a component with this name
     */
    public function isComponent( name:String ):Boolean
    {
        return getDescriptor( name, false ) != null;
    }

    /**
     *     Checks a name is a registered Tide subcontext
     *
     *     @param name component name
     *
     *  @return true if there is a subcontext with this name
     */
    public function isSubcontext( name:String ):Boolean
    {
        var desc:ComponentDescriptor = getDescriptor( name, false );
        return desc != null && desc.factory != null ? desc.factory.type == Subcontext : false;
    }


    /**
     *     @private
     *     Extracts the module prefix from a component name
     *
     *  @param componentName qualified component name
     *
     *  @return module prefix
     */
    public static function getModulePrefix( componentName:String ):String
    {
        var idx:int = componentName.lastIndexOf( "." );
        return idx > 0 ? componentName.substring( 0, idx + 1 ) : "";
    }

    /**
     *     @private
     *     Extracts the module prefix from a component name
     *
     *  @param componentName qualified component name
     *
     *  @return namespace
     */
    public static function getNamespace( componentName:String ):String
    {
        var idx:int = componentName.lastIndexOf( "." );
        return idx > 0 ? componentName.substring( 0, idx ) : "";
    }


    /**
     *     @private
     *     Internal implementation of component instantiation
     *
     *  @param name component name
     *  @param context context
     *  @param noProxy don't create remote proxy by default
     *
     *  @return component instance
     */
    public function newComponentInstance( name:String, context:BaseContext, noProxy:Boolean = false ):Object
    {
        var component:Object = null;

        var componentName:String = name;
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        if( (descriptor == null || descriptor.factory == null) && name.lastIndexOf( "." ) > 0 )
        {
            componentName = name.substring( name.lastIndexOf( "." ) + 1 );
            if( !isComponentGlobal( componentName ) )
                descriptor = getDescriptor( componentName, false );
        }

        if( descriptor == null || descriptor.factory == null )
        {
            if( _componentClass == null || noProxy )
                return null;

            // No descriptor or factory : create remote proxy
            component = new _componentClass( componentName, context );

            if( descriptor == null )
            {
                descriptor = getDescriptor( componentName, true );
                descriptor.proxy = true;
                descriptor.global = true;
            }

            descriptor.restrict = RESTRICT_YES;
            descriptor.remoteSync = true;
        }
        else
        {
            if( context.meta_isGlobal() )
            {
                if( !isComponentInSession( componentName ) )
                    return null;
            }
            else
            {
                if( !isComponentInConversation( componentName ) )
                    return null;
            }

            component = descriptor.factory.newInstance( name, context );
        }

        setComponentScope( componentName, !context.meta_isGlobal() );

        return component;
    }


    /**
     *     @private
     *     Returns a component descriptor
     *
     *  @param name component name
     *  @param create descriptor should be created if it does not exist
     *
     *  @return component descriptor
     */
    public function getDescriptor( name:String, create:Boolean = true ):ComponentDescriptor
    {
        var descriptor:ComponentDescriptor = _componentDescriptors[name];
        if( descriptor == null && create )
        {
            descriptor = new ComponentDescriptor();
            descriptor.name = name;
            _componentDescriptors[name] = descriptor;
        }
        return descriptor;
    }

    /**
     *     Return if the specific component is defined as global (not in a namespace)
     *
     *  @param name component name
     */
    public function isComponentGlobal( name:String ):Boolean
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        return descriptor ? descriptor.global : false;
    }

    /**
     *     Define if the component is defined as global (cannot be in a namespace/subcontext)
     *
     *  @param name component name
     *  @param global true if non component cannot be in a namespace/subcontext
     */
    public function setComponentGlobal( name:String, global:Boolean ):void
    {
        getDescriptor( name ).global = global;
    }

    /**
     *     Define the scope of a component
     *
     *  @param name component name
     *  @param inConversation true if conversation scope
     */
    public function setComponentScope( name:String, inConversation:Boolean ):void
    {
        getDescriptor( name ).scope = inConversation ? SCOPE_CONVERSATION : SCOPE_SESSION;
    }

    /**
     *     Returns the scope of a component
     *
     *  @param name component name
     *  @return true is conversation scoped
     */
    public function isComponentInConversation( name:String ):Boolean
    {
        return getDescriptor( name ).scope == SCOPE_CONVERSATION;
    }

    /**
     *     Returns the scope of a component
     *
     *  @param name component name
     *  @return true if session scoped
     */
    public function isComponentInSession( name:String ):Boolean
    {
        return getDescriptor( name ).scope == SCOPE_SESSION;
    }

    /**
     *     Define the creation policy of the component
     *
     *  @param name component name
     *  @param autoCreate true if component should be automatically instantiated
     */
    public function setComponentAutoCreate( name:String, autoCreate:Boolean ):void
    {
        getDescriptor( name ).autoCreate = autoCreate;
    }

    /**
     *     Returns the creation policy of a component
     *
     *  @param name component name
     *  @return true if automatic instantiation
     */
    public function isComponentAutoCreate( name:String ):Boolean
    {
        return getDescriptor( name ).autoCreate;
    }

    /**
     *     Define the remote synchronization of the component
     *  If false, the component state will never be sent to the server
     *
     *  @param name component name
     *  @param remoteSync true if the component should be remotely synchronized
     */
    public function setComponentRemoteSync( name:String, remoteSync:Boolean ):void
    {
        getDescriptor( name ).remoteSync = remoteSync;
    }

    /**
     *     Return the remote synchronization type of the component
     *
     *  @param name component name
     *  @return true true if the component should be remotely synchronized
     */
    public function isComponentRemoteSync( name:String ):Boolean
    {
        return getDescriptor( name ).remoteSync;
    }

    /**
     *     Define the security restriction of the component
     *  If RESTRICT_YES, the component state will be cleared when user logs out
     *  Default value is RESTRICT_UNKNOWN meaning if will be inferred from server definition when possible
     *
     *  @param name component name
     *  @param restrict RESTRICT_YES if the component is restricted, RESTRICT_NO if not
     */
    public function setComponentRestrict( name:String, restrict:int ):void
    {
        getDescriptor( name ).restrict = restrict;
    }

    /**
     *     Return the proxy status of the component
     *
     *  @param name component name
     *  @return true if the component is a default proxy
     */
    public function isComponentDefaultProxy( name:String ):Boolean
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        return descriptor ? descriptor.proxy : false;
    }

    /**
     *     Return the security restriction of the component
     *
     *  @param name component name
     *  @return true if the component is restricted
     */
    public function isComponentRestrict( name:String ):Boolean
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        return descriptor ? descriptor.restrict == RESTRICT_YES : false;
    }

    /**
     *     Return the security restriction of the component
     *
     *  @param name component name
     *  @return security restriction
     */
    public function getComponentRestrict( name:String ):int
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        return descriptor ? descriptor.restrict : RESTRICT_UNKNOWN;
    }

    /**
     *     Define the Flex application domain of the component
     *
     *  @param name component name
     *  @param appDomain application domain
     */
    public function setComponentDomain( name:String, appDomain:ApplicationDomain ):void
    {
        getDescriptor( name ).applicationDomain = appDomain;
    }

    /**
     *     Return the application domain of the component
     *
     *  @param name component name
     *  @return application domain
     */
    public function getComponentDomain( name:String ):ApplicationDomain
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, false );
        return descriptor ? descriptor.applicationDomain : ApplicationDomain.currentDomain;
    }


    /**
     *     Define the custom XML descriptor for the component
     *
     *  @param name component name
     *  @param xmlDescriptor XML descriptor
     */
    public function setComponentCustomDescriptor( name:String, xmlDescriptor:XML ):void
    {
        var descriptor:ComponentDescriptor = getDescriptor( name, true );
        descriptor.xmlDescriptor = xmlDescriptor;
        setupDescriptor( name, descriptor );
    }


    /**
     *    @private
     *     Setup custom descriptor
     *
     *  @param componentName component name
     *  @param descriptor component descriptor
     */
    private function setupDescriptor( componentName:String, descriptor:ComponentDescriptor ):void
    {
        if( descriptor.xmlDescriptor )
        {
            var componentDomain:ApplicationDomain = descriptor.applicationDomain;

            var description:XML = descriptor.factory.describe();

            var observers:XMLList = descriptor.xmlDescriptor..Observer;
            for each ( var observer:XML in observers )
            {
                var eventTypes:String = observer.@eventType.toXMLString();
                var eventTypeArray:Array = eventTypes.split( "," );
                var methodName:String = observer.@method.toXMLString();
                var me:XMLList = description..factory..method.(@name == methodName);
                var remote:Boolean = (observer.@remote.toXMLString() == "true");
                var create:Boolean = !(observer.@create.toXMLString() == "false");
                var localOnly:Boolean = (observer.@localOnly.toXMLString() == "true");
                for each ( var eventType:String in eventTypeArray )
                {
                    for each ( var m:XML in me )
                    {
                        if( eventType == null || StringUtil.trim( eventType ).length == 0 )
                            eventType = "$TideEvent$" + m..parameter.(@index == '1').@type.toXMLString();
                        internalAddEventObserver( StringUtil.trim( eventType ), componentName, m, remote, create, localOnly );
                    }
                }
            }

            var propName:String;
            var name:String;
            var ve:XMLList;
            var v:XML;
            var modulePrefix:String = getModulePrefix( componentName );

            var injectors:XMLList = descriptor.xmlDescriptor..In;
            for each ( var injector:XML in injectors )
            {
                propName = injector.@property.toXMLString();
                name = propName;
                var sourcePropName:String = injector.@source.toXMLString();
                if( sourcePropName.match( /#{.*}/ ) )
                    continue;

                if( sourcePropName != null && sourcePropName.length > 0 )
                    name = sourcePropName;
                var autoCreate:String = injector.@create.toXMLString();

                if( !isComponent( name ) )
                {
                    ve = description..factory..accessor.(@name.toXMLString() == propName);
                    for each ( v in ve )
                    {
                        addComponentFromXML( modulePrefix, name, v,
                                injector.@global == 'true', descriptor.scope == Tide.SCOPE_CONVERSATION, null,
                                autoCreate, Tide.RESTRICT_UNKNOWN, componentDomain
                                );
                    }
                }
            }

            var outjectors:XMLList = descriptor.xmlDescriptor..Out;
            for each ( var outjector:XML in outjectors )
            {
                propName = outjector.@property.toXMLString();
                name = propName;
                var destPropName:String = outjector.@target.toXMLString();

                if( destPropName != null && destPropName.length > 0 )
                    name = destPropName;

                if( !isComponent( name ) )
                {
                    name = modulePrefix + name;
                    ve = description..factory..accessor.(@name.toXMLString() == propName);
                    for each ( v in ve )
                    {
                        addComponentFromXML( modulePrefix, name, v,
                                outjector.@global == 'true', descriptor.scope == Tide.SCOPE_CONVERSATION, outjector.@remote.toXMLString(),
                                null, Tide.RESTRICT_UNKNOWN, componentDomain
                                );
                    }
                }
            }
        }
    }


    /**
     *    @private
     *     Invoke the observers registered for an event
     *
     *     @param context source context of the event
     *  @param modulePrefix source module prefix of the event
     *  @param type event type
     *  @param params params array
     */
    public function invokeObservers( context:BaseContext, modulePrefix:String, type:String, params:Array ):void
    {
        var localEvent:TideContextEvent = null;
        if( hasEventListener( type ) )
        {
            if( localEvent == null )
                localEvent = new TideContextEvent( type, context, currentParams );
            super.dispatchEvent( localEvent );
        }
        if( context.hasEventListener( type ) )
        {
            if( localEvent == null )
                localEvent = new TideContextEvent( type, context, currentParams );
            context.dispatchEvent( localEvent );
        }

        var observerNames:ArrayCollection = _observersByType[type];
        if( observerNames == null )
        {
            if( type.indexOf( "org.granite.tide" ) != 0 )
                log.debug( "no observer found for type: " + type );
            return;
        }

        var names:Array = observerNames.toArray();

        var saveModulePrefix0:String = _currentModulePrefix;

        var currentParams:Array = null;
        for each ( var o:Object in names )
        {
            if( observerNames.getItemIndex( o ) < 0 )
                continue;

            var localOnly:Boolean = o.localOnly;
            var currentEvent:Event;
            var observerModulePrefix:String = getModulePrefix( o.name );
            if( modulePrefix != "" && (observerModulePrefix == "" ||
                                       ((!localOnly && modulePrefix.indexOf( observerModulePrefix ) != 0) || (localOnly && modulePrefix != observerModulePrefix))) )
            {
                // Ignore events from other modules
                continue;
            }

            var saveModulePrefix:String = _currentModulePrefix;
            _currentModulePrefix = observerModulePrefix;

            var component:Object = context.meta_getInstance( o.name, o.create, !o.create );
            if( component != null )
            {
                var local:Boolean = true;
                var targetContext:Object = context.meta_getContext( o.name, component );
                if( !context.meta_isGlobal() && targetContext !== context )
                {
                    if( localOnly )
                    {
                        // Ignore events from conversation conversation context if we don't observe bubbled events
                        continue;
                    }

                    // If this is an observer for a event coming from a conversation context,
                    // we have to convert the event parameters that are entities to the global context
                    var targetParams:Array = new Array();
                    for( var i:int = 0; i < params.length; i++ )
                    {
                        if( params[i] is IEntity )
                            targetParams.push( targetContext.meta_getCachedObject( params[i] ) );
                        else if( params[i] is Event )
                        {
                            var clonedEvent:Object = null;
                            if( params[i] is IExternalizable )
                                clonedEvent = ObjectUtil.copy( params[i] );
                            else
                                clonedEvent = params[i].clone();

                            var cinfo:Object = ObjectUtil.getClassInfo( clonedEvent, null, { includeReadOnly: false, includeTransient: false } );
                            for each ( var p:String in cinfo.properties )
                            {
                                var val:Object = clonedEvent[p];
                                if( val is IEntity )
                                    clonedEvent[p] = targetContext.meta_getCachedObject( val );
                            }

                            targetParams.push( clonedEvent );
                        }
                        else if( params[i] is DisplayObject )
                            targetParams.push( params[i] );
                        else
                            targetParams.push( ObjectUtil.copy( params[i] ) );
                    }
                    local = false;
                    currentParams = targetParams;
                }
                else
                    currentParams = params;

                if( o.event )
                {
                    if( local )
                    {
                        if( localEvent == null )
                            localEvent = new TideContextEvent( type, context, currentParams );
                        currentEvent = localEvent;
                    }
                    else
                        currentEvent = new TideContextEvent( type, context, currentParams );

                    component[o.methodName].call( component, currentEvent );
                }
                else // slice required by Franck Wolff
                    component[o.methodName].apply( component, currentParams.slice( 0, o.argumentsCount ) );
            }

            _currentModulePrefix = saveModulePrefix;
        }

        _currentModulePrefix = saveModulePrefix0;
    }


    /**
     *    @private
     *     Return the entity descriptor
     *
     *  @param entity an entity
     *  @return the entity descriptor for the entity class
     */
    public function getEntityDescriptor( entity:IEntity ):EntityDescriptor
    {
        var className:String = getQualifiedClassName( entity );
        var desc:EntityDescriptor = _entityDescriptors[className] as EntityDescriptor;
        if( desc == null )
        {
            desc = new EntityDescriptor( entity );
            _entityDescriptors[className] = desc;
        }
        return desc;
    }


    /**
     *     Add an exception handler class
     *  The class should implement IExceptionHandler
     *
     *     @param handlerClass handler class
     */
    public function addExceptionHandler( handlerClass:Class ):void
    {
        try
        {
            var handler:IExceptionHandler = new handlerClass() as IExceptionHandler;
            _exceptionHandlers.unshift( handler );
        }
        catch ( e:Error )
        {
            log.error( "could not add exception handler {0}", e.message );
        }
    }

    /**
     *    Add a context listener for a particular event type
     *
     *  @param type event type
     *  @param handler event handler
     *  @param remote true if the listener observes remote events
     */
    public function addContextEventListener( type:String, handler:Function = null, remote:Boolean = false ):void
    {
        if( handler != null )
            addEventListener( type, handler, false, 0, true );

        if( remote )
            addRemoteEventListener( type, handler );
    }

    /**
     *     Register an event observer
     *
     *     @param eventType event type
     *  @param name name of observer component
     *  @param methodName name of observer method
     *  @param remote observer for remote events
     *  @param create target observer should be instantiated if not existent in context
     *  @param localOnly target observer listens only from events from its own context, not from events bubbled from inner contexts
     */
    public function addEventObserver( eventType:String, componentName:String, methodName:String, remote:Boolean = false, create:Boolean = true, localOnly:Boolean = true ):void
    {
        var descriptor:ComponentDescriptor = getDescriptor( componentName, false );
        if( descriptor == null )
            throw new Error( "Could not add observer: target component not found with name: " + componentName );

        var description:XML = descriptor.factory.describe();
        var me:XMLList = description..factory..method.(@name == methodName);
        if( me.length() != 1 )
            throw new Error( "Could not add observer: target method not found: " + componentName + "." + methodName );

        for each ( var m:XML in me )
            internalAddEventObserver( eventType, componentName, m, remote, create, localOnly );
    }


    /**
     *    @private
     *  Register a remote observer
     *
     *  @param type event type
     *  @param handler event handler
     */
    public function addRemoteEventListener( type:String, handler:Function = null ):void
    {
        if( type.indexOf( "$TideEvent$" ) == 0 )
            type = type.substring( "$TideEvent$".length ).replace( "::", "." );
        var isNew:Boolean = _registeredListeners.getItemIndex( type ) < 0;
        _registeredListeners.addItem( type );
        if( isNew )
            _newListeners.addItem( type );
    }

    /**
     *    Unregister a context event listener
     *
     *  @param type event type
     *  @param handler handler function
     */
    public function removeContextEventListener( type:String, handler:Function ):void
    {
        removeEventListener( type, handler );
    }


    /**
     *    @private
     *     Internal implementation of event dispatching that invoke registered observers
     *
     *     @param event event
     *
     *  @return return of standard dispatchEvent
     */
    public override function dispatchEvent( event:Event ):Boolean
    {
        if( event is TideContextEvent )
        {
            invokeObservers( TideContextEvent( event ).context, "", TideContextEvent( event ).type, TideContextEvent( event ).params );
            return true;
        }
        else
            return super.dispatchEvent( event );
    }


    /**
     *    @private
     *     List of listeners to send to the server for registration
     */
    public function get newListeners():ArrayCollection
    {
        return _newListeners;
    }


    /**
     *    @private
     *     Abtract method: check of user login status
     *
     *  @return true if logged in
     */
    protected function isLoggedIn():Boolean
    {
        throw new Error( "Must be overriden" );
    }

    /**
     *    @private
     *     Abtract method: define user login status
     *
     *  @param value true if logged in
     */
    protected function setLoggedIn( value:Boolean ):void
    {
        throw new Error( "Must be overriden" );
    }


    /**
     *     @private
     *     Implementation of login
     *
     *     @param ctx current context
     *  @param componentName component name of identity
     *  @param username user name
     *  @param password password
     *  @param responder Tide responder
     *
     *  @return token for the remote operation
     */
    public function login( ctx:BaseContext, component:IComponent, username:String, password:String, responder:ITideResponder = null ):AsyncToken
    {
        log.info( "login {0} > {1}", component.meta_name, username );

        _firstCall = false;
        for( var i:int = 0; i < _registeredListeners.length; i++ )
            _newListeners.addItem( _registeredListeners.getItemAt( i ) );

        ro.setCredentials( username, password );
        dispatchEvent( new TidePluginEvent( PLUGIN_SET_CREDENTIALS, { username: username, password: password } ) );
        return null;
    }


    /**
     *     @private
     *     Implementation of login check
     *
     *     @param ctx current context
     *  @param componentName component name of identity
     *  @param responder Tide responder
     *
     *  @return token for the remote operation
     */
    public function checkLoggedIn( ctx:BaseContext, component:IComponent, responder:ITideResponder = null ):AsyncToken
    {
        return null;
    }

    /**
     *     @private
     *     Implementation of logout
     *
     *     @param ctx current context
     *  @param componentName component name of identity
     */
    public function logout( context:BaseContext, component:IComponent ):void
    {
        _logoutInProgress = true;
        _waitForLogout = 1;

        context.raiseEvent( LOGOUT );

        tryLogout();
    }


    /**
     *     Notify the framework that it should wait for a async operation before effectively logging out.
     *  Only if a logout has been requested.
     */
    public function checkWaitForLogout():void
    {
        if( _logoutInProgress )
            _waitForLogout++;
    }

    /**
     *     Try logout. Should be called after all remote operations on a component are finished.
     *  The effective logout is done when all remote operations on all components have been notified as finished.
     */
    public function tryLogout():void
    {
        if( !_logoutInProgress )
            return;

        _waitForLogout--;
        if( _waitForLogout > 0 )
            return;

        dispatchEvent( new TidePluginEvent( PLUGIN_LOGOUT ) );

        if( ro.channelSet )
            ro.channelSet.logout();	// Workaround described in BLZ-310
        ro.logout();

        log.info( "Tide application logout" );

        destroyContexts();

        _logoutInProgress = false;
        _waitForLogout = 0;

        getContext().raiseEvent( LOGGED_OUT );
    }


    /**
     *     @private
     *     Implementation of login success handler
     *
     *     @param sourceContext source context of remote call
     *  @param sourceModulePrefix source module prefix
     *  @param data return object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function loginSuccessHandler( sourceContext:BaseContext, sourceModulePrefix:String, data:Object, componentName:String = null, op:String = null, tideResponder:ITideResponder = null ):void
    {
        // Force reinitialization of all application at login
        currentApplication().executeBindings( true );

        result( sourceContext, sourceModulePrefix, data, componentName, op, tideResponder );

        initAfterLogin( sourceContext );

        _initializing = false;
    }

    /**
     *     @private
     *     Implementation of is logged in success handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function isLoggedInSuccessHandler( sourceContext:BaseContext, sourceModulePrefix:String, data:Object, componentName:String = null, op:String = null, tideResponder:ITideResponder = null ):void
    {
        result( sourceContext, sourceModulePrefix, data, componentName, op, tideResponder );

        if( isLoggedIn() )
            initAfterLogin( sourceContext );

        _initializing = false;
    }

    /**
     *     @private
     *     Called when user is already logged in at application startup
     *
     *     @param sourceContext source context of remote call
     */
    public function initAfterLogin( sourceContext:BaseContext ):void
    {
        dispatchEvent( new TidePluginEvent( PLUGIN_LOGIN_SUCCESS, { sessionId: _sessionId } ) );

        sourceContext.raiseEvent( LOGIN );
    }


    /**
     *     @private
     *     Implementation of login fault handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function loginFaultHandler( sourceContext:BaseContext, sourceModulePrefix:String, info:Object, componentName:String = null, op:String = null, tideResponder:ITideResponder = null ):void
    {
        fault( sourceContext, sourceModulePrefix, info, componentName, op, tideResponder );

        dispatchEvent( new TidePluginEvent( PLUGIN_LOGOUT ) );
        ro.logout();

        dispatchEvent( new TidePluginEvent( PLUGIN_LOGIN_FAULT, { sessionId: _sessionId } ) );
    }

    /**
     *     @private
     *     Implementation of is logged in success handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function isLoggedInFaultHandler( sourceContext:BaseContext, sourceModulePrefix:String, data:Object, componentName:String = null, op:String = null, tideResponder:ITideResponder = null ):void
    {
        if( data is FaultEvent && data.fault is Fault && data.fault.faultCode == "Server.Security.NotLoggedIn" )
            result( sourceContext, sourceModulePrefix, data, componentName, op, tideResponder );
        else
            fault( sourceContext, sourceModulePrefix, data, componentName, op, tideResponder );
    }


    /**
     *     @private
     *     Destroy all contexts
     *
     *  @param force force complete destruction of contexts (all event listeners...), used for testing
     */
    protected function destroyContexts( force:Boolean = false ):void
    {
        _contextsToDestroy = new Array();

        var globalCtx:BaseContext = BaseContext( _ctx[DEFAULT_CONTEXT] );
        for each ( var ctx:BaseContext in _ctx )
        {
            if( ctx.contextId != DEFAULT_CONTEXT && ctx.parentContext === globalCtx )
                destroyContext( ctx.contextId, force );
        }
        globalCtx.meta_clear( force );
    }


    /**
     *     @private
     *     Remove context from the list of contexts to destroy
     *
     *     @param contextId context id
     */
    public function removeFromContextsToDestroy( contextId:String ):void
    {
        if( _contextsToDestroy == null )
            return;

        var idx:int = _contextsToDestroy.indexOf( contextId );
        if( idx >= 0 )
            _contextsToDestroy.splice( idx, 1 );
    }


    public var messageInterceptor:IMessageInterceptor = null;

    /**
     *     @private
     *     Implementation of component invocation
     *
     *     @param ctx current context
     *  @param component component proxy
     *  @param op remote operation
     *  @param args array of operation arguments
     *  @param responder Tide responder
     *  @param withContext send additional context with the call
     *  @param resultHandler additional result handler
     *  @param faultHandler additional fault handler
     *
     *  @return token for the remote operation
     */
    public function invokeComponent( ctx:BaseContext, component:IComponent, op:String, args:Array, responder:ITideResponder,
                                     withContext:Boolean = true, resultHandler:Function = null, faultHandler:Function = null ):AsyncToken
    {
        log.debug( "invokeComponent {0} > {1}.{2}", ctx.contextId, component.meta_name, op );

        for each ( var contextId:String in _contextsToDestroy )
            destroyContext( contextId );
        _contextsToDestroy = new Array();

        var token:AsyncToken = null;
        var operation:AbstractOperation = null;
        if( ro != null )
        {
            operation = ro.getOperation( "invokeComponent" );
            var call:IInvocationCall = ctx.meta_prepareCall( operation, withContext );
            var alias:String = describeType( component ).@alias.toXMLString();
            var componentClassName:String = alias ? alias : null;
            token = operation.send( component.meta_name, componentClassName, op, args, call );
        }
        else
        {
            var roCall:RemoteObject = _rosByDestination[component.meta_name];
            if( roCall == null )
            {
                roCall = createRemoteObject( component.meta_name );
                _rosByDestination[component.meta_name] = roCall;
            }
            var ops:Object = roCall.operations;
            operation = ops[op];
            if( operation == null )
            {
                operation = createOperation( op, roCall );
                ops[op] = operation;
                operation.mx_internal::asyncRequest = roCall.mx_internal::asyncRequest;
            }

            operation.arguments = args;
            token = operation.send();
        }

        var rh:Function = resultHandler != null ? resultHandler : result;
        var fh:Function = faultHandler != null ? faultHandler : fault;
        token.addResponder( new ComponentResponder( ctx, rh, fh, component.meta_name, op, operation, responder ) );

        _firstCall = false;

        checkWaitForLogout();

        return token;
    }


    public function getRemoteObject( dest:String ):RemoteObject
    {
        if( ro != null )
            return ro;
        return _rosByDestination[dest];
    }


    /**
     *     @private
     *     Implementation of context resync
     *
     *     @param ctx current context
     *  @param responder Tide responder
     *
     *  @return token for the remote operation
     */
    public function resyncContext( ctx:BaseContext, responder:ITideResponder ):AsyncToken
    {
        log.debug( "resyncContext {0}", ctx.contextId );

        for each ( var contextId:String in _contextsToDestroy )
            destroyContext( contextId );
        _contextsToDestroy = new Array();

        var operation:AbstractOperation = ro.getOperation( "resyncContext" );
        var call:IInvocationCall = ctx.meta_prepareCall( operation, true );
        var token:AsyncToken = operation.send( call );
        token.addResponder( new ComponentResponder( ctx, result, fault, null, null, operation, responder ) );

        _firstCall = false;

        if( _logoutInProgress )
            _waitForLogout++;

        return token;
    }

    /**
     *     @private
     *     Implementation of lazy initialization
     *
     *     @param ctx current context
     *  @param obj object to initialize (should be a PersistentCollection or PersistentMap)
     *  @param path path of the object in the context
     *
     *  @return token for the remote operation
     */
    public function initializeObject( ctx:BaseContext, obj:Object, path:IExpression ):void
    {
        log.debug( "initializeObject {0} > {1}", ctx.contextId, obj );

        // For now, assumes that obj is a PersistentCollection
        if( !(obj is PersistentCollection || obj is PersistentMap) )
            throw new Error( "Auto initialization works only with PersistentCollection/PersistentMap " + BaseContext.toString( obj ) );

        var entity:Object = obj.entity;

        _objectsInitializing.push( { context: ctx, entity: path ? path.path : obj.entity, propertyName: obj.propertyName } );

        getContext().application.callLater( doInitializeObjects, [ctx] );

        //		    var propertyNames:Array = [ obj.propertyName ];
        //            var cinfo:Object = ObjectUtil.getClassInfo(entity, null, { includeTransient: false });
        //            for each (var p:String in cinfo.properties) {
        //                if (propertyNames.indexOf(p) >= 0)
        //                    continue;
        //
        //                var o:Object = entity[p];
        //                if ((o is PersistentCollection && PersistentCollection(o).isInitializing())
        //                	|| (o is PersistentMap && PersistentMap(o).isInitializing()))
        //                    propertyNames.push(p);
        //            }
    }

    private function doInitializeObjects( ctx:BaseContext ):void
    {

        var initMap:Dictionary = new Dictionary();

        for( var i:int = 0; i < _objectsInitializing.length; i++ )
        {
            if( _objectsInitializing[i].context != ctx )
                continue;

            var propertyNames:Array = initMap[_objectsInitializing[i].entity];
            if( propertyNames == null )
            {
                propertyNames = [ _objectsInitializing[i].propertyName ];
                initMap[_objectsInitializing[i].entity] = propertyNames;
            }
            else
                propertyNames.push( _objectsInitializing[i].propertyName );

            _objectsInitializing.splice( i, 1 );
            i--;
        }

        for( var entity:Object in initMap )
        {
            var operation:AbstractOperation = roInitialize.getOperation( "initializeObject" );
            var call:IInvocationCall = ctx.meta_prepareCall( operation, false );
            var token:AsyncToken = operation.send( entity, initMap[entity], call );
            token.addResponder( new InitializerResponder( ctx, initializerResult, initializerFault, entity, initMap[entity] ) );

            if( _logoutInProgress )
                _waitForLogout++;
        }
    }

    private var _objectsInitializing:Array = new Array();

    /**
     *     @private
     *     Implementation of remote validation
     *
     *     @param ctx current context
     *  @param entity object to validate
     *  @param propertyName property to validate
     *  @param value value to validate
     *
     *  @return token for the remote operation
     */
    public function validateObject( ctx:BaseContext, entity:IEntity, propertyName:String, value:Object ):AsyncToken
    {
        log.debug( "validateObject {0} > {1}", ctx.contextId, entity );

        var operation:AbstractOperation = ro.getOperation( "validateObject" );
        var call:IInvocationCall = ctx.meta_prepareCall( operation, false );
        // For now, assumes that obj is a PeristentCollection
        var token:AsyncToken = operation.send( entity, propertyName, value, call );

        token.addResponder( new ValidatorResponder( ctx, entity, propertyName ) );

        if( _logoutInProgress )
            _waitForLogout++;

        return token;
    }


    /**
     *     @private
     *     Get the contextId from the server response, should be overriden by subclasses
     *
     *     @param event the response message
     *  @param fromFault the message is a fault
     *
     *  @return contextId
     */
    protected function extractContextId( event:MessageEvent, fromFault:Boolean = false ):String
    {
        return DEFAULT_CONTEXT;
    }

    /**
     *     @private
     *     Get the conversation status from the server response, should be overriden by subclasses
     *
     *     @param event the response message
     *
     *  @return true if the conversation was created by the server
     */
    protected function wasConversationCreated( event:MessageEvent ):Boolean
    {
        return false;
    }

    /**
     *     @private
     *     Get the conversation status from the server response, should be overriden by subclasses
     *
     *     @param event the response message
     *
     *  @return true if the conversation was ended by the server
     */
    protected function wasConversationEnded( event:MessageEvent ):Boolean
    {
        return false;
    }


    public function updateContextId( previousContextId:String, context:BaseContext ):void
    {
        if( previousContextId != null )
            delete _ctx[previousContextId];
        _ctx[context.contextId] = context;
    }


    /**
     *    @private
     *  Get the context where the result/fault shall be processed
     *
     *  @param sourceContext context from where the call has been issued
     *  @param event response message
     *  @param fromFault is a fault
     */
    private function extractContext( sourceContext:BaseContext, event:MessageEvent, fromFault:Boolean = false ):BaseContext
    {
        var sessionId:String = event.message.headers['org.granite.sessionId'];
        if( sessionId != null )
            _sessionId = sessionId;

        if( messageInterceptor != null )
            messageInterceptor.after( event.message );

        var contextId:String = extractContextId( event, fromFault );
        var wasConversationCreated:Boolean = wasConversationCreated( event );
        var wasConversationEnded:Boolean = wasConversationEnded( event );

        var context:BaseContext = null;
        if( !sourceContext.meta_isGlobal() && contextId == DEFAULT_CONTEXT && wasConversationEnded )
        {
            // The conversation of the source context was ended
            // Get results in the current conversation when finished
            context = sourceContext;
            context.meta_markAsFinished();
        }
        else if( !sourceContext.meta_isGlobal() && contextId == DEFAULT_CONTEXT && !sourceContext.meta_isContextIdFromServer )
        {
            // A call to a non conversational component was issued from a conversation context
            // Get results in the current conversation
            context = sourceContext;
        }
        else if( !sourceContext.meta_isGlobal() && contextId != DEFAULT_CONTEXT
                && (sourceContext.contextId == null || (sourceContext.contextId != contextId && !wasConversationCreated)) )
        {
            // The conversationId has been updated by the server
            var previousContextId:String = sourceContext.contextId;
            context = sourceContext;
            context.meta_setContextId( contextId, true );
            if( previousContextId != null )
                delete _ctx[previousContextId];
            _ctx[contextId] = context;
        }
        else
        {
            context = getContext( contextId );
            if( contextId != DEFAULT_CONTEXT )
                context.meta_setContextId( contextId, true );
        }

        return context;
    }


    /**
     *     @private
     *     Implementation of result handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function result( sourceContext:BaseContext, sourceModulePrefix:String, data:Object, componentName:String = null, operation:String = null, tideResponder:ITideResponder = null ):void
    {
        var invocationResult:IInvocationResult = null;
        var result:Object = null;
        if( data is ResultEvent )
            result = ResultEvent( data ).result;
        else if( data is MessageEvent )
            result = MessageEvent( data ).message.body;

        if( result is IInvocationResult )
        {
            invocationResult = result as IInvocationResult;
            result = invocationResult.result;
        }

        var context:BaseContext = extractContext( sourceContext, MessageEvent( data ) );

        var saveModulePrefix:String = _currentModulePrefix;
        _currentModulePrefix = sourceModulePrefix;

        context.meta_result( componentName, operation, invocationResult, result,
                tideResponder is ITideMergeResponder ? ITideMergeResponder( tideResponder ).mergeResultWith : null );
        if( invocationResult )
            result = invocationResult.result;

        var handled:Boolean = false;
        if( tideResponder )
        {
            var event:TideResultEvent = new TideResultEvent( TideResultEvent.RESULT, context, false, true, data.token, result );
            tideResponder.result( event );
            if( event.isDefaultPrevented() )
                handled = true;
        }

        _currentModulePrefix = saveModulePrefix;

        context.meta_clearCache();

        // Should be after event result handling and responder: previous could trigger other remote calls
        if( context.meta_finished )
        {
            _contextsToDestroy.push( context.contextId );
            context.meta_scheduleDestroy();
        }

        _initializing = false;

        if( !handled && !_logoutInProgress )
            context.raiseEvent( CONTEXT_RESULT, result );

        tryLogout();
    }

    /**
     *     @private
     *     Implementation of fault handler
     *
     *     @param sourceContext source context of remote call
     *  @param info fault object
     *  @param componentName component name
     *  @param op remote operation
     *  @param tideResponder Tide responder for the remote call
     */
    public function fault( sourceContext:BaseContext, sourceModulePrefix:String, info:Object, componentName:String = null, operation:String = null, tideResponder:ITideResponder = null ):void
    {
        log.error( "fault {0}", info );
        var faultEvent:FaultEvent = info as FaultEvent;

        var context:BaseContext = extractContext( sourceContext, faultEvent, true );

        var emsg:ErrorMessage = faultEvent.message is ErrorMessage ? faultEvent.message as ErrorMessage : null;
        var m:ErrorMessage = emsg;
        do {
            if( m && m.faultCode && m.faultCode.search( "Server.Security." ) == 0 )
            {
                emsg = m;
                break;
            }
            if( m && (m.rootCause is FaultEvent || m.rootCause is ChannelFaultEvent) )
                m = m.rootCause.rootCause as ErrorMessage;
            else if( m )
                m = m.rootCause as ErrorMessage;
        }
        while( m );

        var saveModulePrefix:String = _currentModulePrefix;
        _currentModulePrefix = sourceModulePrefix;

        context.meta_fault( componentName, operation, emsg );

        var handled:Boolean = false;
        if( tideResponder )
        {
            var fault:Fault = null;
            if( emsg != null && emsg !== faultEvent.message )
            {
                fault = new Fault( emsg.faultCode, emsg.faultString, emsg.faultDetail );
                fault.message = faultEvent.fault.message;
                fault.rootCause = faultEvent.fault.rootCause;
            }
            else
                fault = faultEvent.fault;
            var event:TideFaultEvent = new TideFaultEvent( TideFaultEvent.FAULT, context, false, true, info.token, fault );
            tideResponder.fault( event );
            if( event.isDefaultPrevented() )
                handled = true;
        }

        if( !handled )
        {
            if( emsg != null )
            {
                for each ( var handler:IExceptionHandler in _exceptionHandlers )
                {
                    if( handler.accepts( emsg ) )
                    {
                        handler.handle( context, emsg );
                        handled = true;
                        break;
                    }
                }
                if( !handled )
                    log.error( "Unhandled fault: " + emsg.faultCode + ": " + emsg.faultDetail );
            }
            else if( _exceptionHandlers.length > 0 && faultEvent.message is ErrorMessage )
            {
                _exceptionHandlers[0].handler( context, faultEvent.message as ErrorMessage );
            }
            else
            {
                log.error( "Unknown fault: " + faultEvent.toString() );
                Alert.show( "Unknown fault: " + faultEvent.toString() );
            }
        }

        _currentModulePrefix = saveModulePrefix;

        if( !handled && !_logoutInProgress )
            context.raiseEvent( CONTEXT_FAULT, info.message );

        tryLogout();
    }


    /**
     *     @private
     *     Implementation of initializer success handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param entity object to initialize
     *  @param propertyNames array of property names to initialize
     */
    public function initializerResult( sourceContext:BaseContext, data:Object, entity:Object, propertyNames:Array ):void
    {
        var res:Array = data.result.result as Array;

        sourceContext.meta_uninitializeAllowed = false;

        // Assumes objects is a PersistentCollection or PersistentMap
        sourceContext.meta_mergeExternal( data.result.result, entity );

        result( sourceContext, "", data );

        sourceContext.meta_uninitializeAllowed = true;
    }

    /**
     *     @private
     *     Implementation of initializer fault handler
     *
     *     @param sourceContext source context of remote call
     *  @param data return object
     *  @param entity object to initialize
     *  @param propertyNames array of property names to initialize
     */
    public function initializerFault( sourceContext:BaseContext, info:Object, entity:Object, propertyNames:Array ):void
    {
        log.error( "Fault initializing collection " + BaseContext.toString( entity ) + " " + info.toString() );

        fault( sourceContext, "", info );
    }


    /**
     *     @private
     *     Hack to force the context to be [Managed]/[Bindable] and modify the Flex reflection cache
     *
     *     @param record the cache record
     */
    private static function bindabilityInfoHandler( record:DescribeTypeCacheRecord ):*
    {
        // Hideous hack to ensure all context variables and components will be correctly bindable
        if( record.typeDescription.@base == "org.granite.tide::BaseContext"
                || record.typeDescription.@name == "org.granite.tide::Subcontext"
                || record.typeDescription.@name == "org.granite.tide::Component" )
            record.typeDescription.appendChild( <metadata name="Managed"/> );

        // Another hack to force non public Out properties to be bindable
        for each ( var out:XML in record.typeDescription.variable )
        {
            if( out.metadata && out.metadata.@name == 'Out' && out.@uri )
            {
                record.typeDescription.appendChild( <accessor name={out.@name} access="readwrite" type={out.@type}>
                    <metadata name="Bindable">
                        <arg key="event" value="propertyChange"/>
                    </metadata>
                </accessor> );
            }
        }

        return new BindabilityInfo( record.typeDescription );
    }

    /**
     *     @private
     *     Reflection cache for component info
     *
     *     @param record the cache record
     */
    private static function componentInfoHandler( record:DescribeTypeCacheRecord ):*
    {
        return new ComponentInfo( record.typeDescription );
    }

    /**
     *     @private
     *     Reflection cache for component info
     *
     *     @param record the cache record
     */
    private static function entityInfoHandler( record:DescribeTypeCacheRecord ):*
    {
        return new EntityInfo( record.typeDescription );
    }
}
}
