/*
 GRANITE DATA SERVICES
 Copyright (C) 2007-2008 ADEQUATE SYSTEMS SARL

 This file is part of Granite Data Services.

 Granite Data Services is free software; you can redistribute it and/or modify
 it under the terms of the GNU Lesser General Public License as published by
 the Free Software Foundation; either version 3 of the License, or (at your
 option) any later version.

 Granite Data Services is distributed in the hope that it will be useful, but
 WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public License
 for more details.

 You should have received a copy of the GNU Lesser General Public License
 along with this library; if not, see <http://www.gnu.org/licenses/>.
 */

package org.granite.tide.collections
{

import flash.utils.Dictionary;
import flash.utils.flash_proxy;

import mx.collections.IList;
import mx.collections.IViewCursor;
import mx.collections.ListCollectionView;
import mx.collections.errors.ItemPendingError;
import mx.collections.ItemResponder;
import mx.events.CollectionEvent;
import mx.events.CollectionEventKind;
import mx.logging.ILogger;
import mx.logging.Log;
import mx.rpc.AsyncToken;
import mx.rpc.IResponder;
import mx.rpc.events.ResultEvent;
import mx.utils.object_proxy;
import mx.data.utils.Managed;

import org.granite.collections.IPersistentCollection;
import org.granite.tide.IEntity;
import org.granite.tide.IEntityManager;
import org.granite.tide.IWrapper;
import org.granite.tide.IPropertyHolder;
import org.granite.tide.BaseContext;


use namespace flash_proxy;
use namespace object_proxy;


[ExcludeClass]
/**
 *     Internal implementation of persistent collection handling automatic lazy loading.<br/>
 *  Used for wrapping persistent collections received from the server.<br/>
 *     Should not be used directly.
 *
 *     @author William DRAI
 */
public class PersistentCollection extends ListCollectionView implements IPersistentCollection, IPropertyHolder, IWrapper
{

    private static var log:ILogger = Log.getLogger( "org.granite.tide.collections.PersistentCollection" );

    private var _entity:IEntity = null;
    private var _propertyName:String = null;

    private var _localInitializing:Boolean = false;
    private var _itemPendingError:ItemPendingError = null;


    public function get entity():Object
    {
        return _entity;
    }

    public function get propertyName():String
    {
        return _propertyName;
    }

    public function PersistentCollection( entity:IEntity, propertyName:String, collection:IPersistentCollection )
    {
        super( IList( collection ) );
        _entity = entity;
        _propertyName = propertyName;
    }


    public function get object():Object
    {
        return list;
    }

    public function isInitialized():Boolean
    {
        return IPersistentCollection( list ).isInitialized();
    }

    public function isInitializing():Boolean
    {
        return _itemPendingError != null;
    }

    public function initializing():void
    {
        IPersistentCollection( list ).initializing();
        _localInitializing = true;
    }

    private function requestInitialization():void
    {
        var em:IEntityManager = Managed.getEntityManager( _entity );
        if( _localInitializing || BaseContext( em ).meta_lazyInitializationDisabled )
            return;

        if( !_itemPendingError )
        {
            log.debug( "initialization requested " + toString() );
            em.meta_initializeObject( this );

            _itemPendingError = new ItemPendingError( "PersistentCollection initializing" );
        }

        // throw _itemPendingError;
    }

    public function initialize():void
    {
        IPersistentCollection( list ).initialize();
        _localInitializing = false;

        if( _itemPendingError )
        {
            var event:ResultEvent = new ResultEvent( ResultEvent.RESULT, false, true, this );
            log.debug( "notify item pending" );

            if( _itemPendingError.responders )
            {
                for( var k:int = 0; k < _itemPendingError.responders.length; k++ )
                    _itemPendingError.responders[k].result( event );
            }

            _itemPendingError = null;
        }

        var ce:CollectionEvent = new CollectionEvent( CollectionEvent.COLLECTION_CHANGE, false, false,
                CollectionEventKind.REFRESH );
        dispatchEvent( ce );

        log.debug( "initialized" );
    }

    public function uninitialize():void
    {
        IPersistentCollection( list ).uninitialize();
        _itemPendingError = null;
        _localInitializing = false;

        var ce:CollectionEvent = new CollectionEvent( CollectionEvent.COLLECTION_CHANGE, false, false,
                CollectionEventKind.REFRESH );
        dispatchEvent( ce );
    }


    public function meta_propertyResultHandler( propName:String, event:ResultEvent ):void
    {
    }


    override public function get length():int
    {
        if( _localInitializing || isInitialized() )
            return super.length;

        requestInitialization();
        return 0;
        //	        // Must be initialized to >0 to enable ItemPendingError at first getItemAt
        //	        var em:IEntityManager = Managed.getEntityManager(_entity);
        //			return (em && !BaseContext(em).meta_lazyInitializationDisabled) ? 1 : 0;
    }

    override public function getItemAt( index:int, prefetch:int = 0 ):Object
    {
        if( _localInitializing || isInitialized() )
            return super.getItemAt( index, prefetch );

        requestInitialization();
        return null;
    }

    override public function addItem( item:Object ):void
    {
        if( _localInitializing || isInitialized() )
        {
            super.addItem( item );
            return;
        }

        throw new Error( "Cannot modify uninitialized collection: " + _entity + " property " + propertyName );
    }

    override public function addItemAt( item:Object, index:int ):void
    {
        if( _localInitializing || isInitialized() )
        {
            super.addItemAt( item, index );
            return;
        }

        throw new Error( "Cannot modify uninitialized collection: " + _entity + " property " + propertyName );
    }

    override public function setItemAt( item:Object, index:int ):Object
    {
        if( _localInitializing || isInitialized() )
            return super.setItemAt( item, index );

        throw new Error( "Cannot modify uninitialized collection: " + _entity + " property " + propertyName );
    }

    override public function removeItemAt( index:int ):Object
    {
        if( _localInitializing || isInitialized() )
            return super.removeItemAt( index );

        throw new Error( "Cannot modify uninitialized collection: " + _entity + " property " + propertyName );
    }

    override public function removeAll():void
    {
        if( _localInitializing || isInitialized() )
        {
            super.removeAll();
            return;
        }
        super.removeAll();
        initialize();
    }

    override public function refresh():Boolean
    {
        if( _localInitializing || isInitialized() )
            return super.refresh();

        requestInitialization();
        return false;
    }


    override public function toString():String
    {
        if( isInitialized() )
            return "Initialized collection for object: " + BaseContext.toString( _entity ) + " " + _propertyName + ": " + super.toString();

        return "Uninitialized collection for object: " + BaseContext.toString( _entity ) + " " + _propertyName;
    }
}
}
