package org.gridvise.logical
import java.net.InetAddress
import scala.annotation.implicitNotFound
import scala.collection.mutable.HashMap
import scala.collection.mutable.HashSet
import scala.xml.XML
import org.gridvise.mgmtcache.coh.entity.launchable.LaunchableCache
import org.gridvise.util.IdGenerator
import org.gridvise.xmlbindings.Server
import org.gridvise.xmlbindings.JvmConfig
import org.gridvise.xmlbindings.ClusterConfig
import org.gridvise.xmlbindings.JvmGroup
import org.gridvise.logical.os.OSOperations
import org.gridvise.logical.os.MachineInfo

object ClusterConfigParser {

  val hostnameVarName = "HOSTNAME"
  val uniqueNameVarName = "uniqueName"

  val hostname = MachineInfo.getMachineName()

  var clusterName = ""
  var clusterMode = ""

  var clusterconfig: ClusterConfig = _
  var server: Server = _

  val jvmConfigs = new HashMap[String, JvmConfig]
  val jvmGroups = new HashMap[String, JvmGroup]
  val sysVars = new HashMap[String, String]
  val overrideVars = new HashMap[String, String]

  def initialize(xmlIncoming: String) = {
    println("initializing on " + hostname)
    val xmlString = replaceSysVarRefs(xmlIncoming)
    val xml = XML.loadString(xmlIncoming)
    clusterconfig = scalaxb.fromXML[org.gridvise.xmlbindings.ClusterConfig](xml)
    clusterName = clusterconfig.name
    clusterMode = clusterconfig.mode
    assignServer()
    indexJvmConfigs()
    indexJvmGroups()
    setOverideVariables()
    indexSystemVariables()
    insertLaunchables()
  }


  private def getLaunchables(): HashSet[Launchable] = {
    var launchables = new HashSet[Launchable]()
    if (server != null) {
      for (n <- ClusterConfigParser.this.server.NodeGroupRef) yield {
        val g = ClusterConfigParser.this.jvmGroups.get(n.name).get
        for (j <- g.Jvm) yield {
          var c = 0
          times(j.count) ({
            println(jvmConfigs.get(j.configRef).get)
            var jvmConfig = jvmConfigs.get(j.configRef).get
            var l = ClusterConfigParser.this.createLaunchable(jvmConfig, n.name.toString(), c + 1)
            if (!j.delay.isEmpty) {
              l.delay = j.delay.get
            }
            launchables.add(l)
          })
        }
      }
    } else {
      println("no launchables configured for this host")
    }
    launchables
  }

  private def insertLaunchables() {
    getLaunchables().foreach(l => LaunchableCache.putLaunchable(l))
  }

  private def indexJvmConfigs() {
    for (c <- clusterconfig.JvmConfigs.JvmConfig) yield {
      ClusterConfigParser.this.sysVars.put(ClusterConfigParser.this.uniqueNameVarName, c.uniqueName)
      ClusterConfigParser.this.jvmConfigs.put(c.uniqueName, c)
    }
  }

  private def indexSystemVariables() {
    var p = System.getenv()
    for (c <- clusterconfig.RequiredSystemVars.SystemVar) yield {
      val v = getOverrideOption(c.name)
      if (v != null) {
        ClusterConfigParser.this.sysVars.put(c.name, v)
      } else {
        var v = p.get(c.name)
        if (v == null) {
          println("WARNING: " + c.name + " not provided as environment variable.")
          if (c.description != null) {
            println(c.name + " is expected as: " + c.description)
          }
        } else {
          ClusterConfigParser.this.sysVars.put(c.name, v.toString().replace("workspace-jvms", "workspace"))
        }
      }
    }
    ClusterConfigParser.this.sysVars.put(ClusterConfigParser.this.hostnameVarName, hostname)
  }

  private def getOverrideOption(name: String) = {
    val overrideOption = ClusterConfigParser.this.overrideVars.get(name)
    if (!overrideOption.isEmpty) {
      overrideOption.get
    } else {
      null
    }
  }

  private def setOverideVariables() {
    for (c <- clusterconfig.OverrideSystemVars.OverrideVar) yield {
      ClusterConfigParser.this.overrideVars.put(c.name, c.dir)
    }
  }
  private def indexJvmGroups() {
    for (c <- clusterconfig.JvmGroups.JvmGroup) yield {
      ClusterConfigParser.this.jvmGroups.put(c.name, c)
    }
  }

  private def assignServer() = {
    for (s <- clusterconfig.Servers.Server) yield {
      if (ClusterConfigParser.this.hostname.startsWith(s.hostname)) {
        ClusterConfigParser.this.server = s
      }
    }
  }

  private def createLaunchable(jvmConfig: JvmConfig, nodeGroupName: String, ordinal: Integer): Launchable = {
    var l = new Launchable(IdGenerator.next(), ordinal)
    setProperties(jvmConfig, l)
    l.nodeGroupName = nodeGroupName
    println("created launchable " + l)
    l
  }

  private def setProperties(jvmConfig: JvmConfig, l: Launchable): Unit = {

    if (!jvmConfig.parent.isEmpty) {
      val parentJvmConfigName = jvmConfig.parent.get
      if (parentJvmConfigName != null) {
        var parentJvmConfig = ClusterConfigParser.this.jvmConfigs.get(parentJvmConfigName)
        if (parentJvmConfig != null) {
          setProperties(parentJvmConfig.get, l)
        } else {
          println("WARNING: refernce to parent " + jvmConfig.parent.get + " not found");
        }
      }
    }

    l.configName = jvmConfig.uniqueName

    if (!jvmConfig.mainClass.isEmpty()) {
      l.mainClass = jvmConfig.mainClass
    }
    if (!jvmConfig.jvmArgs.isEmpty) {
      l.jvmArgs = jvmConfig.jvmArgs.get
    }
    for (x <- jvmConfig.JavaOptions.Option)
      l.jvmOptions.put(x.name,x.valueAttribute)

    for (x <- jvmConfig.Classpath)
      for (d <- x.Dir)
        l.classPathEntries.add(sysVars.get(d.systemVar.get).get + "/" + d.relativePath.get)
  }

  private def replaceSysVarRefs(xmlString: String): String = {
    val xml = XML.loadString(xmlString)
    clusterconfig = scalaxb.fromXML[org.gridvise.xmlbindings.ClusterConfig](xml)
    setOverideVariables()
    indexSystemVariables()

    var returnString = xmlString
    ClusterConfigParser.this.sysVars foreach { case (varName, varValue) => returnString = returnString.replaceAll("\\$\\{" + varName + "\\}", varValue) }
    returnString
  }

  private def times(n: Int)(code: => Unit) = {
    for (i <- 1 to n) code
  }

}