/*
 * Decompiled with CFR 0.152.
 */
package com.oracle.common.internal.io;

import com.oracle.common.base.Disposable;
import com.oracle.common.io.BufferManager;
import java.nio.ByteBuffer;
import java.util.ConcurrentModificationException;
import java.util.Iterator;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.logging.Level;
import java.util.logging.Logger;

public class SegmentedBufferManager
implements BufferManager,
Disposable {
    private static final int GEN_ID_SHIFT = 6;
    private static final int GEN_ID_BITS = 4;
    private static final int GEN_ID_MASK = 960;
    private static final int GEN_ID_UNPOOLED = 15;
    public static final short DEFAULT_GROWTH_FACTOR = 2;
    public static final int DEFAULT_BUF_SIZE = 1024;
    public static final int DEFAULT_STATS_FREQUENCY = 4095;
    public static final int DEFAULT_SEGMENT_COUNT = 5;
    public static final int DEFAULT_SEGMENT_SIZE = 0xC00000;
    private static final long CLEANUP_FREQUENCY_MILLIS = Long.parseLong(System.getProperty("sbm.cleanup.frequency", "60")) * 1000L;
    protected static final Logger LOGGER = Logger.getLogger(SegmentedBufferManager.class.getName());
    private final BufferAllocator m_allocator;
    private final PoolSegment[] m_aSegments;
    private final int m_cbMin;
    private final int m_cbMax;
    private final int m_nSegmentGrowthFactor;

    public SegmentedBufferManager(BufferAllocator allocator, long cbMax) {
        this(allocator, (int)(cbMax < Integer.MAX_VALUE ? 5L : cbMax / Integer.MAX_VALUE * 5L), (int)(cbMax < Integer.MAX_VALUE ? cbMax / 5L : cbMax / (cbMax / Integer.MAX_VALUE * 5L)), 1024, 2);
    }

    public SegmentedBufferManager(BufferAllocator allocator, int cSegments, int cbSegment, int cbBufferMin, int nGrowthFactor) {
        this.m_allocator = allocator;
        int cbDefaultBuf = 1024;
        if ((cbBufferMin & 0x3FF) != 0) {
            cbBufferMin = cbBufferMin / 1024 * 1024 + (cbBufferMin % 1024 == 0 ? 0 : 1024);
        }
        this.m_nSegmentGrowthFactor = nGrowthFactor;
        int cbBuff = this.m_cbMin = cbBufferMin;
        this.m_aSegments = new PoolSegment[cSegments];
        PoolSegment[] aSegment = this.m_aSegments;
        for (int i = 0; i < cSegments; ++i) {
            aSegment[i] = new PoolSegment(cbBuff, cbSegment / cbBuff, 4095);
            cbBuff <<= nGrowthFactor;
        }
        this.m_cbMax = cbBuff >> nGrowthFactor;
    }

    @Override
    public ByteBuffer acquire(int cbMin) {
        ByteBuffer buff = this.ensureMinSegment(cbMin).acquire();
        if (buff.capacity() > cbMin) {
            buff.limit(cbMin);
        }
        return buff;
    }

    @Override
    public ByteBuffer acquirePref(int cbPref) {
        ByteBuffer buff = this.ensureSegment(cbPref).acquire();
        if (buff.capacity() > cbPref) {
            buff.limit(cbPref);
        }
        return buff;
    }

    @Override
    public ByteBuffer acquireSum(int cbSum) {
        return this.ensureSegment(cbSum).acquire();
    }

    @Override
    public void release(ByteBuffer buffer) {
        this.getSegment(buffer.capacity()).release(buffer);
    }

    @Override
    public ByteBuffer truncate(ByteBuffer buff) {
        int cbSeg = this.decodeSize(buff.capacity());
        int cbSegPre = cbSeg >> this.m_nSegmentGrowthFactor;
        int cbUsed = buff.remaining();
        if (cbSeg > this.m_cbMin && cbUsed <= cbSegPre) {
            ByteBuffer buffNew;
            try {
                buffNew = this.acquire(cbUsed);
            }
            catch (OutOfMemoryError e) {
                return buff;
            }
            buffNew.put(buff).flip();
            this.release(buff);
            return buffNew;
        }
        return buff;
    }

    @Override
    public void dispose() {
        for (PoolSegment pool : this.m_aSegments) {
            pool.dispose();
        }
    }

    private int decodeSize(int cb) {
        return cb & 0xFFFFFC3F;
    }

    private PoolSegment ensureSegment(int cb) {
        if (cb >= this.m_cbMax) {
            return this.m_aSegments[this.m_aSegments.length - 1];
        }
        int iSeg = 0;
        int cbMin = this.m_cbMin;
        while (cb > cbMin) {
            cb >>= this.m_nSegmentGrowthFactor;
            ++iSeg;
        }
        return this.m_aSegments[iSeg];
    }

    private PoolSegment ensureMinSegment(int cb) {
        int iSeg;
        PoolSegment[] aSegment = this.m_aSegments;
        if (cb > this.m_cbMax) {
            throw new OutOfMemoryError("requested buffer size exceeds pool maximum");
        }
        int cSeg = aSegment.length;
        for (iSeg = 0; iSeg < cSeg && aSegment[iSeg].m_cbBuffer < cb; ++iSeg) {
        }
        return aSegment[iSeg];
    }

    private PoolSegment getSegment(int cb) throws IllegalArgumentException {
        int iSeg;
        PoolSegment[] aSegments = this.m_aSegments;
        int cbDecoded = cb = this.decodeSize(cb);
        int cSeg = aSegments.length;
        int cbMin = this.m_cbMin;
        for (iSeg = 0; cb > cbMin && iSeg < cSeg; cb >>= this.m_nSegmentGrowthFactor, ++iSeg) {
        }
        if (iSeg < cSeg && cbDecoded == aSegments[iSeg].getBufferSize()) {
            return aSegments[iSeg];
        }
        throw new IllegalArgumentException("No pool segment for size: " + cbDecoded + " in " + cSeg + " segment(s) between " + aSegments[0].getBufferSize() + " .. " + aSegments[cSeg - 1].getBufferSize());
    }

    public static interface BufferAllocator {
        public ByteBuffer allocate(int var1);

        public void release(ByteBuffer var1);
    }

    private final class PoolSegment
    implements Disposable {
        private static final int GEN_ID_LOCKED = -1;
        private final int m_cbBuffer;
        private final int m_cBufferGen;
        private final ConcurrentLinkedQueue<ByteBuffer> m_queue;
        private final AtomicInteger m_cGeneration;
        private volatile long m_ldtNextEvaluation;
        private final AtomicLong m_cAcquired;
        private final AtomicLong m_cReleased;
        private final int m_cbNonPooled;
        private final int m_cStatsFreq;
        private int m_cMaxBuffers;

        private PoolSegment(int cbBuffer, int cBuffers, int cReevalFreq) {
            this.m_cbBuffer = cbBuffer;
            this.m_cBufferGen = Math.max(cBuffers / 15, 1);
            this.m_cStatsFreq = cReevalFreq;
            this.m_queue = new ConcurrentLinkedQueue();
            this.m_cGeneration = new AtomicInteger(0);
            this.m_cReleased = new AtomicLong(0L);
            this.m_cAcquired = new AtomicLong(0L);
            this.m_cbNonPooled = this.encodeGeneration(15);
            this.allocateGeneration(0);
        }

        public ByteBuffer acquire() {
            this.m_cAcquired.incrementAndGet();
            ByteBuffer buffer = this.m_queue.poll();
            return buffer == null ? this.ensureBuffer() : buffer;
        }

        public void release(ByteBuffer buffer) {
            int nGeneration = this.decodeGeneration(buffer.capacity());
            if ((this.m_cReleased.incrementAndGet() & (long)this.m_cStatsFreq) == 0L) {
                this.recordUsage();
                this.evaluateCapacity(true);
            }
            int cCurrentGen = this.getGenerationId();
            if (nGeneration != 15 && (nGeneration <= cCurrentGen || cCurrentGen == -1)) {
                buffer.clear();
                this.m_queue.offer(buffer);
            } else {
                SegmentedBufferManager.this.m_allocator.release(buffer);
            }
        }

        @Override
        public void dispose() {
            this.trim(0);
        }

        public int getBufferSize() {
            return this.m_cbBuffer;
        }

        public int getGenerationSize() {
            return this.m_cBufferGen;
        }

        private int getGenerationId() {
            return this.m_cGeneration.get();
        }

        private int getAcquired() {
            return Math.max(0, (int)(this.m_cAcquired.get() - this.m_cReleased.get()));
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private ByteBuffer ensureBuffer() {
            AtomicInteger atomicGen = this.m_cGeneration;
            ConcurrentLinkedQueue<ByteBuffer> queue = this.m_queue;
            block7: while (true) {
                int cGen = atomicGen.get();
                ByteBuffer buffer = queue.poll();
                if (buffer != null) {
                    return buffer;
                }
                switch (cGen) {
                    case 15: {
                        return this.allocateNonPooledBuffer();
                    }
                    case -1: {
                        continue block7;
                    }
                }
                if (!atomicGen.compareAndSet(cGen, -1)) continue;
                try {
                    this.allocateGeneration(cGen + 1);
                    continue;
                }
                finally {
                    atomicGen.set(cGen + 1);
                    continue;
                }
                break;
            }
        }

        private ByteBuffer allocateNonPooledBuffer() {
            return SegmentedBufferManager.this.m_allocator.allocate(this.m_cbNonPooled);
        }

        private void allocateGeneration(int nGeneration) {
            block4: {
                int cbBuffer = this.encodeGeneration(nGeneration);
                ConcurrentLinkedQueue<ByteBuffer> queue = this.m_queue;
                if (nGeneration > 0) {
                    LOGGER.log(Level.FINE, "Growing SegmentedBufferManager segment '" + this.getBufferSize() + "' to " + (nGeneration + 1) + " generations");
                }
                this.m_ldtNextEvaluation = System.currentTimeMillis() + CLEANUP_FREQUENCY_MILLIS;
                try {
                    int c = this.getGenerationSize();
                    for (int i = 0; i < c; ++i) {
                        queue.offer(SegmentedBufferManager.this.m_allocator.allocate(cbBuffer));
                    }
                }
                catch (OutOfMemoryError e) {
                    if (!queue.isEmpty()) break block4;
                    throw e;
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void evaluateCapacity(boolean fEvalPeers) {
            long ldtNow = System.currentTimeMillis();
            int nGen = this.getGenerationId();
            if (nGen > 0 && ldtNow > this.m_ldtNextEvaluation && this.m_cGeneration.compareAndSet(nGen, -1)) {
                this.m_ldtNextEvaluation = ldtNow + CLEANUP_FREQUENCY_MILLIS;
                int nDesiredGen = Math.min(this.m_cMaxBuffers / this.getGenerationSize() + 1, 14);
                try {
                    int cCapacity = Math.min(nGen + 1, 15);
                    if (nDesiredGen < cCapacity && nDesiredGen != nGen) {
                        LOGGER.log(Level.FINE, "Shrinking SegmentedBufferManager segment '" + this.getBufferSize() + "' by " + (nGen - nDesiredGen) + " generation(s)");
                        if (nDesiredGen + 1 > nGen) {
                            this.trim(nDesiredGen);
                        }
                    }
                    if (fEvalPeers) {
                        for (PoolSegment pool : SegmentedBufferManager.this.m_aSegments) {
                            if (pool == this) continue;
                            pool.evaluateCapacity(false);
                        }
                    }
                }
                finally {
                    this.m_cMaxBuffers = 0;
                    this.m_cGeneration.set(nDesiredGen);
                }
            }
        }

        private void recordUsage() {
            this.m_cMaxBuffers = Math.max(this.m_cMaxBuffers, this.getAcquired());
        }

        private void trim(int nGeneration) {
            int cbCutoff = this.encodeGeneration(nGeneration + 1);
            try {
                Iterator<ByteBuffer> iter = this.m_queue.iterator();
                while (iter.hasNext()) {
                    ByteBuffer buf = iter.next();
                    if (buf == null || buf.capacity() < cbCutoff) continue;
                    iter.remove();
                }
            }
            catch (ConcurrentModificationException concurrentModificationException) {
                // empty catch block
            }
        }

        private int encodeGeneration(int nGenId) {
            return this.getBufferSize() | nGenId << 6;
        }

        private int decodeGeneration(int cbBuffer) {
            return (cbBuffer & 0x3C0) >> 6;
        }
    }
}

