package org.gridvise.coherence.cache.cachetree;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;

import org.gridvise.coherence.cache.entity.ICache;

import com.tangosol.util.Filter;
import com.tangosol.util.ValueExtractor;
import com.tangosol.util.extractor.ReflectionExtractor;
import com.tangosol.util.filter.AndFilter;
import com.tangosol.util.filter.EqualsFilter;
import com.tangosol.util.filter.PresentFilter;

public class TreeModel<K, V> {

	public static final String ROOT_VALUE = "ROOT_VALUE";
	private static final PresentFilter PRESENT_FILTER = new PresentFilter();

	private ICache<K, V> cache;

	public TreeModel(ICache<K, V> cache) {
		super();
		this.cache = cache;
	}

	public ICacheTreeNode<K, V> getRootNode(String[] extractorMethodNameArray) {
		List<ValueExtractor> vel = new ArrayList<ValueExtractor>();
		for (String m : extractorMethodNameArray) {
			vel.add(new ReflectionExtractor(m));
		}
		return getRootNode(vel);
	}

	public ICacheTreeNode<K, V> getRootNode(List<ValueExtractor> extrators) {
		return new Node<K, V>(ROOT_VALUE, this.cache, extrators, 0);

	}

	private class Node<K, V> implements ICacheTreeNode<K, V> {

		private ICache<K, V> cache;
		private List<ValueExtractor> extractorList;
		private final int extractorIndex;
		private Object extractedValue;
		private Node<K, V> parentNode;

		private Node(Object extractedValue, ICache<K, V> cache,
				List<ValueExtractor> extractorList, int extractorIndex) {
			this(extractedValue, cache, extractorList, extractorIndex, true);
		}

		private Node(Object extractedValue, ICache<K, V> cache,
				List<ValueExtractor> extractorList, int extractorIndex,
				boolean addIndexes) {
			super();
			this.extractedValue = extractedValue;
			this.cache = cache;
			this.extractorList = extractorList;
			this.extractorIndex = extractorIndex;
			if (addIndexes) {
				for (ValueExtractor ve : extractorList) {
					cache.addIndex(ve);
				}
			}
		}

		private Node(Object extractedValue, Node<K, V> parentNode) {
			this(extractedValue, parentNode.cache, parentNode.extractorList,
					parentNode.extractorIndex + 1);
			this.parentNode = parentNode;
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * org.gridvise.coherence.cache.treemodel.ICacheTreeNode#getChildNodes()
		 */
		@Override
		public List<ICacheTreeNode<K, V>> getChildNodes() {
			List<ICacheTreeNode<K, V>> returnList = new ArrayList<ICacheTreeNode<K, V>>();
			List<Object> extractedValues = getSortedDistinctValues();
			for (Object extractedValue : extractedValues) {
				returnList.add(new Node<K, V>(extractedValue, this));
			}
			return returnList;
		}

		private List<Object> getSortedDistinctValues() {
			if (isLeaf()) {
				return new ArrayList<Object>();
			} else {
				List<Object> extractedValues = new ArrayList<Object>(
						cache.distictValues(buildFilter(this),
								getValueExtractorForDistinctValues()));
				Collections.sort(extractedValues, new Comparator<Object>() {
					@Override
					public int compare(Object arg0, Object arg1) {
						return arg0.toString().compareTo(arg1.toString());
					}

				});
				return extractedValues;
			}
		}

		/*
		 * (non-Javadoc)
		 * 
		 * @see
		 * org.gridvise.coherence.cache.treemodel.ICacheTreeNode#getExtractedValue
		 * ()
		 */
		@Override
		public Object getExtractedValue() {
			return extractedValue;
		}

		@Override
		public Map<K, V> getValues() {
			return this.cache.values(buildFilter(this));
		}

		@Override
		public Collection<K> getKeys() {
			Filter buildFilter = buildFilter(this);
			return this.cache.keySet(buildFilter);
		}

		private ValueExtractor getValueExtractorForKeysAndValues() {
			return this.extractorList.get(this.extractorIndex - 1);
		}

		private ValueExtractor getValueExtractorForDistinctValues() {
			return this.extractorList.get(this.extractorIndex);
		}

		private boolean isLeaf() {
			return this.extractorIndex >= this.extractorList.size();
		}

		private Filter buildFilter(Node<K, V> node) {
			if (node.parentNode != null) {
				Filter f = new EqualsFilter(
						node.getValueExtractorForKeysAndValues(),
						node.getExtractedValue());
				return new AndFilter(f, buildFilter(node.parentNode));
			} else {
				return PRESENT_FILTER;
			}
		}

		@Override
		public String toString() {
			return "Node [extractorList=" + extractorList + ", extractorIndex="
					+ extractorIndex + ", extractedValue=" + extractedValue
					+ "]";
		}

		@Override
		public String getId() {
			return this.toString();
		}

	}

}
