package org.gridvise.coherence.cache.entity;

import java.util.Collection;
import java.util.Date;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;

import com.tangosol.net.NamedCache;
import com.tangosol.util.Filter;
import com.tangosol.util.ValueExtractor;
import com.tangosol.util.aggregator.DistinctValues;
import com.tangosol.util.extractor.KeyExtractor;
import com.tangosol.util.extractor.ReflectionExtractor;
import com.tangosol.util.filter.EqualsFilter;
import com.tangosol.util.filter.LessEqualsFilter;

public abstract class AbstractDateSensitiveCache<K, V> extends AbstractCache<K, V> implements
        IDateSensitiveCache<K, V> {

    private static final ValueExtractor KEY_EFFECTIVE_DATE_EXTRACTOR = new KeyExtractor("getDate");
    private static final ValueExtractor VALUE_EFFECTIVE_DATE_EXTRACTOR = new ReflectionExtractor("getDate");

    private boolean keyBased;

    public AbstractDateSensitiveCache() {
        this(true);
    }

    public AbstractDateSensitiveCache(boolean keyBased) {
        this.keyBased = keyBased;
    }

    @Override
    public void evictDaysOlderOrEquals(Date effectiveDate) {

        Collection<K> keys = this.keySet(new LessEqualsFilter(getEffectiveDateExtractor(), effectiveDate));
        this.remove(keys);

    }

    @Override
    public ValueExtractor getEffectiveDateExtractor() {
        return this.keyBased ? KEY_EFFECTIVE_DATE_EXTRACTOR : VALUE_EFFECTIVE_DATE_EXTRACTOR;
    }

    @Override
    public Filter getEffectiveDateFilter(Date effectiveDate) {
        return new EqualsFilter(getEffectiveDateExtractor(), effectiveDate);
    }

    @Override
    public void evictDay(Date effectiveDate) {
        this.remove(getKeysForDate(effectiveDate));
    }

    @SuppressWarnings("unchecked")
    @Override
    public Collection<V> getValuesForDate(Date effectiveDate) {
        return (Collection<V>) this.getCache().getAll(getKeysForDate(effectiveDate));
    }

    @Override
    public Collection<K> getKeysForDate(Date effectiveDate) {
        return this.keySet(new EqualsFilter(this.getEffectiveDateExtractor(), effectiveDate));
    }

    @Override
    public Date getLatestEffectiveDate() {
        return getSortedEffectiveDates().last();
    }

    @Override
    public Date getEarliestEffectiveDate() {
        return getSortedEffectiveDates().first();
    }

    @Override
    public SortedSet<Date> getSortedEffectiveDates() {
        SortedSet<Date> sortedEffectiveDates = new TreeSet<Date>();
        sortedEffectiveDates.addAll(this.getEffectiveDates());
        return sortedEffectiveDates;
    }

    @SuppressWarnings("unchecked")
    @Override
    public Set<Date> getEffectiveDates() {
        return (Set<Date>) this.getCache().aggregate(PRESENT_FILTER,
                new DistinctValues(this.getEffectiveDateExtractor()));
    }

    @Override
    protected void addIndexes(NamedCache namedCache) {
        addIndex(getEffectiveDateExtractor(), namedCache);
    }

    @Override
    public <E> Collection<E> distictValuesForDate(Date effectiveDate, Enum<?> extractMethodName) {
        Filter effectiveDateFilter = new EqualsFilter(this.getEffectiveDateExtractor(), effectiveDate);
        return super.distictValues(effectiveDateFilter, extractMethodName);
    }

    @Override
    public <E> Collection<E> distictSetValuesForDate(Date effectiveDate, Enum<?> setExtractMethodName) {
        Filter effectiveDateFilter = new EqualsFilter(this.getEffectiveDateExtractor(), effectiveDate);
        return super.distictSetValues(effectiveDateFilter, setExtractMethodName);
    }

}