package org.gridvise.coherence.cache.config;

import java.io.IOException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import org.apache.log4j.Logger;
import org.apache.log4j.spi.Configurator;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;
import org.springframework.core.io.Resource;

/**
 * Provides certain default values but also has the option to work of a
 * configuration file called 'coherence-utils.properties' or a name provided via
 * system propery 'coherence-utils.configfile'. Also config items can be
 * overriden via system properties.
 * 
 * @author christoph
 * 
 */
public class Configuration {

	private static final String SPRING_CONFIG_XML = "spring-config.xml";
	public static String CONFIG_FILE_NAME = "coherence-utils.properties";
	public static String SYSPROP_CONFIG_FILE_NAME = "coherence-utils.configfile";

	private static final Map<String, Map<String, String>> PROPERTIES_MAP = new HashMap<String, Map<String, String>>();


	private interface InbuildProperty {
		String getSystemPropertyName();

		String getDefaultValue();
	}

	enum INBUILD_PROPERTIES implements InbuildProperty {

		SCHEDULING_CACHE_NAME {
			@Override
			public String getDefaultValue() {
				return "scheduling";
			}

			@Override
			public String getSystemPropertyName() {
				return "coherence-utils.scheduling-cache-name";
			}

		}

	}

	public static final Logger LOG = Logger.getLogger(Configurator.class);

	static {
		initializeConfigFileName();
		loadPropertyFiles();
	}

	private static void loadPropertyFiles() {
		ApplicationContext context = new ClassPathXmlApplicationContext(
				SPRING_CONFIG_XML);

		Resource[] xmlResources;
		try {
			xmlResources = context.getResources("classpath:/**/*.properties");
			for (Resource r : xmlResources) {
				loadProperties(r);
			}
			overrideWithSystemProperties();
			LOG.info("initialized Configurator with "
					+ xmlResources.length
					+ " config files plus system properies and environment varialbes\n "
					+ PROPERTIES_MAP);
		} catch (IOException e) {
			e.printStackTrace();
		}
	}

	private static void loadProperties(Resource propertiesResource)
			throws IOException {
		LOG.info("loading configuration from "
				+ propertiesResource.getFile().getAbsolutePath());
		Properties properties = new Properties();
		properties.load(propertiesResource.getInputStream());
		PROPERTIES_MAP.put(propertiesResource.getFilename(), asMap(properties));
	}

	private static Map<String, String> asMap(Properties properties) {
		Map<String, String> map = new HashMap<String, String>();
		Enumeration<?> keys = properties.propertyNames();
		while (keys.hasMoreElements()) {
			String key = keys.nextElement().toString();
			map.put(key, properties.getProperty(key));
		}
		return map;
	}

	private static void initializeConfigFileName() {
		if (System.getProperty(SYSPROP_CONFIG_FILE_NAME) != null) {
			CONFIG_FILE_NAME = System.getProperty(SYSPROP_CONFIG_FILE_NAME);
		}
		if (System.getenv(SYSPROP_CONFIG_FILE_NAME) != null) {
			CONFIG_FILE_NAME = System.getenv(SYSPROP_CONFIG_FILE_NAME);
		}
	}

	private static void put(String propName, String propValue) {
		put(CONFIG_FILE_NAME, propName, propValue);
	}

	private static void put(String propFileName, String propName,
			String propValue) {
		Map<String, String> properties = getPropertiesMap(propFileName);
		properties.put(propName, propValue);
	}

	private static void putAll(Map<String, String> properties) {
		putAll(CONFIG_FILE_NAME, properties);
	}

	private static void putAll(String propFileName,
			Map<String, String> propertiesIncoming) {
		Map<String, String> properties = getPropertiesMap(propFileName);
		properties.putAll(propertiesIncoming);
	}

	private static Map<String, String> getPropertiesMap(String propFileName) {
		Map<String, String> properties = PROPERTIES_MAP.get(propFileName);
		if (properties == null) {
			properties = new HashMap<String, String>();
			PROPERTIES_MAP.put(propFileName, properties);
		}
		return properties;
	}

	private static void overrideWithSystemProperties() {
		for (INBUILD_PROPERTIES ib : INBUILD_PROPERTIES.values()) {
			put(ib.getSystemPropertyName(), ib.getDefaultValue());
		}
		putAll(System.getenv());
		putAll(asMap(System.getProperties()));
	}

	public static String get(String fileName, String name) {
		Map<String, String> props = PROPERTIES_MAP.get(fileName);
		if (props != null)
		    return props.get(name);
        else
            return loadAndGet(fileName, name);
	}

    private static String loadAndGet(String fileName, String name) {

        Properties p  = new Properties();
        try {
            p.load(Configuration.class.getClass().getClassLoader().getResourceAsStream(fileName));
        } catch (IOException e) {
            LOG.warn("couldn't read property file "+fileName+" for "+name);
        }
        String v = p.getProperty(name);
        if(v!=null){
            PROPERTIES_MAP.put(fileName, asMap(p));
        }
        return v;

    }

    public static String get(String name) {
		return get(CONFIG_FILE_NAME, name);
	}

	public static String getSchedulingCacheName() {
		return get(INBUILD_PROPERTIES.SCHEDULING_CACHE_NAME
				.getSystemPropertyName());
	}

	public static Set<String> getPropertyFiles() {
		return PROPERTIES_MAP.keySet();
	}

	public static Set<String> getPropertyFiles(String fileName) {
		Map<String, String> m = PROPERTIES_MAP.get(fileName);
		if (m == null) {
			return new HashSet<String>();
		} else {
			return m.keySet();
		}
	}

}
