package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNegative;

/**
 * PluginResources contains information about how many resources will be used for a plugin. The most common resources to specify are CPU and memory (RAM).
 * <p>
 * CPU resources are measured in cpu units. One cpu is equivalent to 1 vCPU/Core for cloud providers and 1 hyperthread on bare-metal Intel processors.
 * Also, fractional requests are allowed. A plugin that asks 0.5 CPU uses half as much CPU as one that asks for 1 CPU.
 * <p>
 * Memory resources are measured in Megabytes.
 * <p>
 * Here is an example to set resources for a plugin:
 * <blockquote><pre>
 *     Resources.builder()
 *         .maximumCpu("1.5")
 *         .maximumMemory("1024")
 *         .build();
 * </pre></blockquote>
 * In this example the plugin has a limit of 1 cpu and 1G of memory.
 */
public final class PluginResources {

    private final Float _maxCpu;
    private final Integer _maxMemory;

    private PluginResources(final Float maxCpu, final Integer maxMemory) {
        _maxCpu = maxCpu;
        _maxMemory = maxMemory;
    }

    public Float maximumCpu() {
        return _maxCpu;
    }

    public Integer maximumMemory() {
        return _maxMemory;
    }

    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder for {@link PluginResources}.
     */
    public static final class Builder {

        private Float _maxCpu;
        private Integer _maxMemory;

        private Builder() {
        }

        public Builder maximumCpu(final float quantity) {
            _maxCpu = argNotNegative("quantity", quantity);
            return this;
        }

        public Builder maximumMemory(final int quantity) {
            _maxMemory = argNotNegative("quantity", quantity);
            return this;
        }

        public PluginResources build() {
            return new PluginResources(_maxCpu, _maxMemory);
        }
    }
}
