package org.hansken.plugin.extraction.api;

import static java.util.Locale.ROOT;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;

/**
 * Identifier of a plugin, consisting of domain, category and name. Needs to be unique among all tools/plugins.
 *
 * @author Netherlands Forensic Institute
 */
public class PluginId {
    private final String _domain;
    private final String _category;
    private final String _name;

    /**
     * Create a unique identifier for a plugin, consisting of domain, category and name.
     *
     * @param domain   the domain of the organisation, for example "nfi.nl"
     * @param category the action group of the plugin, for example `extract`, `carve`, `classify` (read the SDK documentation for more details).
     * @param name     the name of the plugin, or in the classic sense, a description detailing the action(s) of the plugin. Note that the <b>name</b> can contain (forward) slashes.<br>
     *                 example: "nfi.nl/extract/ocr/detection/plugin".<br>
     *                 in this example <i>nfi.nl</i> is the domain, <i>extract</i> is the category, and <i>ocr/detection/plugin</i> is the name.
     */
    public PluginId(final String domain, final String category, final String name) {
        _domain = argNotEmpty("domain", domain);
        _category = argNotEmpty("category", category);
        _name = argNotEmpty("name", name);
    }

    public String domain() {
        return _domain;
    }

    public String category() {
        return _category;
    }

    public String name() {
        return _name;
    }

    @Override
    public String toString() {
        return (_domain + '/' + _category + '/' + _name).toLowerCase(ROOT);
    }
}
