package org.hansken.plugin.extraction.util;

import static java.util.Locale.ROOT;

import java.util.Collection;
import java.util.List;

/**
 * A collection of methods for defensively checking arguments passed to methods.
 */
public final class ArgChecks {

    private ArgChecks() {
    }

    /**
     * Check that the value with given name is not {@code null}, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     private final Author author;
     *
     *     public Book(final Author author) {
     *          // this throws a NullPointerException when the author is null
     *          this.author = argNotNull("author", author);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the value (most likely the argument name)
     * @param value the value itself
     * @param <T> the type of the value
     * @return the value, so it can be used to inline in an assignment
     * @throws NullPointerException if {@code value == null}
     */
    public static <T> T argNotNull(final String name, final T value) {
        if (value == null) {
            throw new NullPointerException(String.format(ROOT, "argument '%s' cannot be null", name));
        }
        return value;
    }

    /**
     * Check that the value with given name is of type {@code type}, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     public Book(final List<Object> authors) {
     *          // this throws a ClassCastException when authors contains a type other than Author.class
     *          argsIsType("author", authors, Author.class);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the value (most likely the argument name)
     * @param value the value itself
     * @param <T> the type of the value
     * @param type the class type we want to check against {@code value}
     * @return the value, so it can be used to inline in an assignment
     * @throws ClassCastException if {@code value.allMatch(type::isInstance)} is false
     */
    public static <T> List<T> argsIsType(final String name, final List<T> value, final Class<?> type) {
        for (final T t : value) {
            if (!type.isInstance(t)) {
                throw new ClassCastException(String.format(ROOT, "argument '%s' is not of type %s", name, type.getSimpleName()));
            }
        }
        return value;
    }

    /**
     * Check that not all values with given name are {@code null}, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     private final Author author;
     *
     *     public Book(final Author... authors) {
     *          // this throws a NullPointerException when the author is null
     *          this.author = argNotAllNull("authors", authors);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the values (most likely the argument name)
     * @param values the values, which can be different types
     * @return the values, so it can be used to inline in an assignment
     * @throws NullPointerException if {@code values.allMatch(Objects::isNull)}
     */
    public static Object[] argNotAllNull(final String name, final Object... values) {
        for (final Object o : values) {
            if (o != null) {
                return values;
            }
        }
        throw new NullPointerException(String.format(ROOT, "argument '%s' cannot all be null", name));
    }

    /**
     * Check that the array value with given name is not <b>empty</b>, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     private final Author[] author;
     *
     *     public Book(final Author[] author) {
     *          // this throws a NullPointerException when the author array is empty
     *          this.author = argNotEmpty("author", author);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the value (most likely the argument name)
     * @param value the value itself
     * @param <T> the type of the value
     * @return the value, so it can be used to inline in an assignment
     * @throws IllegalArgumentException if {@code value == null || value.length == 0}
     */
    public static <T> T[] argNotEmpty(final String name, final T[] value) {
        if (value == null || value.length == 0) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be empty", name));
        }
        return value;
    }

    /**
     * Check that the collection value with given name is not <b>empty</b>, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     private final Collection<Author> author;
     *
     *     public Book(final Collection<Author> author) {
     *          // this throws a NullPointerException when the author collection is empty
     *          this.author = argNotEmpty("author", author);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the value (most likely the argument name)
     * @param value the value itself
     * @param <T> the type of the value
     * @return the value, so it can be used to inline in an assignment
     * @throws IllegalArgumentException if {@code value == null || value.isEmpty()}
     */
    public static <T> Collection<T> argNotEmpty(final String name, final Collection<T> value) {
        if (value == null || value.isEmpty()) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be empty", name));
        }
        return value;
    }

    /**
     * Check that the string with given name is not {@code null} or empty, otherwise throw an exception.
     * <p>
     * Example usage:
     * <pre>
     * {@code
     *     private final String title;
     *
     *     public Book(final String title) {
     *          // this throws a NullPointerException when the author is null
     *          // or an IllegalArgumentException when author is an empty String
     *          this.title = argNotEmpty("title", title);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the string (most likely the argument name)
     * @param value the string itself
     * @return the string, so it can be used to inline in an assignment
     * @throws NullPointerException if {@code value == null}
     * @throws IllegalArgumentException if {@code value.isEmpty()}
     */
    public static String argNotEmpty(final String name, final String value) {
        argNotNull(name, value);
        if (value.isEmpty()) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be empty", name));
        }
        return value;
    }

    /**
     * Check that the int with given name is not negative, otherwise throw an exception.
     * Example usage:
     * <pre>
     * {@code
     *     private final int pageCount;
     *
     *     public Book(final int pageCount) {
     *          // this throws an IllegalArgumentException when pageCount is negative
     *          this.pageCount = argNotNegative("pageCount", pageCount);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the int (most likely the argument name)
     * @param value the int itself
     * @return the int, so it can be used in an inline assignment
     * @throws IllegalArgumentException if {@code value < 0}
     */
    public static int argNotNegative(final String name, final int value) {
        if (value < 0) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be negative: %d", name, value));
        }
        return value;
    }

    /**
     * Check that the long with given name is not negative, otherwise throw an exception.
     * Example usage:
     * <pre>
     * {@code
     *     private final long pageCount;
     *
     *     public Book(final long pageCount) {
     *          // this throws an IllegalArgumentException when pageCount is negative
     *          this.pageCount = argNotNegative("pageCount", pageCount);
     *     }
     * }
     * </pre>
     *
     * @param name a descriptive name for the long (most likely the argument name)
     * @param value the long itself
     * @return the long, so it can be used in an inline assignment
     * @throws IllegalArgumentException if {@code value < 0}
     */
    public static long argNotNegative(final String name, final long value) {
        if (value < 0) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be negative: %d", name, value));
        }
        return value;
    }

    /**
     * Check that the float with given name is not negative, otherwise throw an exception.
     * Example usage:
     * <pre>
     * {@code
     *     private final float pageCount;
     *
     *     public Book(final float pageCount) {
     *          // this throws an IllegalArgumentException when pageCount is negative
     *          this.pageCount = argNotNegative("pageCount", pageCount);
     *     }
     * }
     * </pre>
     *
     * @param name  a descriptive name for the float (most likely the argument name)
     * @param value the float itself
     * @return the float, so it can be used in an inline assignment
     * @throws IllegalArgumentException if {@code value < 0}
     */
    public static float argNotNegative(final String name, final float value) {
        if (value < 0) {
            throw new IllegalArgumentException(String.format(ROOT, "argument '%s' cannot be negative: %d", name, value));
        }
        return value;
    }
}