package org.hansken.plugin.extraction.api;

import java.util.concurrent.ExecutionException;

/**
 * Allows searching for traces within the scope of the process function.
 */
public interface TraceSearcher {
    /**
     * Searches in Hansken for Traces matching provided query, scoped to the image under extraction.
     *
     * @param query Search query to match traces. This is a HQL query.
     * @param count Maximum number of traces to return.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    default SearchResult search(String query, int count) throws InterruptedException, ExecutionException {
        return search(query, count, SearchScope.IMAGE);
    }

    /**
     * Searches in Hansken for Traces matching provided query.
     *
     * @param query Search query to match traces. This is a HQL query.
     * @param count Maximum number of traces to return.
     * @param scope Scope the search to image or project level.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    SearchResult search(String query, int count, SearchScope scope) throws InterruptedException, ExecutionException;

    /**
     * Scope options for scoping a search to an image or project level.
     */
    enum SearchScope {
        /**
         * Searches will be limited to the traces in the same <strong>image</strong> that is being extracted.
         */
        IMAGE,

        /**
         * Searches will be limited to the traces in the same <strong>project</strong> that is being extracted.
         */
        PROJECT
    }
}
