package org.hansken.plugin.extraction.api.transformations;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

/**
 * A {@link RangedDataTransformation} describes a data transformation consisting of a list of {@link DataRange}s.
 *
 * @author Netherlands Forensic Institute
 */
public class RangedDataTransformation implements DataTransformation {

    private List<DataRange> _ranges;

    /**
     * Create a new {@link RangedDataTransformation} given a list of {@link DataRange}s.
     * @param ranges the list ranges that define the {@link RangedDataTransformation}
     */
    public RangedDataTransformation(final List<DataRange> ranges) {
        _ranges = ranges;
    }

    /**
     * Create a new {@link RangedDataTransformation} given one or more {@link DataRange}s.
     * @param ranges the list ranges that define the {@link RangedDataTransformation}
     */
    public RangedDataTransformation(final DataRange... ranges) {
        _ranges = Arrays.asList(ranges);
    }

    public List<DataRange> getRanges() {
        return _ranges;
    }

    public void setRanges(final List<DataRange> ranges) {
        _ranges = ranges;
    }

    /** Create a {@link Builder} that can build a {@link RangedDataTransformation}.
     * @return the builder
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder for creating {@link RangedDataTransformation} using a syntax that is sometimes shorter than using
     * constructors.
     */
    public static class Builder {
        private final List<DataRange> _ranges = new ArrayList<>();

        public Builder add(final List<DataRange> dataRange) {
            _ranges.addAll(dataRange);
            return this;
        }

        public Builder add(final DataRange... dataRange) {
            add(List.of(dataRange));
            return this;
        }

        public Builder addRange(final long offset, final long length) {
            _ranges.add(new DataRange(offset, length));
            return this;
        }

        public RangedDataTransformation build() {
            return new RangedDataTransformation(_ranges);
        }
    }
}