package org.hansken.plugin.extraction.api;

import java.io.IOException;
import java.util.concurrent.ExecutionException;

import org.hansken.plugin.extraction.util.ThrowingConsumer;

/**
 * Deferred meta extraction plugins can be used by Hansken to process {@link Trace traces} during the extraction process.
 * A processed trace can be enriched with new information and new child traces can also be created.
 * <p>
 * The difference between this and a normal {@link ExtractionPlugin} is that this plugin is able to run a secondary query
 * for traces and combine the results with previously retrieved traces, and that this plugin does not receive or
 * process any data, only a trace itself.
 * <p>
 * When a plugin matches on the trace which is currently processed by Hansken
 * (for example, because it has certain properties), the plugin will receive the
 * matched trace in order to process it (see {@link #process(Trace, TraceSearcher)}).
 * <p>
 * <strong>Note for Hansken core developers:</strong> specifying {@code 'meta'} in the matcher is
 * not necessary, the framework takes care of this.
 */
public abstract class DeferredMetaExtractionPlugin implements DeferredExtractionPlugin {

    @Override
    public void process(final Trace trace, final DataContext dataContext, final TraceSearcher searcher) throws ExecutionException, InterruptedException, IOException {
        process(trace, searcher);
    }

    /**
     * Start processing a {@link Trace trace}.
     * When processing a given trace, new properties may be set on it. New children can be added using
     * {@link Trace#newChild(String, ThrowingConsumer)}.
     * <p>
     * <strong>Note:</strong> the given trace should only be modified within the scope of this method.
     * Any modifications afterward may be guarded against or result in undefined behaviour.
     *
     * @param trace the trace to process
     * @param searcher the searcher for the trace
     * @throws ExecutionException when an exception occurs while searching for traces
     * @throws InterruptedException when a thread gets interrupted while searching for traces
     * @throws IOException when an I/O exception occurs
     **/
    public abstract void process(Trace trace, TraceSearcher searcher) throws ExecutionException, InterruptedException, IOException;
}
