package org.hansken.plugin.extraction.api;

import static java.util.Locale.ROOT;

import static org.hansken.plugin.extraction.api.PluginType.DEFERRED_EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.api.PluginType.DEFERRED_META_EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.api.PluginType.EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.api.PluginType.META_EXTRACTION_PLUGIN;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;
import static org.slf4j.LoggerFactory.getLogger;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;

/**
 * Information about an {@link ExtractionPlugin extraction plugin}, such as the author or
 * a human-readable description.
 * <p>
 * Example PluginInfo:
 * <blockquote><pre>
 *     PluginInfo.builderFor(this)
 *         .id("nfi.nl", "digest", "sha111")
 *         .pluginVersion("0.2.1")
 *         .description("Calculates hashes from data streams.")
 *         .author(Author.builder()
 *             .name("name")
 *             .email("test@email.com")
 *             .organisation("organisation")
 *             .build())
 *         .maturityLevel(MaturityLevel.PROOF_OF_CONCEPT)
 *         .hqlMatcher("hql-lite query")
 *         .webpageUrl("https://github.com/myOrg/digestPlugin")
 *         .license("Apache License 2.0")
 *         .resources(Resources.builder()
 *             .maximumCpu(1)
 *             .maximumMemory(1000)
 *             .build())
 *         .build();
 * </pre></blockquote>
 */
public final class PluginInfo {
    private static final Logger LOG = getLogger(PluginInfo.class);

    private final PluginType _type;
    @Deprecated
    private final String _name;
    private final String _pluginVersion;
    private final String _description;
    private final Author _author;
    private final MaturityLevel _maturityLevel;
    private final String _hqlMatcher;
    private final String _webpageUrl;
    private final int _deferredIterations;
    private final PluginId _id;
    private final String _license;
    private final PluginResources _resources;
    private final List<TransformerLabel> _transformers;

    private PluginInfo(final PluginType type,
                       final String name,
                       final String pluginVersion,
                       final String description,
                       final Author author,
                       final MaturityLevel maturityLevel,
                       final String hqlMatcher,
                       final String webpageUrl,
                       final int deferredIterations,
                       final PluginId id,
                       final String license,
                       final PluginResources resources,
                       final List<TransformerLabel> transformers) {
        if ((name == null || name.isEmpty()) && id == null) {
            throw new NullPointerException("argument 'id' cannot be null");
        }
        // if name == null, we assume this is version > 0.4.0, and the other fields are required
        _name = name;
        _id = (name == null ? argNotNull("id", id) : id);
        _license = (name == null ? argNotEmpty("license", license) : license);

        _type = argNotNull("type", type);
        _pluginVersion = argNotNull("pluginVersion", pluginVersion);
        _description = argNotNull("description", description);
        _author = argNotNull("author", author);
        _maturityLevel = argNotNull("maturityLevel", maturityLevel);
        _hqlMatcher = argNotNull("hqlMatcher", hqlMatcher);
        _webpageUrl = argNotNull("webpageUrl", webpageUrl);
        if (deferredIterations < 1 || deferredIterations > 20) {
            throw new IllegalArgumentException(String.format(ROOT, "Invalid value for deferredIterations: %s. Valid values are 1 <= 20.", deferredIterations));
        }
        _deferredIterations = deferredIterations;
        _resources = resources;
        _transformers = transformers;
    }

    /**
     * Start creating new {@link PluginInfo plugin information} for the given plugin.
     *
     * @param plugin the plugin to create the plugin information for
     * @return a builder for plugin metadata
     */
    public static Builder builderFor(final BaseExtractionPlugin plugin) {
        argNotNull("plugin", plugin);
        return switch (plugin) {
            case final MetaExtractionPlugin meta -> builderFor(META_EXTRACTION_PLUGIN);
            case final DeferredMetaExtractionPlugin deferredMeta -> builderFor(DEFERRED_META_EXTRACTION_PLUGIN);
            case final DeferredExtractionPlugin deferred -> builderFor(DEFERRED_EXTRACTION_PLUGIN);
            case final ExtractionPlugin extractionPlugin -> builderFor(EXTRACTION_PLUGIN);
            default -> throw new RuntimeException("Unsupported type of plugin: " + plugin.getClass().getName());
        };
    }

    /**
     * Start creating new {@link PluginInfo plugin information} for a plugin of given type.
     *
     * @param type the type of plugin to create the plugin information for
     * @return a builder for plugin metadata
     */
    public static Builder builderFor(final PluginType type) {
        return new Builder(argNotNull("type", type));
    }

    /**
     * Get the {@link PluginType type} of this {@link ExtractionPlugin plugin}.
     *
     * @return the type of this plugin
     */
    public PluginType pluginType() {
        return _type;
    }

    /**
     * Get the name of this {@link ExtractionPlugin plugin}.
     *
     * @return the name of this plugin
     * @deprecated Use {@link #id()} instead.
     */
    @Deprecated
    public String name() {
        return _name;
    }

    /**
     * Get the full name of this {@link ExtractionPlugin plugin}, based on the id. This name will be shown in the
     * Hansken GUI.
     *
     * @return the full name of this plugin, based on the id
     */
    public String fullName() {
        if (_id != null) {
            return _id.toString();
        }
        if (!_name.isBlank()) {
            LOG.warn("The full name of the plugin is based on the deprecated field 'name' because 'id' was not set. " +
                "Make sure you use the latest version of the Extraction Plugins SDK and consider setting 'id'.");
            return _name;
        }
        throw new IllegalArgumentException("PluginInfo has no fullName. Set the id.");
    }

    /**
     * Get the version of this {@link ExtractionPlugin plugin}.
     *
     * @return the version of this plugin
     */
    public String pluginVersion() {
        return _pluginVersion;
    }

    /**
     * Get a human readable description of this plugin.
     *
     * @return a description of this plugin
     */
    public String description() {
        return _description;
    }

    /**
     * Get the {@link Author author} of this plugin.
     *
     * @return the author of this plugin.
     */
    public Author author() {
        return _author;
    }

    /**
     * Get the {@link MaturityLevel maturity level} of this plugin.
     *
     * @return the maturity level of this plugin
     */
    public MaturityLevel maturityLevel() {
        return _maturityLevel;
    }

    /**
     * Get the hqlMatcher of this plugin in string format.
     *
     * @return the matcher of this plugin
     */
    public String hqlMatcher() {
        return _hqlMatcher;
    }

    /**
     * Get the url of this plugin, could point to git repo or webpage that explains the plugin.
     *
     * @return the url of this plugin
     */
    public String webpageUrl() {
        return _webpageUrl;
    }

    /**
     * Get the number of extraction iterations before the deferred plugin can be applied on traces.
     * Only relevant for deferred plugins; default value is 1.
     *
     * @return the number of iterations set of this plugin
     */
    public int deferredIterations() {
        return _deferredIterations;
    }

    /**
     * Get the the unique id of this plugin, consisting of domain, category and name.
     *
     * @return the url of this plugin
     */
    public PluginId id() {
        return _id;
    }

    /**
     * Get the name of the license of this plugin.
     *
     * @return the name of the license of this plugin
     */
    public String license() {
        return _license;
    }

    /**
     * Get a list of possible transform methods this plugin provides.
     *
     * @return list of transform methods
     */
    public List<TransformerLabel> transformers() {
        return _transformers;
    }

    /**
     * Get the {@link PluginResources resources} of this plugin.
     *
     * @return the resources of this plugin
     */
    public PluginResources resources() {
        return _resources;
    }

    /**
     * A builder for {@link PluginInfo plugin information}.
     * <p>
     * <strong>Note: </strong> all methods throw a {@link NullPointerException} when {@code null}
     * is passed to them.
     */
    public static final class Builder {
        private static final int DEFERRED_ITERATIONS_DEFAULT = 1;

        private final PluginType _type;
        private String _name;
        private String _pluginVersion;
        private String _description;
        private Author _author;
        private MaturityLevel _maturityLevel;
        private String _hqlMatcher;
        private String _webpageUrl;
        private int _deferredIterations;
        private PluginId _id;
        private String _license;
        private PluginResources _resources;
        private List<TransformerLabel> _transformers;

        Builder(final PluginType type) {
            this(type, "/extraction-plugin-build.properties");
        }

        Builder(final PluginType type, final String pluginInfoResourceName) {
            _type = argNotNull("type", type);
            _deferredIterations = DEFERRED_ITERATIONS_DEFAULT;

            if (pluginInfoResourceName != null) {
                loadFromProperties(pluginInfoResourceName);
            }
        }

        private void loadFromProperties(final String pluginResourceName) {
            // try to load the plugin version from an extraction-plugin-build.properties file
            //  this file is generated for plugins when using the super-pom as a basis for your plugin
            //  the version can be overwritten by the plugin dev if desired by calling the setPluginVersion(..) method
            try (InputStream buildInfoStream = PluginInfo.class.getResourceAsStream(pluginResourceName)) {
                if (buildInfoStream != null) {
                    final Properties buildProperties = new Properties();
                    buildProperties.load(buildInfoStream);
                    if (buildProperties.containsKey("project.version")) {
                        _pluginVersion = buildProperties.getProperty("project.version");
                    }
                }
            }
            catch (final IOException e) {
                // pass
            }

        }

        /**
         * Set the name of this {@link ExtractionPlugin plugin}.
         *
         * @param name the name
         * @return {@code this}
         * @throws IllegalArgumentException if the name is empty
         * @deprecated Use {@link #id(PluginId)} instead.
         */
        @Deprecated
        public Builder name(final String name) {
            _name = argNotEmpty("name", name);
            return this;
        }

        /**
         * Set the {@link PluginInfo#pluginVersion() plugin version}.
         *
         * @param version the plugin version
         * @return {@code this}
         * @throws IllegalArgumentException if the version is empty
         */
        public Builder pluginVersion(final String version) {
            _pluginVersion = argNotEmpty("version", version);
            return this;
        }

        /**
         * Set the {@link PluginInfo#description() description}.
         *
         * @param description the description
         * @return {@code this}
         * @throws IllegalArgumentException if the description is empty
         */
        public Builder description(final String description) {
            _description = argNotEmpty("description", description);
            return this;
        }

        /**
         * Set the {@link PluginInfo#author() author}.
         *
         * @param author the author
         * @return {@code this}
         */
        public Builder author(final Author author) {
            _author = argNotNull("author", author);
            return this;
        }

        /**
         * Set the {@link PluginInfo#maturityLevel() maturity level}.
         *
         * @param maturityLevel the maturity level
         * @return {@code this}
         */
        public Builder maturityLevel(final MaturityLevel maturityLevel) {
            _maturityLevel = argNotNull("maturityLevel", maturityLevel);
            return this;
        }

        /**
         * Set the hqlMatcher query in string format.
         *
         * @param hqlMatcher the matcher
         * @return {@code this}
         */
        public Builder hqlMatcher(final String hqlMatcher) {
            _hqlMatcher = argNotNull("hqlMatcher", hqlMatcher);
            return this;
        }

        /**
         * Set the url to a webpage that belongs to this plugin. This can also be a link to a webpage of the git repository of the remote plugin.
         *
         * @param webpageUrl url to webpage
         * @return {@code this}
         */
        public Builder webpageUrl(final String webpageUrl) {
            _webpageUrl = argNotNull("webpageUrl", webpageUrl);
            return this;
        }

        /**
         * Set the number of extraction iterations needed for this deferred plugin.
         * Only relevant for deferred plugins.
         * If this method is not called upon, default number of iterations will be set to 1.
         *
         * @param deferredIterations number of iterations needed for this plugin. Should be between 1 and 20
         * @return {@code this}
         */
        public Builder deferredIterations(final int deferredIterations) {
            _deferredIterations = deferredIterations;
            return this;
        }

        /**
         * Set the unique id of this plugin, consisting of domain, category and name.
         * <p>
         * example: "nfi.nl/extract/ocr/detection/plugin".<br>
         * in this example <i>nfi.nl</i> is the domain, <i>extract</i> is the category, and <i>ocr/detection/plugin</i> is the name.
         *
         * @param domain   the domain of the organisation, for example "nfi.nl"
         * @param category the action group of the plugin, for example `extract`, `carve`, `classify` (read the SDK documentation for more details).
         * @param name     the name of the plugin, or in the classic sense, a description detailing the action(s) of the plugin. Note that the <b>name</b> can contain (forward) slashes.<br>
         * @return {@code this}
         */
        public Builder id(final String domain, final String category, final String name) {
            return id(new PluginId(domain, category, name));
        }

        /**
         * Set the unique id of this plugin, consisting of domain, category and name.
         *
         * @param id the unique id of this plugin, consisting of domain, category and name.
         * @return {@code this}
         */
        public Builder id(final PluginId id) {
            _id = argNotNull("id", id);
            return this;
        }

        /**
         * Set the name of the license of this plugin.
         *
         * @param license name of the license of the plugin
         * @return {@code this}
         */
        public Builder license(final String license) {
            _license = argNotEmpty("license", license);
            return this;
        }

        /**
         * Set the resources of this plugin (optional).
         *
         * @param resources the resources of this plugin
         * @return {@code this}
         */
        public Builder resources(final PluginResources resources) {
            _resources = argNotNull("resources", resources);
            return this;
        }

        public Builder transformer(final List<TransformerLabel> transformers) {
            argNotEmpty("transformers", transformers);
            _transformers = transformers;
            return this;
        }

        /**
         * Create the {@link PluginInfo plugin information} from the properties set on this {@link Builder builder}.
         *
         * @return the created plugin information
         */
        public PluginInfo build() {
            return new PluginInfo(_type, _name, _pluginVersion, _description, _author, _maturityLevel, _hqlMatcher,
                _webpageUrl, _deferredIterations, _id, _license, _resources, _transformers);
        }
    }
}
