package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNegative;

import java.util.List;

/**
 * Represents search options including the search scope, the start offset of the results, and sorting preferences.
 *
 * @param scope the {@link SearchScope} indicating where the search should be performed.
 * @param start the starting offset of the search results.
 * @param sort  the list of {@link SearchSortOption} defining how the results should be sorted.
 */
public record SearchOptions(SearchScope scope, int start, List<SearchSortOption> sort) {
    /**
     * A default {@link SearchOptions} instance with.
     * <ul>
     *     <li>{@link SearchScope#IMAGE} as the scope</li>
     *     <li>{@code 0} as the start offset</li>
     *     <li>list of {@link SearchSortOption} containing just the uid in ascending order</li>
     * </ul>
     */
    public static final SearchOptions DEFAULT = SearchOptions.builder().build();

    /**
     * Creates a new {@link Builder} instance for constructing a {@code SearchOptions} object.
     *
     * @return a new {@code Builder}.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Builder for creating {@link SearchOptions} instances.
     */
    public static class Builder {
        private SearchScope _scope = SearchScope.IMAGE;
        private int _start;
        private List<SearchSortOption> _sort = List.of(SearchSortOption.builder().build());

        /**
         * Sets the search scope.
         *
         * @param scope the {@link SearchScope} where the search should be performed.
         * @return this builder instance for chaining.
         */
        public Builder scope(final SearchScope scope) {
            _scope = scope;
            return this;
        }

        /**
         * Sets the starting offset for the search results.
         *
         * @param start the start offset.
         * @return this builder instance for chaining.
         */
        public Builder offset(final int start) {
            _start = argNotNegative("start", start);
            return this;
        }

        /**
         * Sets the sort options for the search results.
         *
         * @param sort the list of {@link SearchSortOption}.
         * @return this builder instance for chaining.
         */
        public Builder sort(final List<SearchSortOption> sort) {
            _sort = sort;
            return this;
        }

        /**
         * Builds a new {@link SearchOptions} instance with the configured values.
         *
         * @return a new {@code SearchOptions}.
         */
        public SearchOptions build() {
            return new SearchOptions(_scope, _start, _sort);
        }
    }
}
