package org.hansken.plugin.extraction.api;

/**
 * Represents a sort option used in searches, including the field to sort by and the direction.
 *
 * @param field     the name of the field to sort on.
 * @param direction the sort direction (ascending or descending).
 */
public record SearchSortOption(String field, Direction direction) {
    /**
     * Creates a new {@link Builder} instance for constructing a {@code SearchSortOption}.
     *
     * @return a new {@code Builder}.
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * The sort direction.
     */
    public enum Direction {
        /**
         * Sort results in ascending order.
         */
        ASCENDING,

        /**
         * Sort results in descending order.
         */
        DESCENDING
    }

    /**
     * Builder for creating {@link SearchSortOption} instances.
     */
    public static class Builder {
        private String _field = "uid";
        private Direction _direction = Direction.ASCENDING;

        /**
         * Sets the field to sort by.
         *
         * @param field the field name to sort on.
         * @return this builder instance for chaining.
         */
        public Builder field(final String field) {
            _field = field;
            return this;
        }

        /**
         * Sets the sort direction.
         *
         * @param direction the sort direction.
         * @return this builder instance for chaining.
         */
        public Builder direction(final Direction direction) {
            _direction = direction;
            return this;
        }

        /**
         * Builds a new {@link SearchSortOption} instance with the configured values.
         *
         * @return a new {@code SearchSortOption}.
         */
        public SearchSortOption build() {
            return new SearchSortOption(_field, _direction);
        }
    }
}
