package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;

import org.hansken.plugin.extraction.api.transformations.DataTransformation;
import org.hansken.plugin.extraction.util.ThrowingConsumer;

/**
 * A trace contains information about processed data. A trace should conform to the trace model defined by Hansken.
 * <p>
 * A trace can have multiple types (e.g. file or chat). For these types, the trace can contain a set of properties and
 * associated values. A trace can be associated with a set of {@link RandomAccessData data sequences}.
 * During extraction, an {@link ExtractionPlugin extraction plugin} may receive a currently processed trace
 * and accompanying data sequence.
 * <p>
 * Implementers of a plugin should ensure that setting a property on a trace is valid as per the trace model
 * which is defined for the traces generated by the plugin.
 * <p>
 * <strong>Note:</strong> implementations are not required to implement any kind of thread safety.
 * It is up to the client to ensure this if necessary.
 *
 * @see ExtractionPlugin#process(Trace, DataContext)
 */
public interface Trace extends ImmutableTrace {
    /**
     * Add a type to this trace (for example: {@code "file"}). If this trace already had given type,
     * nothing changes.
     *
     * @param type the type to add
     * @return {@code this}
     */
    Trace addType(String type);

    /**
     * Set a property on this trace with a given {@code value}.
     *
     * @param name the name of the property to set
     * @param value the value of the property
     * @return {@code this}
     * @throws IllegalArgumentException if the property or value of the property is invalid as per the trace model
     */
    Trace set(String name, Object value);

    /**
     * Add a tracelet to the trace.
     * <p>
     * Example usage:
     * <pre>{@code
     *     trace.addTracelet("prediction", tracelet -> tracelet
     *         .set("type", "type")
     *         .set("model", "model")
     *         .set("label", "label")
     *         .set("confidence", .50));
     *     });
     * }</pre>
     *
     * @param type     the type of tracelet to add
     * @param callback callback to set the tracelet properties
     * @return {@code this}
     */
    default Trace addTracelet(final String type, final Consumer<TraceletBuilder> callback) {
        argNotEmpty("type", type);
        argNotNull("callback", callback);
        final List<TraceletProperty> properties = new ArrayList<>();
        callback.accept(new TraceletBuilder() {
            @Override
            public TraceletBuilder set(final String name, final Object value) {
                final String prefix = type + '.';
                final String fqn = name.startsWith(prefix) ? name : prefix + name;
                properties.add(new TraceletProperty(fqn, value));
                return this;
            }

            @SuppressWarnings("unchecked")
            @Override
            public <T> T get(final String name) {
                final String prefix = type + '.';
                final String fqn = name.startsWith(prefix) ? name : prefix + name;
                final Optional<Object> result = properties.stream()
                    .filter(prop -> prop.getName().equals(fqn))
                    .findFirst()
                    .map(TraceletProperty::getValue);
                return (T) result.orElse(null);
            }

            @Override
            public String getType() {
                return type;
            }
        });
        return addTracelet(new Tracelet(type, properties));
    }

    /**
     * Add a tracelet to the trace.
     *
     * @param tracelet the name and properties of the Tracelet
     * @return {@code this}
     * @deprecated use {@code addTracelet(type, callback)}
     */
    @Deprecated
    Trace addTracelet(Tracelet tracelet);

    /**
     * Retrieve the tracelets that were added by the plugin.
     * <p>
     * Note that this does not return the tracelets that were already present on the trace prior to the execution
     * of the plugin.
     *
     * @return the list of tracelets that were added by the plugin.
     */
    List<Tracelet> getNewTracelets();

    /**
     * Add a data stream of a given type to this {@link Trace} (for example, {@code 'raw'} or {@code 'html'}).
     * A trace can only have a single data stream for each data type. A callback function can be passed
     * which receives an {@link OutputStream} to write the data to.
     * <p>
     * <strong>Note:</strong> the received output stream should <i>not</i> be closed by the user. It should
     * also only be used within the scope of the callback, other usage may be guarded against or otherwise
     * result in undefined behaviour.
     * <p>
     * Example usage:
     * <pre>{@code
     *     final PacketSequencer sequencer = ...;
     *     final RandomAccessData input = dataContext.data();
     *
     *     trace.setData("packets", data -> {
     *         sequencer.process(input).forEach(packet -> {
     *             data.write(packet);
     *         });
     *     });
     * }</pre>
     *
     * @param dataType the type of the data stream to add
     * @param writer callback that receives the stream to write to as input
     * @return {@code this}
     * @throws IllegalArgumentException if this trace already had an associated data stream of given type
     * @throws IOException when the given callback throws one
     */
    Trace setData(String dataType, DataWriter writer) throws IOException;

    /**
     * Set a series of data transformations for a specific dataType.
     *
     * Transformations are pointers to actual raw data. They describe how data can be obtained by reading on certain
     * positions, using decryption, or combinations of these. The benefit of using Transformations is that they take up
     * less space than the actual data.
     *
     * @param dataType the type of the datastream
     * @param transformations the data transformations
     * @return {@code this}
     */
    default Trace setData(final String dataType, final DataTransformation... transformations) {
        return setData(dataType, Arrays.asList(transformations));
    }

    /**
     * Set a series of data transformations for a specific dataType.
     * @param dataType the type of the datastream
     * @param transformation the data transformations
     * @return {@code this}
     */
    Trace setData(String dataType, List<DataTransformation> transformation);

    /**
     * Add a data stream of a given type to this {@link Trace} (for example, {@code 'raw'} or {@code 'html'}).
     * A trace can only have a single data stream for each data type.
     * <p>
     * The given {@link InputStream} will not be closed by this method.
     * <p>
     * Example usage:
     * <pre>{@code
     *     final ThumbnailDetector detector = ...;
     *     final RandomAccessData input = dataContext.data();
     *     final InputStream thumbnails = detector.detect(input);
     *     trace.setData("preview", thumbnails);
     * }</pre>
     *
     * @param dataType the type of the data stream to add
     * @param stream the data stream itself
     * @return {@code this}
     * @throws IllegalArgumentException if this trace already had an associated data stream of given type
     * @throws IOException when an I/O error occurs while reading the input
     */
    default Trace setData(final String dataType, final InputStream stream) throws IOException {
        argNotNull("dataType", dataType);
        argNotNull("stream", stream);
        return setData(dataType, stream::transferTo);
    }

    /**
     * Adds a value to a property that's a List.
     * This method will create a new list if the property has no value yet.
     *
     * @param property the name of the property to add the value to
     * @param value the value to add to the list property
     * @param <T> the type of the value to set
     */
    default <T> void addValue(final String property, final T value) {
        argNotNull("property", property);
        argNotNull("value", value);

        List<T> values = this.get(property);
        // values could be the result of asList or emptyList, we really need a mutable list here
        if (values instanceof ArrayList) {
            values.add(value);
        }
        else {
            values = values == null ? new ArrayList<>() : new ArrayList<>(values);
            values.add(value);
            this.set(property, values);
        }
    }

    /**
     * Create and store new child {@link Trace trace} of this trace. A callback function
     * can be passed in order to enrich the created child trace (which will have no types
     * or properties set yet), or even recursively add new children under it. After the
     * trace goes out of scope of the callback, it can no longer be updated.
     * <p>
     * As an example, say we have a {@code Node} type with the following API
     *<pre>{@code
     *   interface Node {
     *     String name();
     *     List<Node> children();
     *   }
     *}</pre>
     * A recursive function for creating traces from a node, where the initial node
     * should be mapped to a single child under the trace being processed:
     *<pre>{@code
     *     public void process(final Trace trace, final DataContext dataContext) {
     *         final Node node = createNode(); // get the node from somewhere
     *         createTree(node, trace);
     *     }
     *
     *     void createTree(final Node childNode, final Trace parentTrace) {
     *         parentTrace.newChild(childNode.name(), child -> {
     *             // set information from child node on child trace
     *             child.type("file").set("file.name", "password.txt");
     *             // recursively create more children
     *             childNode.children().forEach(node -> {
     *                 createTree(node, child);
     *             });
     *         });
     *     }
     * }</pre>
     *
     * @param name the name of the child
     * @param enrichChildCallback callback that gets the new child trace as input
     * @return {@code this}
     * @throws IllegalStateException if the trace was not yet saved
     * @throws IOException when the given callback throws one
     */
    Trace newChild(String name, ThrowingConsumer<Trace, IOException> enrichChildCallback) throws IOException;

    /**
     * a TraceletProperty is a property of a Tracelet.
     */
    class TraceletProperty {
        private final String _name;
        private final Object _value;
        public TraceletProperty(final String name, final Object value) {
            _name = name;
            _value = value;
        }

        public String getName() {
            return _name;
        }

        public Object getValue() {
            return _value;
        }
    }

    interface TraceletBuilder {
        /**
         * Set a property on this tracelet with a given {@code value}.
         *
         * @param name  the name of the property to set
         * @param value the value to set
         * @return {@code this}
         * @throws IllegalArgumentException if the property or value of the property is invalid as per the trace model
         */
        TraceletBuilder set(String name, Object value);

        /**
         * Returns the currently set value (if any) for the requested property.
         *
         * @param name property name
         * @param <T> the type of the value to get
         * @return the currently set value of this property
         */
        <T> T get(String name);

        /**
         * Returns the type of this tracelet.
         *
         * @return the tracelet type
         */
        String getType();
    }

    /**
     * a Tracelet represents tracedata that can be present multiple times within a trace.
     * The API doesn't specify the cardinality , but the implementation is limited to
     * cardinality Few.
     */
    class Tracelet {
        /**
         * The name of the Tracelet type (cardinality Few, Many not supported).
         */
        private final String _name;
        /**
         * The properties of a specific tracelet. A single tracelet can hold multiple
         * {@link TraceletProperty} elements.
         */
        private final List<TraceletProperty> _value;
        public Tracelet(final String name, final List<TraceletProperty> value) {
            _name = name;
            _value = value;
        }

        public String getName() {
            return _name;
        }

        public List<TraceletProperty> getValue() {
            return _value;
        }
    }
}
