package org.hansken.plugin.extraction.api;

import java.util.List;
import java.util.concurrent.ExecutionException;

/**
 * Allows searching for traces within the scope of the process function.
 */
public interface TraceSearcher {
    int ALL_SEARCH_RESULTS = -1;

    /**
     * Searches in Hansken for Traces matching provided query, scoped to the image under extraction.
     *
     * @param query Search query to match traces. This is an HQL query.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    default SearchResult search(final String query) throws InterruptedException, ExecutionException {
        return search(query, ALL_SEARCH_RESULTS, SearchOptions.DEFAULT);
    }

    /**
     * Searches in Hansken for Traces matching provided query, scoped to the image under extraction.
     *
     * @param query Search query to match traces. This is an HQL query.
     * @param count Maximum number of traces to return.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    default SearchResult search(final String query, final int count) throws InterruptedException, ExecutionException {
        return search(query, count, SearchOptions.DEFAULT);
    }

    /**
     * Searches in Hansken for Traces matching provided query.
     *
     * @param query Search query to match traces. This is an HQL query.
     * @param options {@link SearchOptions} object containing various search options.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    default SearchResult search(final String query, final SearchOptions options) throws InterruptedException, ExecutionException {
        return search(query, ALL_SEARCH_RESULTS, options.scope(), options.start(), options.sort());
    }

    /**
     * Searches in Hansken for Traces matching provided query.
     *
     * @param query Search query to match traces. This is an HQL query.
     * @param count Maximum number of traces to return.
     * @param options {@link SearchOptions} object containing various search options.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    default SearchResult search(final String query, final int count, final SearchOptions options) throws InterruptedException, ExecutionException {
        return search(query, count, options.scope(), options.start(), options.sort());
    }

    /**
     * Searches in Hansken for Traces matching provided query.
     *
     * @param query Search query to match traces. This is an HQL query.
     * @param count Maximum number of traces to return.
     * @param scope Scope the search to image or project level.
     * @param start Starting index of traces.
     * @param sort  Field to sort on.
     * @return SearchResult containing traces matching the provided query.
     * @throws InterruptedException if a thread searching for traces is interrupted.
     * @throws ExecutionException if searching for traces is throws an exception.
     */
    SearchResult search(String query, int count, SearchScope scope, int start, List<SearchSortOption> sort) throws InterruptedException, ExecutionException;
}
