package org.hansken.plugin.extraction.api;

import static java.lang.String.format;
import static java.util.Locale.ROOT;

import java.time.ZonedDateTime;
import java.util.List;
import java.util.Map;

/**
 * An argument or return value that can be passed to or returned from a Transformer.
 *
 * @author Netherlands Forensic Institute
 */
public class TransformerArgument {

    /** The actual value of the transformer argument stored as a generic Object. */
    private final Object _argument;

    /** The allowed types of a transformer argument. */
    private final List<Class<?>> _allowedTypes = List.of(
        Boolean.class,
        byte[].class,
        Long.class,
        Double.class,
        String.class,
        Vector.class,
        LatLong.class,
        ZonedDateTime.class,
        List.class,
        Map.class
    );

    /**
     * Takes an Object and validates if it is of an acceptable transformer argument or return type.
     * @param argument The object that is to be passed to or returned from a transformer.
     */
    public TransformerArgument(final Object argument) {
        _argument = validate(argument);
    }

    /**
     * Validates if the argument is of a valid Transformer argument or return type.
     * @param argument The argument that is to be validated
     * @return The validated argument.
     */
    private Object validate(final Object argument) {
        final boolean validType = _allowedTypes.stream().anyMatch(type -> type.isInstance(argument));
        if (!validType) {
            throw new IllegalArgumentException("Invalid object type used to construct TransformerArgument. " +
                format(ROOT, "Passed value is %s but may only be %s", argument.getClass().getName(), _allowedTypes));
        }
        return argument;
    }

    /**
     * Retrieves the Transformer argument as an Object.
     * @return The Transformer argument as an Object.
     */
    public Object getArgument() {
        return _argument;
    }
}
