package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.util.Arrays;
import java.util.Base64;
import java.util.Collection;

/**
 * An opaque vector of floating point values.
 *
 * @author Netherlands Forensic Institute
 */
public final class Vector {
    // the internal storage is byte[]
    // to minimize conversions when large vectors are passed around
    private final byte[] _encoded;

    private Vector(final byte[] encoded) {
        _encoded = encoded;
    }

    /**
     * Creates a Vector from an array of floating point values.
     *
     * @param values the values to use
     * @return a vector
     */
    public static Vector of(final float... values) {
        argNotNull("values", values);
        final ByteBuffer bytes = ByteBuffer.allocate(Float.BYTES * values.length);
        bytes.asFloatBuffer().put(values);
        return new Vector(bytes.array());
    }

    /**
     * Creates a Vector from a collection of numbers.
     * Note that all numbers are converted to floats internally, so loss of precision may occur when doubles or longs are offered.
     *
     * @param values the values to use
     * @return a vector
     */
    public static Vector of(final Collection<Float> values) {
        argNotNull("values", values);
        final ByteBuffer bytes = ByteBuffer.allocate(Float.BYTES * values.size());
        final FloatBuffer floats = bytes.asFloatBuffer();
        for (final Number value: values) {
            floats.put(value.floatValue());
        }
        return new Vector(bytes.array());
    }

    /**
     * Creates a Vector from a base64 encoded string.
     *
     * @param base64 the base64 encoded string to use
     * @return a vector
     */
    public static Vector ofBase64(final String base64) {
        argNotNull("base64", base64);
        final byte[] bytes = Base64.getDecoder().decode(base64);
        return new Vector(bytes);
    }

    /**
     * Creates a vector from a binary representation.
     * {@code Vector.asBinary()} can be used to obtain a binary representation.
     * Note that this directly sets the internal state of the Vector, use {code asVector(bytes.clone()} to store a safe, immutable copy.
     *
     * @param bytes the bytes to convert to a vector.
     * @return a vector
     */
    public static Vector asVector(final byte[] bytes) {
        argNotNull("bytes", bytes);
        return new Vector(bytes);
    }

    /**
     * Returns the binary representation of the Vector.
     * {@code Vector.asVector(byte[])} can be used to convert the binary representation back to the original vector.
     * The format of the returned bytes is a sequence of the floating point values of the vector, stored as big-endian IEEE 754 encoded 32-bit floating point values.
     * Note that this exposes the internal state of the Vector, use {code asBinary().clone()} to obtain a safely mutable copy.
     *
     * @return a binary representation of the vector.
     */
    public byte[] asBinary() {
        return _encoded;
    }

    /**
     * Returns the number of dimensions of the vector.
     * @return the number of dimensions of the vector.
     */
    public int size() {
        return _encoded.length / 4;
    }

    /**
     * Returns the values of the Vector as an array of floats.
     * @return the values of the Vector as an array of floats.
     */
    public float[] values() {
        final float[] floats = new float[size()];
        ByteBuffer.wrap(_encoded).asFloatBuffer().get(floats);
        return floats;
    }

    /**
     * Returns a base64 encoded string of the Vector.
     * {@code Vector.ofBase64()} can be used to convert the return base64 encoded string back to a Vector.
     *
     * @return a base64 encoded string of the Vector.
     */
    public String toBase64() {
        return Base64.getEncoder().encodeToString(_encoded);
    }

    @Override
    public String toString() {
        return toBase64();
    }

    @Override
    public int hashCode() {
        return Arrays.hashCode(_encoded);
    }

    @Override
    public boolean equals(final Object obj) {
        if (obj == this) {
            return true;
        }
        if (!(obj instanceof Vector)) {
            return false;
        }
        final Vector other = (Vector) obj;
        return Arrays.equals(_encoded, other._encoded);
    }
}
