package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

import java.util.Map;

/**
 *  Description of a transform method of a plugin.
 *  The transform method is a method, specified by method name, of a Transformer capable plugin
 *  that transforms input to output of a specified type (returnType)
 */
public class TransformerLabel {

    /** The name of the transformer method in the extraction plugin. */
    private final String _methodName;

    /** The parameters that the transformer method takes represented as a mapping of name and type. */
    private final Map<String, String> _parameters;

    /** The return type of the transformer method. Only a limited set of types is allowed. */
    private final String _returnType;

    /**
     * Plain constructor that constructs a TransformerLabel.
     * @param methodName The method name of the Transformer.
     * @param parameters The parameters of the Transformer as a name, type mapping. Currently only str is supported.
     * @param returnType The return type of the Transformer method. Currently only vector is supported.
     */
    public TransformerLabel(final String methodName, final Map<String, String> parameters, final String returnType) {
        _methodName = argNotNull("methodName", methodName);
        _parameters = argNotNull("parameters", parameters);
        _returnType = argNotNull("returnType", returnType);
    }

    /**
     * Retrieves the method name of the transformer.
     * @return The method name of the transformer.
     */
    public String getMethodName() {
        return _methodName;
    }

    /**
     * Retrieves the parameters of the transformer method as a name, type mapping.
     * @return The parameters of the transformer method.
     */
    public Map<String, String> getParameters() {
        return _parameters;
    }

    /**
     * Retrieves the return type of the transformer method. Currently only vector is supported.
     * @return The return type of the transformer method.
     */
    public String getReturnType() {
        return _returnType;
    }
}
