package org.hansken.plugin.extraction.api;

import static java.lang.Double.parseDouble;
import static java.util.Locale.ROOT;
import static java.util.regex.Pattern.compile;

import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A geographical location consisting of a latitude and a longitude.
 */
public class LatLong {
    private static final Pattern PATTERN = compile("(?<latitude>[+\\-][0-9.]+)(?<longitude>[+\\-][0-9.]+)");
    private final double _latitude;
    private final double _longitude;

    protected LatLong(final double latitude, final double longitude) {
        _latitude = latitude;
        _longitude = longitude;
    }

    /**
     * Create a new geographical point with given {@link #latitude() latitude}
     * and {@link #longitude() longitude}.
     * <p>
     * <strong>Note:</strong> this does not do any input validation. Valid
     * values should be between -90 and +90 inclusive for latitude and
     * between -180 and +180 inclusive for longitude.
     *
     * @param latitude the latitude
     * @param longitude the longitude
     * @return the new point
     */
    public static LatLong of(final double latitude, final double longitude) {
        return new LatLong(latitude, longitude);
    }

    public static LatLong of(final String coordinates) {
        final Matcher matcher = PATTERN.matcher(coordinates);
        if (!matcher.matches()) {
            throw new IllegalArgumentException("Cannot parse LatLong coordinates");
        }
        return new LatLong(parseDouble(matcher.group("latitude")), parseDouble(matcher.group("longitude")));
    }

    /**
     * The north-south position of a point on earth.
     *
     * @return the latitude
     */
    public double latitude() {
        return _latitude;
    }

    /**
     * The east-west position of a point on earth.
     *
     * @return the longitude
     */
    public double longitude() {
        return _longitude;
    }

    /**
     * Format to {@link String} using ISO 6709. The output is in 5 decimals.
     * The decimal degree precision is 1.1132m at the equator
     *
     * @return notation according to ISO 6709
     */
    public String toISO6709() {
        return String.format(ROOT, "%+09.5f%+010.5f", _latitude, _longitude);
    }

    @Override
    public boolean equals(final Object o) {
        if (this == o) {
            return true;
        }
        if (!(o instanceof LatLong)) {
            return false;
        }
        final LatLong other = (LatLong) o;
        return Double.compare(other._latitude, _latitude) == 0 && Double.compare(other._longitude, _longitude) == 0;
    }

    @Override
    public int hashCode() {
        return Objects.hash(_latitude, _longitude);
    }

    @Override
    public String toString() {
        return toISO6709();
    }
}
