package org.hansken.plugin.extraction.api;

import java.io.Closeable;
import java.io.IOException;

import org.hansken.plugin.extraction.util.ArgChecks;

/**
 * A random access readable byte sequence. It has a fixed length and an associated data type.
 * <p>
 * <strong>Note:</strong> implementations are not required to implement any kind of thread safety.
 * It is up to the client to ensure this if necessary.
 */
public interface RandomAccessData extends Closeable {

    /**
     * Get the number of bytes contained in this data sequence.
     *
     * @return the size in bytes
     */
    long size();

    /**
     * Get the position in the sequence. The position will be a number in range of
     * 0 and {@link #size()}, both inclusive.
     * <p>
     * For example: if the size is equal to 16, the position can take any positive
     * value up to and including 16.
     *
     * @return the current position
     */
    long position();

    /**
     * Get the number of remaining bytes in this data sequence.
     *
     * @return {@link #size()} - {@link #position()}
     */
    default long remaining() {
        return size() - position();
    }

    /**
     * Move to the given absolute position in the data sequence.
     *
     * @param position the position to move to
     * @throws IllegalArgumentException if given position is not in range 0 and {@link #size()}, both inclusive
     * @throws IOException              when an I/O error occurs
     */
    void seek(long position) throws IOException;

    /**
     * Read bytes into the given buffer, starting at position {@code 0} in the buffer. The data will be read from the
     * current {@link #position() position} and the amount of bytes copied will equal the length of the buffer, unless
     * the data sequence contains fewer remaining bytes. In that case, data is read until the end of the sequence.
     *
     * @param buffer the buffer to read into
     * @return the number of bytes actually read
     * @throws IOException when an I/O error occurs
     */
    default int read(final byte[] buffer) throws IOException {
        return read(buffer, min(buffer.length, remaining()));
    }

    /**
     * Read bytes into the given buffer, starting at position {@code 0} in the buffer. The data will be read from the
     * current {@link #position() position} and the amount of bytes copied will equal {@code count}, unless the data
     * sequence contains fewer remaining bytes. In that case, data is read until the end of the sequence.
     *
     * @param buffer the buffer to read into
     * @param count  the amount of bytes to read
     * @return the number of bytes actually read
     * @throws IllegalArgumentException if count is negative or larger than the size of the buffer
     * @throws IOException              when an I/O error occurs
     */
    default int read(final byte[] buffer, final int count) throws IOException {
        return read(buffer, 0, min(count, remaining()));
    }

    /**
     * Read data into the given buffer, starting at position {@code offset} in the buffer. The data will be read from
     * the current {@link #position() position} and the amount of bytes read will equal {@code count}, unless the
     * sequence contains fewer remaining bytes. In that case, data is read until the end of the sequence.
     *
     * @param buffer the buffer to read into
     * @param offset the offset in the buffer from which to start writing
     * @param count  the amount of bytes to read
     * @return the number of bytes actually read
     * @throws IllegalArgumentException if count or offset is negative,
     *                                  or if offset + count is larger than the size of the buffer
     * @throws IOException              when an I/O error occurs
     */
    int read(byte[] buffer, int offset, int count) throws IOException;

    /**
     * Read from the data sequence, returning the read bytes as an array.The data will be read from the current
     * {@link #position() position} and the amount of bytes read will equal {@code count}, unless the sequence contains
     * fewer remaining bytes. In that case, data is read until the end of the sequence and a smaller array is returned,
     * with a length equal to the number of read bytes.
     * <p>
     * <strong>Note:</strong> this method will allocate a new buffer each time it is called. It is intended for simple
     * cases where it is convenient to read a specified number of bytes into a byte array.
     *
     * @param count the amount of bytes to read
     * @return a buffer containing the read bytes, or an empty array if we were at the end of the stream
     * @throws IllegalArgumentException if {@code count} is negative
     * @throws IOException when an I/O error occurs
     */
    default byte[] readNBytes(final int count) throws IOException {
        ArgChecks.argNotNegative("count", count);

        final byte[] buffer = new byte[min(count, remaining())];
        read(buffer);
        return buffer;
    }

    private static int min(final long a, final long b) {
        return Math.toIntExact(Math.min(a, b));
    }
}
