package org.hansken.plugin.extraction.api;

import static org.hansken.plugin.extraction.util.ArgChecks.argNotEmpty;
import static org.hansken.plugin.extraction.util.ArgChecks.argNotNull;

/**
 * An author of a certain {@link ExtractionPlugin extraction plugin}.
 * <p>
 * Example for creating an author:
 * <pre>
 * {@code
 * Author.builder()
 *     .name("John")
 *     .email("johndoe@email.com")
 *     .organisation("Netherlands Forensic Institute")
 *     .build();
 * }
 * </pre>
 */
public final class Author {

    private final String _name;
    private final String _email;
    private final String _organisation;

    private Author(final String name, final String email, final String organisation) {
        _name = argNotNull("name", name);
        _email = argNotNull("email", email);
        _organisation = argNotNull("organisation", organisation);
    }

    /**
     * Start creating a new {@link Author author}.
     *
     * @return a builder for an author
     */
    public static Builder builder() {
        return new Builder();
    }

    /**
     * Get the name of the author.
     *
     * @return the name of the author
     */
    public String name() {
        return _name;
    }

    /**
     * Get the email address of the author.
     *
     * @return the email address of the author.
     */
    public String email() {
        return _email;
    }

    /**
     * Get the name of the organisation the author belongs to.
     *
     * @return the organisation of the author
     */
    public String organisation() {
        return _organisation;
    }

    /**
     * A builder for an {@link Author author}.
     * <p>
     * <strong>Note: </strong> all methods throw a {@link NullPointerException} when {@code null}
     * is passed to them, and all properties must be set.
     */
    public static final class Builder {

        private String _name;
        private String _email;
        private String _organisation;

        /**
         * Set the {@link Author#name() name}.
         *
         * @param name the name
         * @return {@code this}
         * @throws IllegalArgumentException if the given name is empty
         */
        public Builder name(final String name) {
            _name = argNotEmpty("name", name);
            return this;
        }

        /**
         * Set the {@link Author#email() email}.
         *
         * @param email the email
         * @return {@code this}
         * @throws IllegalArgumentException if the given email is empty
         */
        public Builder email(final String email) {
            _email = argNotEmpty("email", email);
            return this;
        }

        /**
         * Set the {@link Author#organisation() organisation}.
         *
         * @param organisation the organisation
         * @return {@code this}
         * @throws IllegalArgumentException if the given organisation is empty
         */
        public Builder organisation(final String organisation) {
            _organisation = argNotEmpty("organisation", organisation);
            return this;
        }

        /**
         * Create the author from the properties set on this {@link Builder builder}.
         *
         * @return the created author
         */
        public Author build() {
            return new Author(_name, _email, _organisation);
        }
    }
}