package org.hansken.plugin.extraction.api;

import java.io.IOException;

import org.hansken.plugin.extraction.util.ThrowingConsumer;

/**
 * Meta extraction plugins can be used by Hansken to process {@link Trace traces} during the extraction process.
 * A processed trace can be enriched with new information and new child traces can also be created.
 * <p>
 * The difference between this and a normal {@link ExtractionPlugin} is that this plugin does not receive
 * or process any data, only a trace itself.
 * <p>
 * When a plugin matches on the trace which is currently processed by Hansken
 * (for example, because it has certain properties), the plugin will receive the
 * matched trace in order to process it (see {@link #process(Trace)}).
 * <p>
 * <strong>Note for Hansken core developers:</strong> specifying {@code 'meta'} in the matcher is
 * not necessary, the framework takes care of this.
 */
public abstract class MetaExtractionPlugin implements ExtractionPlugin {

    @Override
    public final void process(final Trace trace, final DataContext dataContext) throws IOException {
        process(trace);
    }

    /**
     * Start processing a {@link Trace trace} without any of its associated data streams.
     * When processing a given trace, new properties may be set on it. New children can be added using
     * {@link Trace#newChild(String, ThrowingConsumer)}.
     * <p>
     * <strong>Note:</strong> the given trace should only be modified within the scope of this method.
     * Any modifications afterwards may be guarded against or result in undefined behaviour.
     *
     * @param trace the trace to process
     * @throws IOException when an I/O error occurs
     */
    public abstract void process(Trace trace) throws IOException;
}
