package org.hansken.plugin.extraction.runtime.grpc.client.api;

import java.io.IOException;

import org.hansken.plugin.extraction.util.ThrowingConsumer;
import org.hansken.plugin.extraction.api.Trace;

/**
 * {@link Trace} for client side implementations. Compared to other trace implementations it has
 * a different way of creating children. Instead of enriching a child trace within a given scope,
 * the child is returned for the caller to enrich.
 * <p>
 * The reason for this is that keeping this scope synchronized between client and server is difficult
 * and requires us to keep state in a complex way (because the full trace tree is not necessarily sent
 * at once from server to client).
 */
public interface ClientTrace extends Trace, AutoCloseable {

    /**
     * Get the name of the {@link ClientTrace}. It should be unique among its siblings.
     *
     * @return the name of the trace
     */
    String name();

    /**
     * Implementation which by default throws an exception.
     *
     * @param name the name of the child
     * @param enrichChildCallback consumer that gets the new child trace as input
     * @return nothing
     */
    @Override
    default Trace newChild(final String name, final ThrowingConsumer<Trace, IOException> enrichChildCallback) throws IOException {
        throw new UnsupportedOperationException("default implementation of a client trace does not support creating children by using a callback");
    }

    /**
     * See {@link Trace#newChild(String, ThrowingConsumer)}, but returns the new child {@link Trace trace}.
     * <p>
     * <strong>Note:</strong> clients using this method must take care to {@link #save()} the
     * returned traces manually.
     *
     * @param name the name of the child trace
     * @return the newly created child
     */
    ClientTrace newChild(String name);

    /**
     * Default implementation which calls {@link ClientTrace#save()}.
     *
     * @throws Exception if the close failed
     */
    @Override
    default void close() throws Exception {
        save();
    }

    /**
     * Save this child {@link Trace trace} in the backing storage.
     *
     * @throws Exception if the save failed
     */
    void save() throws Exception;
}
